jQuery(document).ready(function ($) {

    var lastFocusedField = null;
    var debounce = function (fn, delay) {
        var timer;
        return function () {
            var context = this;
            var args = arguments;
            clearTimeout(timer);
            timer = setTimeout(function () {
                fn.apply(context, args);
            }, delay);
        };
    };

    $('.seo-fury-tab').on('click', function () {
        var targetTab = $(this).data('tab');

        $('.seo-fury-tab').removeClass('active');
        $('.seo-fury-tab-content').removeClass('active');

        $(this).addClass('active');
        $('#tab-' + targetTab).addClass('active');
    });

    $(document).on('click', '.seo-fury-edit-snippet', function (e) {
        e.preventDefault();
        e.stopPropagation();

        console.log('Открываем модальное окно');

        $('#seo-fury-snippet-modal').show().css('display', 'block');
        $('body').addClass('modal-open').css('overflow', 'hidden');

        var targetTab = $(this).data('modal-tab') || 'basic';
        $('.seo-fury-modal-tab').removeClass('active');
        $('.seo-fury-modal-panel').removeClass('active');
        $('.seo-fury-modal-tab[data-tab="' + targetTab + '"]').addClass('active');
        $('.seo-fury-modal-panel[data-panel="' + targetTab + '"]').addClass('active');

        updateCharCount();
        updateSeoAnalysis();
        updateReadabilityAnalysis();
        updateExtraSeoAnalysis();
    });

    $('.seo-fury-modal-close').on('click', function (e) {
        e.preventDefault();
        $('#seo-fury-snippet-modal').hide();
        $('body').removeClass('modal-open').css('overflow', '');
    });

    $('.seo-fury-modal-overlay').on('click', function (e) {
        e.preventDefault();
        $('#seo-fury-snippet-modal').hide();
        $('body').removeClass('modal-open').css('overflow', '');
    });

    $('.seo-fury-modal-save').on('click', function (e) {
        e.preventDefault();
        $('#seo-fury-snippet-modal').hide();
        $('body').removeClass('modal-open').css('overflow', '');

        updateGooglePreview();
        updateSeoAnalysis();
        updateReadabilityAnalysis();
        updateExtraSeoAnalysis();
    });

    $(document).on('keydown', function (e) {
        if (e.key === 'Escape' && $('#seo-fury-snippet-modal').is(':visible')) {
            $('#seo-fury-snippet-modal').hide();
            $('body').removeClass('modal-open').css('overflow', '');
        }
    });

    $('#seo-fury-snippet-modal').on('focus', 'input[type="text"], textarea', function () {
        lastFocusedField = $(this);
        console.log('Активное поле:', $(this).attr('id'));
    });

    $(document).on('click', '.seo-fury-modal-tab', function () {
        var tab = $(this).data('tab');
        $('.seo-fury-modal-tab').removeClass('active');
        $('.seo-fury-modal-panel').removeClass('active');
        $(this).addClass('active');
        $('.seo-fury-modal-panel[data-panel="' + tab + '"]').addClass('active');
    });

    $(document).on('click', '.seo-fury-social-tab', function () {
        var $container = $(this).closest('.seo-fury-section, .seo-fury-modal-panel');
        var tab = $(this).data('social-tab');
        $container.find('.seo-fury-social-tab').removeClass('active');
        $container.find('.seo-fury-social-panel').removeClass('active');
        $(this).addClass('active');
        $container.find('.seo-fury-social-panel[data-social-panel="' + tab + '"]').addClass('active');
    });

    $(document).on('click', '.seo-fury-social-settings-tab', function () {
        var $container = $(this).closest('.seo-fury-social-settings');
        var tab = $(this).data('settings-tab');
        $container.find('.seo-fury-social-settings-tab').removeClass('active');
        $container.find('.seo-fury-social-settings-panel').removeClass('active');
        $(this).addClass('active');
        $container.find('.seo-fury-social-settings-panel[data-settings-panel="' + tab + '"]').addClass('active');
    });

    function syncSocialAllFields($container) {
        var title = $container.find('.seo-fury-social-all-title').val() || '';
        var description = $container.find('.seo-fury-social-all-description').val() || '';
        var image = $container.find('.seo-fury-social-all-image').val() || '';

        if (title) {
            $('#my_seo_og_title, #my_seo_fb_title, #my_seo_twitter_title').val(title).trigger('input');
        }
        if (description) {
            $('#my_seo_og_description, #my_seo_fb_description, #my_seo_twitter_description').val(description).trigger('input');
        }
        if (image) {
            $('#my_seo_og_image, #my_seo_fb_image, #my_seo_twitter_image').val(image).trigger('input');
        }
    }

    $(document).on('input', '.seo-fury-social-all-title, .seo-fury-social-all-description, .seo-fury-social-all-image', function () {
        var $container = $(this).closest('.seo-fury-social-settings');
        if ($container.length) {
            syncSocialAllFields($container);
        }
    });

    function getPostTitle() {
        var postTitle = $('#title').val();
        if (!postTitle && typeof wp !== 'undefined' && wp.data && wp.data.select) {
            try {
                postTitle = wp.data.select('core/editor').getEditedPostAttribute('title');
            } catch (e) {
                postTitle = postTitle || '';
            }
        }
        if (!postTitle && typeof seoFuryData !== 'undefined' && seoFuryData.postTitle) {
            postTitle = seoFuryData.postTitle;
        }
        return postTitle || '';
    }

    function getPostExcerpt() {
        var postExcerpt = '';
        if (typeof wp !== 'undefined' && wp.data && wp.data.select) {
            try {
                postExcerpt = wp.data.select('core/editor').getEditedPostAttribute('excerpt');
            } catch (e) {
                postExcerpt = postExcerpt || '';
            }
        }
        if (!postExcerpt && typeof seoFuryData !== 'undefined' && seoFuryData.postExcerpt) {
            postExcerpt = seoFuryData.postExcerpt;
        }
        return postExcerpt || '';
    }

    function getPostContent() {
        var content = '';
        if (typeof wp !== 'undefined' && wp.data && wp.data.select) {
            try {
                content = wp.data.select('core/editor').getEditedPostContent();
            } catch (e) {
                content = content || '';
            }
        }
        if (!content) {
            content = $('#content').val() || '';
        }
        return content || '';
    }

    function getPostSlug() {
        var slug = '';
        if (typeof wp !== 'undefined' && wp.data && wp.data.select) {
            try {
                slug = wp.data.select('core/editor').getEditedPostAttribute('slug') || '';
            } catch (e) {
                slug = slug || '';
            }
        }
        return slug || '';
    }

    function stripHtml(input) {
        if (!input) return '';
        return input.replace(/<[^>]*>/g, ' ').replace(/\s+/g, ' ').trim();
    }

    function getContentMetrics() {
        var html = getPostContent();
        var parser = new DOMParser();
        var doc = parser.parseFromString(html, 'text/html');
        var text = doc.body ? doc.body.textContent || '' : '';
        var words = text.trim().split(/\s+/).filter(Boolean);
        var sentences = text.split(/[.!?]+/).map(function (s) { return s.trim(); }).filter(Boolean);
        var paragraphs = [];
        if (doc.body) {
            doc.body.querySelectorAll('p, li, blockquote').forEach(function (node) {
                var pText = (node.textContent || '').trim();
                if (pText) {
                    paragraphs.push(pText);
                }
            });
        }
        if (paragraphs.length === 0 && text.trim()) {
            paragraphs = text.split(/\n+/).map(function (p) { return p.trim(); }).filter(Boolean);
        }
        var images = doc.querySelectorAll('img');
        var imagesWithAlt = 0;
        images.forEach(function (img) {
            var alt = (img.getAttribute('alt') || '').trim();
            if (alt.length > 0) {
                imagesWithAlt += 1;
            }
        });
        var links = doc.querySelectorAll('a[href]');
        var internalLinks = 0;
        var externalLinks = 0;
        var host = window.location.host;
        links.forEach(function (link) {
            var href = link.getAttribute('href') || '';
            if (!href) return;
            if (href.indexOf('#') === 0 || href.indexOf('mailto:') === 0 || href.indexOf('tel:') === 0) {
                return;
            }
            if (href.indexOf('/') === 0 || href.indexOf(host) !== -1) {
                internalLinks += 1;
            } else if (href.indexOf('http://') === 0 || href.indexOf('https://') === 0) {
                externalLinks += 1;
            }
        });

        var h1Count = doc.querySelectorAll('h1').length;
        var h2Count = doc.querySelectorAll('h2').length;
        var listsCount = doc.querySelectorAll('ul, ol').length;

        return {
            wordCount: words.length,
            textLower: text.toLowerCase(),
            sentenceCount: sentences.length,
            sentences: sentences,
            paragraphs: paragraphs,
            images: images.length,
            imagesWithAlt: imagesWithAlt,
            internalLinks: internalLinks,
            externalLinks: externalLinks,
            h1Count: h1Count,
            h2Count: h2Count,
            listsCount: listsCount
        };
    }

    function updateSeoAnalysis() {
        var focusKeyword = ($('#my_seo_focus_keyword').val() || '').trim();
        var focusLower = focusKeyword.toLowerCase();
        var title = $('#my_seo_title').val() ? parseVariables($('#my_seo_title').val()) : (getPostTitle() || '');
        var description = $('#my_seo_description').val() ? parseVariables($('#my_seo_description').val()) : (getPostExcerpt() || '');
        var metrics = getContentMetrics();
        var content = metrics.textLower;
        var slug = getPostSlug().toLowerCase();
        var titleLength = title.length;
        var descriptionLength = description.length;
        var minWordCount = 300;
        var keywordInSlug = focusLower.length > 0 && slug.replace(/-/g, ' ').indexOf(focusLower) !== -1;
        var keywordInContent = focusLower.length > 0 && content.indexOf(focusLower) !== -1;
        var keywordInTitle = focusLower.length > 0 && title.toLowerCase().indexOf(focusLower) !== -1;
        var keywordInDescription = focusLower.length > 0 && description.toLowerCase().indexOf(focusLower) !== -1;
        var words = metrics.wordCount;
        var keywordCount = 0;
        if (focusLower.length > 0 && words > 0) {
            var pattern = new RegExp(focusLower.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'gi');
            keywordCount = (content.match(pattern) || []).length;
        }
        var keywordDensity = words > 0 ? (keywordCount / words) * 100 : 0;
        var keywordDensityOk = focusLower.length > 0 && keywordDensity >= 0.5 && keywordDensity <= 2.5;
        var hasH1 = metrics.h1Count > 0 || title.length > 0;
        var hasH2 = metrics.h2Count > 0;
        var imagesOk = metrics.images === 0 || (metrics.imagesWithAlt / metrics.images) >= 0.8;
        var hasInternalLink = metrics.internalLinks > 0;

        var checks = [
            {
                ok: focusLower.length > 0,
                textOk: seoFuryData.i18n.focusKeywordSet,
                textWarn: seoFuryData.i18n.addFocusKeyword
            },
            {
                ok: keywordInTitle,
                textOk: seoFuryData.i18n.keywordInTitle,
                textWarn: seoFuryData.i18n.addKeywordToTitle
            },
            {
                ok: keywordInDescription,
                textOk: seoFuryData.i18n.keywordInDescription,
                textWarn: seoFuryData.i18n.addKeywordToDescription
            },
            {
                ok: keywordInContent,
                textOk: seoFuryData.i18n.keywordInContent,
                textWarn: seoFuryData.i18n.addKeywordToContent
            },
            {
                ok: keywordInSlug,
                textOk: seoFuryData.i18n.keywordInUrl,
                textWarn: seoFuryData.i18n.addKeywordToUrl
            },
            {
                ok: keywordDensityOk,
                textOk: seoFuryData.i18n.keywordDensityOk,
                textWarn: seoFuryData.i18n.keywordDensityBad
            },
            {
                ok: titleLength >= 50 && titleLength <= 60,
                textOk: seoFuryData.i18n.titleLengthOk,
                textWarn: seoFuryData.i18n.titleLengthBad
            },
            {
                ok: descriptionLength >= 120 && descriptionLength <= 160,
                textOk: seoFuryData.i18n.descLengthOk,
                textWarn: seoFuryData.i18n.descLengthBad
            },
            {
                ok: words >= minWordCount,
                textOk: seoFuryData.i18n.contentLengthOk,
                textWarn: seoFuryData.i18n.contentLengthBad
            },
            {
                ok: hasH1,
                textOk: seoFuryData.i18n.h1Present,
                textWarn: seoFuryData.i18n.addH1
            },
            {
                ok: hasH2,
                textOk: seoFuryData.i18n.h2Present,
                textWarn: seoFuryData.i18n.addH2
            },
            {
                ok: imagesOk,
                textOk: seoFuryData.i18n.imagesAltOk,
                textWarn: seoFuryData.i18n.addImageAlt
            },
            {
                ok: hasInternalLink,
                textOk: seoFuryData.i18n.internalLinkPresent,
                textWarn: seoFuryData.i18n.addInternalLink
            }
        ];

        var $list = $('#seo-fury-analysis-list');
        if (!$list.length) {
            return;
        }

        $list.empty();
        var passed = 0;
        checks.forEach(function (check) {
            var statusClass = check.ok ? 'is-ok' : 'is-warn';
            if (check.ok) passed += 1;
            $list.append('<li class="seo-fury-analysis-item ' + statusClass + '">' + (check.ok ? check.textOk : check.textWarn) + '</li>');
        });

        var score = Math.round((passed / checks.length) * 100);
        var $score = $('#seo-fury-score');
        $score.text(score + '/100');
        $score.removeClass('seo-fury-score--good seo-fury-score--ok seo-fury-score--warn');
        if (score >= 80) {
            $score.addClass('seo-fury-score--good');
        } else if (score >= 50) {
            $score.addClass('seo-fury-score--ok');
        } else {
            $score.addClass('seo-fury-score--warn');
        }

        var densityText = focusLower.length > 0 ? keywordDensity.toFixed(2) + '%' : '—';
        $('#seo-fury-keyword-density').text(densityText);
        var densityBar = $('#seo-fury-keyword-density-bar');
        var densityPercent = Math.min(100, Math.round((keywordDensity / 3) * 100));
        densityBar.css('width', densityPercent + '%');
        densityBar.removeClass('is-good is-ok is-warn');
        if (keywordDensityOk) {
            densityBar.addClass('is-good');
        } else if (keywordDensity > 0) {
            densityBar.addClass('is-warn');
        } else {
            densityBar.addClass('is-ok');
        }
    }

    function updateReadabilityAnalysis() {
        var metrics = getContentMetrics();
        var sentences = metrics.sentences;
        var paragraphs = metrics.paragraphs;

        var sentenceLengths = sentences.map(function (s) {
            return s.split(/\s+/).filter(Boolean).length;
        });
        var avgSentenceLength = sentenceLengths.length ? Math.round(sentenceLengths.reduce(function (a, b) { return a + b; }, 0) / sentenceLengths.length) : 0;
        var maxSentenceLength = sentenceLengths.length ? Math.max.apply(null, sentenceLengths) : 0;
        var paragraphLengths = paragraphs.map(function (p) {
            return p.split(/\s+/).filter(Boolean).length;
        });
        var maxParagraphLength = paragraphLengths.length ? Math.max.apply(null, paragraphLengths) : 0;

        var checks = [
            {
                ok: sentences.length >= 3,
                textOk: seoFuryData.i18n.enoughSentences,
                textWarn: seoFuryData.i18n.addMoreSentences
            },
            {
                ok: avgSentenceLength > 0 && avgSentenceLength <= 20,
                textOk: seoFuryData.i18n.avgSentenceLengthOk,
                textWarn: seoFuryData.i18n.makeSentencesShorter
            },
            {
                ok: maxSentenceLength === 0 || maxSentenceLength <= 30,
                textOk: seoFuryData.i18n.noLongSentences,
                textWarn: seoFuryData.i18n.longSentences
            },
            {
                ok: paragraphs.length >= 2,
                textOk: seoFuryData.i18n.multipleParagraphs,
                textWarn: seoFuryData.i18n.splitParagraphs
            },
            {
                ok: maxParagraphLength === 0 || maxParagraphLength <= 150,
                textOk: seoFuryData.i18n.paragraphLengthOk,
                textWarn: seoFuryData.i18n.makeParagraphsShorter
            },
            {
                ok: metrics.listsCount > 0,
                textOk: seoFuryData.i18n.listsPresent,
                textWarn: seoFuryData.i18n.addLists
            }
        ];

        var $list = $('#seo-fury-readability-list');
        if (!$list.length) {
            return;
        }

        $list.empty();
        var passed = 0;
        checks.forEach(function (check) {
            var statusClass = check.ok ? 'is-ok' : 'is-warn';
            if (check.ok) passed += 1;
            $list.append('<li class="seo-fury-analysis-item ' + statusClass + '">' + (check.ok ? check.textOk : check.textWarn) + '</li>');
        });

        var score = Math.round((passed / checks.length) * 100);
        var $score = $('#seo-fury-readability-score');
        if ($score.length) {
            $score.text(score + '/100');
            $score.removeClass('seo-fury-score--good seo-fury-score--ok seo-fury-score--warn');
            if (score >= 80) {
                $score.addClass('seo-fury-score--good');
            } else if (score >= 50) {
                $score.addClass('seo-fury-score--ok');
            } else {
                $score.addClass('seo-fury-score--warn');
            }
        }

        var readabilityText = score >= 80 ? seoFuryData.i18n.excellent : (score >= 50 ? seoFuryData.i18n.okay : seoFuryData.i18n.needsWork);
        $('#seo-fury-readability-text').text(readabilityText);
        var readabilityBar = $('#seo-fury-readability-bar');
        readabilityBar.css('width', score + '%');
        readabilityBar.removeClass('is-good is-ok is-warn');
        if (score >= 80) {
            readabilityBar.addClass('is-good');
        } else if (score >= 50) {
            readabilityBar.addClass('is-ok');
        } else {
            readabilityBar.addClass('is-warn');
        }
    }

    function updateExtraSeoAnalysis() {
        var metrics = getContentMetrics();
        var focusKeyword = ($('#my_seo_focus_keyword').val() || '').trim().toLowerCase();
        var slug = getPostSlug();
        var firstParagraph = metrics.paragraphs.length ? metrics.paragraphs[0].toLowerCase() : '';

        var keywordInFirstParagraph = focusKeyword.length > 0 && firstParagraph.indexOf(focusKeyword) !== -1;
        var urlLengthOk = slug.length > 0 && slug.length <= 75;
        var hasExternalLink = metrics.externalLinks > 0;
        var hasInternalLink = metrics.internalLinks > 0;
        var hasImage = metrics.images > 0;

        var checks = [
            {
                ok: keywordInFirstParagraph,
                textOk: seoFuryData.i18n.keywordInFirstParagraph,
                textWarn: seoFuryData.i18n.addKeywordToFirstParagraph
            },
            {
                ok: urlLengthOk,
                textOk: seoFuryData.i18n.urlLengthOk,
                textWarn: seoFuryData.i18n.shortenUrl
            },
            {
                ok: hasInternalLink,
                textOk: seoFuryData.i18n.internalLinkPresent,
                textWarn: seoFuryData.i18n.addInternalLink
            },
            {
                ok: hasExternalLink,
                textOk: seoFuryData.i18n.externalLinkPresent,
                textWarn: seoFuryData.i18n.addExternalLink
            },
            {
                ok: hasImage,
                textOk: seoFuryData.i18n.imagesInContent,
                textWarn: seoFuryData.i18n.addImage
            }
        ];

        var $list = $('#seo-fury-extra-list');
        if (!$list.length) {
            return;
        }

        $list.empty();
        var passed = 0;
        checks.forEach(function (check) {
            var statusClass = check.ok ? 'is-ok' : 'is-warn';
            if (check.ok) passed += 1;
            $list.append('<li class="seo-fury-analysis-item ' + statusClass + '">' + (check.ok ? check.textOk : check.textWarn) + '</li>');
        });

        var score = Math.round((passed / checks.length) * 100);
        var $score = $('#seo-fury-extra-score');
        if ($score.length) {
            $score.text(score + '/100');
            $score.removeClass('seo-fury-score--good seo-fury-score--ok seo-fury-score--warn');
            if (score >= 80) {
                $score.addClass('seo-fury-score--good');
            } else if (score >= 50) {
                $score.addClass('seo-fury-score--ok');
            } else {
                $score.addClass('seo-fury-score--warn');
            }
        }
    }

    function parseVariables(text) {
        if (!text) return text;

        var postTitle = getPostTitle();
        if (!postTitle) {
            postTitle = 'Page Title';
        }

        var postExcerpt = getPostExcerpt() || 'Page description';

        var siteName = typeof seoFuryData !== 'undefined' ? seoFuryData.siteName : 'Site name';
        var separator = typeof seoFuryData !== 'undefined' ? seoFuryData.separator : '—';

        var currentDate = new Date();
        var year = currentDate.getFullYear();
        var month = String(currentDate.getMonth() + 1).padStart(2, '0');
        var day = String(currentDate.getDate()).padStart(2, '0');
        var date = day + '.' + month + '.' + year;

        text = text.replace(/%title%/g, postTitle);
        text = text.replace(/%excerpt%/g, postExcerpt);
        text = text.replace(/%sitename%/g, siteName);
        text = text.replace(/%sep%/g, separator);
        text = text.replace(/%separator%/g, separator);
        text = text.replace(/%date%/g, date);
        text = text.replace(/%year%/g, year);
        text = text.replace(/%month%/g, month);
        text = text.replace(/%day%/g, day);

        return text;
    }

    function escapeHtml(text) {
        return String(text || '')
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#039;');
    }

    function escapeRegExp(text) {
        return text.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    }

    function highlightKeyword(text, keyword) {
        if (!keyword) {
            return escapeHtml(text);
        }
        var safeText = escapeHtml(text);
        var pattern = new RegExp(escapeRegExp(keyword), 'gi');
        return safeText.replace(pattern, function (match) {
            return '<mark class="seo-fury-focus-highlight">' + match + '</mark>';
        });
    }

    var textMeasureCanvas = document.createElement('canvas');
    var textMeasureContext = textMeasureCanvas.getContext('2d');
    function getTextWidth(text, font) {
        textMeasureContext.font = font;
        return Math.ceil(textMeasureContext.measureText(text).width);
    }

    function updateCharCount() {
        var titleVal = $('#my_seo_title').val() || '';
        var descVal = $('#my_seo_description').val() || '';

        var titleLength = parseVariables(titleVal).length;
        var descLength = parseVariables(descVal).length;

        var titlePixels = getTextWidth(parseVariables(titleVal), '16px Arial');
        var descPixels = getTextWidth(parseVariables(descVal), '13px Arial');

        $('#title-length').text(titleLength);
        $('#description-length').text(descLength);
        $('#title-pixels').text(titlePixels);
        $('#description-pixels').text(descPixels);

        if (titleLength > 60) {
            $('#title-length').css('color', '#ef4444');
        } else if (titleLength > 50) {
            $('#title-length').css('color', '#f59e0b');
        } else {
            $('#title-length').css('color', '#10b981');
        }

        if (descLength > 160) {
            $('#description-length').css('color', '#ef4444');
        } else if (descLength > 150) {
            $('#description-length').css('color', '#f59e0b');
        } else {
            $('#description-length').css('color', '#10b981');
        }

        if (titlePixels > 580) {
            $('#title-pixels').css('color', '#ef4444');
        } else if (titlePixels > 520) {
            $('#title-pixels').css('color', '#f59e0b');
        } else {
            $('#title-pixels').css('color', '#10b981');
        }

        if (descPixels > 920) {
            $('#description-pixels').css('color', '#ef4444');
        } else if (descPixels > 820) {
            $('#description-pixels').css('color', '#f59e0b');
        } else {
            $('#description-pixels').css('color', '#10b981');
        }
    }

    var debouncedCharCount = debounce(updateCharCount, 150);
    $('#my_seo_title, #my_seo_description').on('input', debouncedCharCount);
    $('#title').on('input', debouncedCharCount);

    setTimeout(function () {
        updateCharCount();
    }, 100);

    function updateGooglePreview() {
        var titleVal = $('#my_seo_title').val();
        var descVal = $('#my_seo_description').val();
        var focus = ($('#my_seo_focus_keyword').val() || '').trim();

        var title = titleVal ? parseVariables(titleVal) : (getPostTitle() || 'Page Title');
        var description = descVal ? parseVariables(descVal) : 'Page description will appear here';

        $('.preview-title').html(highlightKeyword(title, focus));
        $('.preview-description').html(highlightKeyword(description, focus));
    }

    var debouncedGooglePreview = debounce(updateGooglePreview, 150);
    var debouncedSeoAnalysis = debounce(updateSeoAnalysis, 200);
    var debouncedReadability = debounce(updateReadabilityAnalysis, 200);
    var debouncedExtraSeo = debounce(updateExtraSeoAnalysis, 200);

    $('#my_seo_title, #my_seo_description, #my_seo_focus_keyword').on('input', debouncedGooglePreview);
    $('#title').on('input', debouncedGooglePreview);

    $('#my_seo_title, #my_seo_description, #my_seo_focus_keyword').on('input', function () {
        debouncedSeoAnalysis();
        debouncedReadability();
        debouncedExtraSeo();
    });

    setTimeout(function () {
        updateGooglePreview();
        updateSeoAnalysis();
        updateReadabilityAnalysis();
        updateExtraSeoAnalysis();
    }, 100);

    function updateOGPreview() {
        var title = $('#my_seo_og_title').val() || $('#my_seo_title').val() || getPostTitle() || seoFuryData.i18n.titlePlaceholder;
        var description = $('#my_seo_og_description').val() || $('#my_seo_description').val() || seoFuryData.i18n.descriptionPlaceholder;
        var image = $('#my_seo_og_image').val();
        var focus = ($('#my_seo_focus_keyword').val() || '').trim();

        title = parseVariables(title);
        description = parseVariables(description);

        $('.og-title').html(highlightKeyword(title, focus));
        $('.og-description').html(highlightKeyword(description, focus));

        if (image) {
            if ($('.og-image img').length === 0) {
                $('.og-image').html('<img src="' + image + '" alt="OG Preview" />');
            } else {
                $('.og-image img').attr('src', image);
            }
            $('.og-image').show();
        } else {
            $('.og-image').hide();
        }
    }

    var debouncedOgPreview = debounce(updateOGPreview, 150);
    $('#my_seo_og_title, #my_seo_og_description, #my_seo_og_image, #my_seo_title, #my_seo_description, #my_seo_focus_keyword').on('input', debouncedOgPreview);
    $('#title').on('input', debouncedOgPreview);

    setTimeout(function () {
        updateOGPreview();
    }, 100);

    function updateFacebookPreview() {
        var title = $('#my_seo_fb_title').val() || $('#my_seo_og_title').val() || $('#my_seo_title').val() || getPostTitle() || seoFuryData.i18n.titlePlaceholder;
        var description = $('#my_seo_fb_description').val() || $('#my_seo_og_description').val() || $('#my_seo_description').val() || seoFuryData.i18n.descriptionPlaceholder;
        var image = $('#my_seo_fb_image').val() || $('#my_seo_og_image').val();
        var focus = ($('#my_seo_focus_keyword').val() || '').trim();

        title = parseVariables(title);
        description = parseVariables(description);

        $('.facebook-card .fb-title').html(highlightKeyword(title, focus));
        $('.facebook-card .fb-description').html(highlightKeyword(description, focus));

        if (image) {
            var $img = $('.facebook-card .social-post-image img');
            if ($img.length === 0) {
                $('.facebook-card .social-post-image').html('<img src="' + image + '" alt="FB Preview" />');
            } else {
                $img.attr('src', image);
            }
            $('.facebook-card .social-post-image').show();
        } else {
            $('.facebook-card .social-post-image').hide();
        }
    }

    var debouncedFacebookPreview = debounce(updateFacebookPreview, 150);
    $('#my_seo_fb_title, #my_seo_fb_description, #my_seo_fb_image').on('input', debouncedFacebookPreview);
    $('#my_seo_og_title, #my_seo_og_description, #my_seo_og_image').on('input', debouncedFacebookPreview);
    $('#my_seo_title, #my_seo_description, #my_seo_focus_keyword').on('input', debouncedFacebookPreview);
    $('#title').on('input', debouncedFacebookPreview);

    setTimeout(function () {
        updateFacebookPreview();
    }, 100);

    function updateTwitterPreview() {
        var title = $('#my_seo_twitter_title').val() || $('#my_seo_og_title').val() || $('#my_seo_title').val() || getPostTitle() || seoFuryData.i18n.titlePlaceholder;
        var description = $('#my_seo_twitter_description').val() || $('#my_seo_og_description').val() || $('#my_seo_description').val() || seoFuryData.i18n.descriptionPlaceholder;
        var image = $('#my_seo_twitter_image').val() || $('#my_seo_og_image').val();
        var focus = ($('#my_seo_focus_keyword').val() || '').trim();

        title = parseVariables(title);
        description = parseVariables(description);

        $('.twitter-card .tw-title').html(highlightKeyword(title, focus));
        $('.twitter-card .tw-description').html(highlightKeyword(description, focus));

        if (image) {
            var $img = $('.twitter-card .social-post-image img');
            if ($img.length === 0) {
                $('.twitter-card .social-post-image').html('<img src="' + image + '" alt="Twitter Preview" />');
            } else {
                $img.attr('src', image);
            }
            $('.twitter-card .social-post-image').show();
        } else {
            $('.twitter-card .social-post-image').hide();
        }
    }

    var debouncedTwitterPreview = debounce(updateTwitterPreview, 150);
    $('#my_seo_twitter_title, #my_seo_twitter_description, #my_seo_twitter_image, #my_seo_twitter_card_type').on('input change', debouncedTwitterPreview);
    $('#my_seo_og_title, #my_seo_og_description, #my_seo_og_image').on('input', debouncedTwitterPreview);
    $('#my_seo_title, #my_seo_description, #my_seo_focus_keyword').on('input', debouncedTwitterPreview);
    $('#title').on('input', debouncedTwitterPreview);

    setTimeout(function () {
        updateTwitterPreview();
    }, 100);

    var mediaUploader;

    $(document).on('click', '.my-seo-upload-image', function (e) {
        e.preventDefault();
        var button = $(this);
        var targetField = button.data('target');

        mediaUploader = wp.media({
            title: seoFuryData.i18n.selectImage,
            button: {
                text: seoFuryData.i18n.useThisImage
            },
            multiple: false
        });

        mediaUploader.on('select', function () {
            var attachment = mediaUploader.state().get('selection').first().toJSON();
            $('#' + targetField).val(attachment.url);

            $('[data-target="' + targetField + '"].my-seo-remove-image-button').show();

            var preview = $('#' + targetField).siblings('.my-seo-image-preview');
            if (preview.length) {
                preview.find('img').attr('src', attachment.url);
            } else {
                $('#' + targetField).after('<div class="my-seo-image-preview"><img src="' + attachment.url + '" style="max-width: 200px; margin-top: 10px;" /></div>');
            }

            $('#' + targetField).trigger('input');
        });

        mediaUploader.open();
    });

    $(document).on('click', '.my-seo-remove-image-button', function (e) {
        e.preventDefault();
        var button = $(this);
        var targetField = button.data('target');

        console.log('Удаление изображения для поля:', targetField);

        $('#' + targetField).val('');

        button.hide();

        $('#' + targetField).siblings('.my-seo-image-preview').remove();

        $('#' + targetField).trigger('input');
    });

    $('#my_seo_canonical').each(function () {
        if (!$(this).val()) {
            var currentUrl = window.location.href.split('?')[0];
            if (typeof wp !== 'undefined' && wp.data && wp.data.select && wp.data.select('core/editor')) {
                try {
                    var permalink = wp.data.select('core/editor').getPermalink();
                    if (permalink) {
                        $(this).attr('placeholder', permalink);
                    }
                } catch (e) {
                    console.log('Failed to get permalink');
                }
            }
        }
    });

    $('.seo-fury-tab[data-tab="basic"]').trigger('click');

    var schemaItems = [];

    function loadSchemaItems() {
        try {
            var raw = $('#my_seo_schema_items').val() || '[]';
            var parsed = JSON.parse(raw);
            schemaItems = Array.isArray(parsed) ? parsed : [];
        } catch (e) {
            schemaItems = [];
        }
    }

    function saveSchemaItems() {
        $('#my_seo_schema_items').val(JSON.stringify(schemaItems));
    }

    function getActiveSchemaId() {
        return $('#my_seo_schema_active').val() || '';
    }

    function setActiveSchemaId(id) {
        $('#my_seo_schema_active').val(id || '');
    }

    function getActiveSchemaItem() {
        var activeId = getActiveSchemaId();
        return schemaItems.find(function (item) { return item.id === activeId; }) || null;
    }

    function setSchemaMode(mode) {
        $('#my_seo_schema_mode').val(mode);
    }

    function setSchemaType(type) {
        $('#my_seo_schema_type').val(type || 'none');
    }

    function setDeepValue(obj, path, value) {
        var current = obj;
        for (var i = 0; i < path.length; i++) {
            var key = path[i];
            if (i === path.length - 1) {
                current[key] = value;
            } else {
                if (current[key] === undefined) {
                    current[key] = isNaN(path[i + 1]) ? {} : [];
                }
                current = current[key];
            }
        }
    }

    function getSchemaDataFromInputs() {
        var data = {};
        $('[name^="schema_data["]').each(function () {
            var name = $(this).attr('name');
            if (!name) return;
            var value = $(this).val();
            var path = name.replace(/^schema_data\[/, '').replace(/\]$/, '').split('][');
            setDeepValue(data, path, value);
        });
        return data;
    }

    function clearSchemaDataInputs() {
        $('[name^="schema_data["]').each(function () {
            $(this).val('');
        });
    }

    function setSchemaDataInputs(data, base) {
        base = base || 'schema_data';
        Object.keys(data || {}).forEach(function (key) {
            var value = data[key];
            var name = base + '[' + key + ']';
            if (typeof value === 'object' && value !== null) {
                setSchemaDataInputs(value, name);
            } else {
                $('[name="' + name + '"]').val(value);
            }
        });
    }

    function storeActiveSchemaData() {
        var item = getActiveSchemaItem();
        if (!item) return;
        var mode = $('#my_seo_schema_mode').val() || 'generator';
        var type = $('#my_seo_schema_type').val() || 'none';
        item.mode = mode;
        item.type = mode === 'custom' ? 'custom' : type;
        if (mode === 'custom') {
            item.custom = $('#my_seo_schema_custom').val() || '';
            item.data = {};
        } else {
            item.custom = '';
            item.data = getSchemaDataFromInputs();
        }
        saveSchemaItems();
    }

    function applyActiveSchemaItem(item) {
        if (!item) {
            setSchemaMode('generator');
            setSchemaType('none');
            clearSchemaDataInputs();
            return;
        }
        setSchemaMode(item.mode || 'generator');
        setSchemaType(item.mode === 'custom' ? 'none' : (item.type || 'none'));
        clearSchemaDataInputs();
        if (item.mode === 'custom') {
            $('#my_seo_schema_custom').val(item.custom || '');
            $('#schema-custom-json-output').text(item.custom || seoFuryData.i18n.pasteJsonLd);
        } else {
            setSchemaDataInputs(item.data || {});
        }
    }

    function getSchemaIconClass(type, mode) {
        if (mode === 'custom') {
            return 'dashicons-media-code';
        }
        var map = {
            article: 'dashicons-media-document',
            blog_posting: 'dashicons-welcome-write-blog',
            news_article: 'dashicons-megaphone',
            product: 'dashicons-cart',
            review: 'dashicons-star-filled',
            recipe: 'dashicons-carrot',
            faq: 'dashicons-editor-help',
            how_to: 'dashicons-welcome-learn-more',
            breadcrumb: 'dashicons-menu',
            table: 'dashicons-editor-table',
            service: 'dashicons-hammer',
            job_posting: 'dashicons-id-alt',
            course: 'dashicons-welcome-learn-more',
            event: 'dashicons-calendar-alt',
            local_business: 'dashicons-location',
            organization: 'dashicons-building',
            person: 'dashicons-admin-users',
            video: 'dashicons-video-alt3',
            book: 'dashicons-book-alt',
            software: 'dashicons-desktop'
        };
        return map[type] || 'dashicons-tag';
    }

    function normalizeSchemaType(type) {
        if (!type) {
            return '';
        }
        var raw = Array.isArray(type) ? type[0] : type;
        if (!raw) {
            return '';
        }
        var normalized = String(raw).trim();
        var map = {
            Article: 'article',
            BlogPosting: 'blog_posting',
            NewsArticle: 'news_article',
            FAQPage: 'faq',
            HowTo: 'how_to',
            VideoObject: 'video',
            Product: 'product',
            Review: 'review',
            Recipe: 'recipe',
            Event: 'event',
            Course: 'course',
            LocalBusiness: 'local_business',
            Organization: 'organization',
            Person: 'person',
            Book: 'book',
            SoftwareApplication: 'software',
            BreadcrumbList: 'breadcrumb',
            Table: 'table',
            Service: 'service',
            JobPosting: 'job_posting'
        };
        return map[normalized] || '';
    }

    function extractName(value) {
        if (!value) {
            return '';
        }
        if (typeof value === 'string') {
            return value;
        }
        if (Array.isArray(value)) {
            return extractName(value[0]);
        }
        return value.name || '';
    }

    function extractUrl(value) {
        if (!value) {
            return '';
        }
        if (typeof value === 'string') {
            return value;
        }
        if (Array.isArray(value)) {
            return extractUrl(value[0]);
        }
        return value.url || value['@id'] || '';
    }

    function toDateTimeLocal(value) {
        if (!value) {
            return '';
        }
        var str = String(value);
        if (str.length >= 16 && str.indexOf('T') > -1) {
            return str.slice(0, 16);
        }
        if (str.length === 10) {
            return str + 'T00:00';
        }
        return str;
    }

    function parseJsonLd(input) {
        if (!input) {
            return null;
        }
        var cleaned = input.replace(/<script[^>]*>/gi, '').replace(/<\/script>/gi, '');
        try {
            var parsed = JSON.parse(cleaned);
            $('#seo-fury-schema-error').hide().text('');
            return parsed;
        } catch (e) {
            var message = e && e.message ? e.message : 'Invalid JSON';
            $('#seo-fury-schema-error').text(message).show();
            return null;
        }
    }

    function findSchemaNode(parsed) {
        if (!parsed) {
            return null;
        }
        if (Array.isArray(parsed)) {
            return parsed.find(function (item) { return item && item['@type']; }) || parsed[0] || null;
        }
        if (parsed['@graph'] && Array.isArray(parsed['@graph'])) {
            return parsed['@graph'].find(function (item) { return item && item['@type']; }) || parsed['@graph'][0] || null;
        }
        return parsed;
    }

    function extractSchemaData(schemaType, node) {
        var data = {};
        if (!node) {
            return data;
        }
        if (schemaType === 'article' || schemaType === 'blog_posting' || schemaType === 'news_article') {
            if (node.headline) data.headline = node.headline;
            if (node.author) data.author = extractName(node.author);
            if (node.publisher) data.publisher = extractName(node.publisher);
            if (node.publisher && node.publisher.logo) data.publisher_logo = extractUrl(node.publisher.logo);
            if (node.image) data.image = extractUrl(node.image);
            if (node.datePublished) data.date_published = toDateTimeLocal(node.datePublished);
            if (node.dateModified) data.date_modified = toDateTimeLocal(node.dateModified);
        } else if (schemaType === 'product') {
            if (node.name) data.name = node.name;
            if (node.offers) {
                var offers = Array.isArray(node.offers) ? node.offers[0] : node.offers;
                if (offers && offers.price) data.price = offers.price;
                if (offers && offers.priceCurrency) data.currency = offers.priceCurrency;
                if (offers && offers.availability) {
                    data.availability = String(offers.availability).split('/').pop();
                }
            }
            if (node.brand) data.brand = extractName(node.brand);
            if (node.sku) data.sku = node.sku;
        } else if (schemaType === 'review') {
            if (node.itemReviewed) data.item_name = extractName(node.itemReviewed);
            if (node.reviewRating && node.reviewRating.ratingValue) data.rating = node.reviewRating.ratingValue;
            if (node.author) data.author = extractName(node.author);
            if (node.datePublished) data.review_date = String(node.datePublished).slice(0, 10);
        } else if (schemaType === 'recipe') {
            if (node.name) data.name = node.name;
            if (node.prepTime) data.prep_time = node.prepTime;
            if (node.cookTime) data.cook_time = node.cookTime;
            if (node.recipeYield) data.recipe_yield = node.recipeYield;
            if (node.nutrition && node.nutrition.calories) data.calories = node.nutrition.calories;
            if (node.recipeIngredient) data.ingredients = Array.isArray(node.recipeIngredient) ? node.recipeIngredient.join('\n') : node.recipeIngredient;
            if (node.recipeInstructions) {
                if (Array.isArray(node.recipeInstructions)) {
                    var instructions = node.recipeInstructions.map(function (step) {
                        return typeof step === 'string' ? step : (step.text || step.name || '');
                    });
                    data.instructions = instructions.filter(Boolean).join('\n');
                } else {
                    data.instructions = node.recipeInstructions;
                }
            }
        } else if (schemaType === 'faq') {
            if (Array.isArray(node.mainEntity)) {
                data.faq_items = node.mainEntity.map(function (entry) {
                    return {
                        question: entry.name || '',
                        answer: entry.acceptedAnswer ? (entry.acceptedAnswer.text || '') : ''
                    };
                }).filter(function (entry) { return entry.question || entry.answer; });
            }
        } else if (schemaType === 'how_to') {
            if (node.name) data.name = node.name;
            if (node.description) data.description = node.description;
            if (Array.isArray(node.step)) {
                data.steps = node.step.map(function (step) {
                    return {
                        name: step.name || '',
                        text: step.text || ''
                    };
                }).filter(function (entry) { return entry.name || entry.text; });
            }
        } else if (schemaType === 'event') {
            if (node.name) data.name = node.name;
            if (node.startDate) data.start_date = toDateTimeLocal(node.startDate);
            if (node.endDate) data.end_date = toDateTimeLocal(node.endDate);
            if (node.location) {
                data.location_name = extractName(node.location);
                if (node.location.address) data.address = typeof node.location.address === 'string' ? node.location.address : (node.location.address.streetAddress || '');
            }
            if (node.offers) {
                var eventOffer = Array.isArray(node.offers) ? node.offers[0] : node.offers;
                if (eventOffer && eventOffer.price) data.price = eventOffer.price;
            }
        } else if (schemaType === 'video') {
            if (node.name) data.name = node.name;
            if (node.description) data.description = node.description;
        } else if (schemaType === 'breadcrumb') {
            if (Array.isArray(node.itemListElement)) {
                var lines = node.itemListElement.map(function (item) {
                    var name = item.name || '';
                    var url = item.item || (item.item && item.item['@id']) || '';
                    if (name && url) {
                        return name + ' | ' + url;
                    }
                    return '';
                }).filter(Boolean);
                data.items = lines.join('\n');
            }
        } else if (schemaType === 'service') {
            if (node.name) data.name = node.name;
            if (node.description) data.description = node.description;
            if (node.provider) data.provider = extractName(node.provider);
            if (node.areaServed) data.area_served = extractName(node.areaServed) || node.areaServed;
            if (node.serviceType) data.service_type = node.serviceType;
        } else if (schemaType === 'job_posting') {
            if (node.title) data.title = node.title;
            if (node.description) data.description = node.description;
            if (node.datePosted) data.date_posted = String(node.datePosted).slice(0, 10);
            if (node.validThrough) data.valid_through = String(node.validThrough).slice(0, 10);
            if (node.employmentType) data.employment_type = node.employmentType;
            if (node.hiringOrganization) data.hiring_org = extractName(node.hiringOrganization);
            if (node.jobLocation) data.job_location = extractName(node.jobLocation.address) || extractName(node.jobLocation);
        } else if (schemaType === 'table') {
            if (node.name) data.name = node.name;
            if (node.description) data.description = node.description;
            if (node.tableRows) data.rows = Array.isArray(node.tableRows) ? node.tableRows.join('\n') : node.tableRows;
        }
        return data;
    }

    function renderSchemaLists() {
        var $lists = $('.seo-fury-schema-list');
        var labelMap = {};
        $('#my_seo_schema_type option').each(function () {
            var value = $(this).attr('value');
            var label = $(this).text();
            labelMap[value] = label;
        });
        $lists.each(function () {
            var $list = $(this);
            if (!schemaItems.length) {
                $list.html('<p class="description">' + seoFuryData.i18n.noSchemaSelected + '</p>');
                return;
            }
            var html = schemaItems.map(function (item) {
                var label = item.mode === 'custom' ? seoFuryData.i18n.customJsonLd : (item.label || labelMap[item.type] || item.type);
                var icon = getSchemaIconClass(item.type, item.mode);
                return '' +
                    '<div class="seo-fury-schema-card ' + (item.id === getActiveSchemaId() ? 'is-active' : '') + '" data-schema-id="' + item.id + '">' +
                    '<div class="seo-fury-schema-label">' +
                    '<span class="seo-fury-schema-icon dashicons ' + icon + '" aria-hidden="true"></span>' +
                    '<span class="seo-fury-schema-badge">' + label + '</span>' +
                    '<span class="seo-fury-schema-name seo-fury-sr-only">' + label + '</span>' +
                    '</div>' +
                    '<div class="seo-fury-schema-actions">' +
                    '<a href="#" class="seo-fury-icon-link seo-fury-schema-preview" data-schema-id="' + item.id + '" title="' + seoFuryData.i18n.preview + '" aria-label="' + seoFuryData.i18n.preview + '">' +
                    '<span class="dashicons dashicons-visibility"></span>' +
                    '</a>' +
                    '<a href="#" class="seo-fury-icon-link seo-fury-schema-edit" data-schema-id="' + item.id + '" title="' + seoFuryData.i18n.edit + '" aria-label="' + seoFuryData.i18n.edit + '">' +
                    '<span class="dashicons dashicons-edit"></span>' +
                    '</a>' +
                    '<a href="#" class="seo-fury-icon-link seo-fury-schema-remove" data-schema-id="' + item.id + '" title="' + seoFuryData.i18n.remove + '" aria-label="' + seoFuryData.i18n.remove + '">' +
                    '<span class="dashicons dashicons-trash"></span>' +
                    '</a>' +
                    '</div>' +
                    '</div>';
            }).join('');
            $list.html(html);
        });
    }


    function updateSchemaUI() {
        var mode = $('#my_seo_schema_mode').val() || 'generator';
        var type = $('#my_seo_schema_type').val() || 'none';

        $('.seo-fury-schema-fields').hide();
        $('.seo-fury-schema-fields[data-schema-type="' + type + '"]').show();
        $('.seo-fury-schema-type-card').removeClass('is-active');
        $('.seo-fury-schema-type-card[data-schema-type="' + type + '"]').addClass('is-active');

        if (mode === 'custom') {
            $('#schema-preview-section').hide();
            $('#schema-custom-preview-section').show();
        } else {
            $('#schema-custom-preview-section').hide();
            $('#schema-preview-section').toggle(type !== 'none');
        }
    }

    function openSchemaModal(tab, focusSettings) {
        $('#seo-fury-schema-modal').show().css('display', 'block');
        $('body').addClass('modal-open').css('overflow', 'hidden');
        if (tab) {
            $('.seo-fury-schema-modal-tab[data-schema-tab="' + tab + '"]').trigger('click');
        }
        renderSchemaLists();
        updateSchemaUI();
        if (focusSettings) {
            var $target = $('#seo-fury-schema-settings');
            if ($target.length) {
                var $body = $('#seo-fury-schema-modal .seo-fury-modal-body');
                $body.animate({ scrollTop: $body.scrollTop() + $target.position().top - 8 }, 200);
                $target.addClass('seo-fury-highlight');
                setTimeout(function () {
                    $target.removeClass('seo-fury-highlight');
                }, 1000);
            }
        }
    }

    loadSchemaItems();
    renderSchemaLists();
    applyActiveSchemaItem(getActiveSchemaItem());
    updateSchemaUI();

    $(document).on('click', '.seo-fury-open-schema-modal', function (e) {
        if (e && e.preventDefault) {
            e.preventDefault();
        }
        openSchemaModal('templates');
    });

    $(document).on('click', '.seo-fury-schema-modal-close, #seo-fury-schema-modal .seo-fury-modal-overlay', function (e) {
        e.preventDefault();
        storeActiveSchemaData();
        $('#seo-fury-schema-modal').hide();
        $('body').removeClass('modal-open').css('overflow', '');
    });

    $(document).on('click', '.seo-fury-schema-modal-tab', function () {
        var tab = $(this).data('schema-tab');
        $('.seo-fury-schema-modal-tab').removeClass('active');
        $('.seo-fury-schema-modal-panel').removeClass('active');
        $(this).addClass('active');
        $('.seo-fury-schema-modal-panel[data-schema-panel="' + tab + '"]').addClass('active');
        if (tab === 'import') {
            setSchemaMode('custom');
            setSchemaType('none');
        } else {
            setSchemaMode('generator');
        }
        updateSchemaUI();
    });

    $(document).on('click', '.seo-fury-schema-apply', function () {
        storeActiveSchemaData();
        var type = $(this).data('schema-type');
        var label = $(this).closest('.seo-fury-schema-type-card').find('.seo-fury-schema-type-name').text() || type;
        var existing = schemaItems.find(function (item) {
            return item.mode === 'generator' && item.type === type;
        });
        if (existing) {
            setActiveSchemaId(existing.id);
            applyActiveSchemaItem(existing);
            saveSchemaItems();
            renderSchemaLists();
            updateSchemaUI();
            return;
        }
        var item = {
            id: 'schema_' + Date.now(),
            type: type,
            mode: 'generator',
            label: label,
            data: {},
            custom: ''
        };
        schemaItems.push(item);
        setActiveSchemaId(item.id);
        setSchemaType(type);
        setSchemaMode('generator');
        clearSchemaDataInputs();
        saveSchemaItems();
        renderSchemaLists();
        updateSchemaUI();
    });

    $(document).on('click', '.seo-fury-schema-import-apply', function () {
        storeActiveSchemaData();
        var custom = $('#my_seo_schema_custom').val() || '';
        $('#seo-fury-schema-error').hide().text('');
        if (!custom.trim()) {
            $('#seo-fury-schema-error').text(seoFuryData.i18n.jsonLdEmpty).show();
            return;
        }
        var parsed = parseJsonLd(custom);
        if (!parsed) {
            return;
        }
        var node = findSchemaNode(parsed);
        var detectedType = node ? normalizeSchemaType(node['@type']) : '';
        if (detectedType) {
            var labelMap = {};
            $('#my_seo_schema_type option').each(function () {
                labelMap[$(this).attr('value')] = $(this).text();
            });
            var existing = schemaItems.find(function (item) {
                return item.mode === 'generator' && item.type === detectedType;
            });
            var data = extractSchemaData(detectedType, node);
            if (existing) {
                existing.data = data;
                setActiveSchemaId(existing.id);
                applyActiveSchemaItem(existing);
            } else {
                var item = {
                    id: 'schema_' + Date.now(),
                    type: detectedType,
                    mode: 'generator',
                    label: labelMap[detectedType] || detectedType,
                    data: data,
                    custom: ''
                };
                schemaItems.push(item);
                setActiveSchemaId(item.id);
                applyActiveSchemaItem(item);
            }
            setSchemaMode('generator');
            setSchemaType(detectedType);
            saveSchemaItems();
            renderSchemaLists();
            updateSchemaUI();
            $('#my_seo_schema_custom').val('');
            $('#schema-custom-json-output').text(seoFuryData.i18n.pasteJsonLd);
            return;
        }

        var item = {
            id: 'schema_' + Date.now(),
            type: 'custom',
            mode: 'custom',
            label: seoFuryData.i18n.customJsonLd,
            data: {},
            custom: custom
        };
        schemaItems.push(item);
        setActiveSchemaId(item.id);
        setSchemaMode('custom');
        setSchemaType('none');
        saveSchemaItems();
        renderSchemaLists();
        updateSchemaUI();
        $('#schema-custom-json-output').text(custom || seoFuryData.i18n.pasteJsonLd);
        $('#my_seo_schema_custom').val('');
    });

    $(document).on('click', '.seo-fury-schema-edit', function (e) {
        if (e && e.preventDefault) {
            e.preventDefault();
        }
        if (e && e.stopPropagation) {
            e.stopPropagation();
        }
        storeActiveSchemaData();
        var id = $(this).data('schema-id');
        setActiveSchemaId(id);
        var item = getActiveSchemaItem();
        applyActiveSchemaItem(item);
        saveSchemaItems();
        renderSchemaLists();
        openSchemaModal(item && item.mode === 'custom' ? 'import' : 'templates', item && item.mode !== 'custom');
    });

    $(document).on('click', '.seo-fury-schema-preview', function (e) {
        if (e && e.preventDefault) {
            e.preventDefault();
        }
        if (e && e.stopPropagation) {
            e.stopPropagation();
        }
        storeActiveSchemaData();
        var id = $(this).data('schema-id');
        setActiveSchemaId(id);
        var item = getActiveSchemaItem();
        applyActiveSchemaItem(item);
        saveSchemaItems();
        renderSchemaLists();
        var targetTab = item && item.mode === 'custom' ? 'import' : 'templates';
        openSchemaModal(targetTab, false);
        var $target = item && item.mode === 'custom'
            ? $('#schema-custom-preview-section')
            : $('#schema-preview-section');
        if ($target.length) {
            var $body = $('#seo-fury-schema-modal .seo-fury-modal-body');
            $body.animate({ scrollTop: $body.scrollTop() + $target.position().top - 8 }, 200);
            $target.addClass('seo-fury-highlight');
            setTimeout(function () {
                $target.removeClass('seo-fury-highlight');
            }, 1000);
        }
    });

    $(document).on('click', '.seo-fury-schema-card', function (e) {
        if ($(e.target).closest('.seo-fury-schema-actions').length) {
            return;
        }
        if ($(e.target).closest('.seo-fury-schema-drag').length) {
            return;
        }
        var id = $(this).data('schema-id');
        if (!id) {
            return;
        }
        storeActiveSchemaData();
        setActiveSchemaId(id);
        var item = getActiveSchemaItem();
        applyActiveSchemaItem(item);
        saveSchemaItems();
        renderSchemaLists();
        openSchemaModal(item && item.mode === 'custom' ? 'import' : 'templates', item && item.mode !== 'custom');
    });

    $(document).on('click', '.seo-fury-schema-remove', function () {
        var id = $(this).data('schema-id');
        schemaItems = schemaItems.filter(function (item) { return item.id !== id; });
        if (schemaItems.length) {
            setActiveSchemaId(schemaItems[0].id);
        } else {
            setActiveSchemaId('');
            setSchemaMode('generator');
            setSchemaType('none');
        }
        saveSchemaItems();
        renderSchemaLists();
        applyActiveSchemaItem(getActiveSchemaItem());
        updateSchemaUI();
    });


    $(document).on('input', '#my_seo_schema_custom', function () {
        var value = $(this).val() || seoFuryData.i18n.pasteJsonLd;
        $('#seo-fury-schema-error').hide().text('');
        $('#schema-custom-json-output').text(value);
    });

    $(document).on('submit', '#post', function () {
        storeActiveSchemaData();
        saveSchemaItems();
    });

    if (typeof wp !== 'undefined' && wp.data && wp.data.subscribe) {
        var lastTitle = getPostTitle();
        var lastExcerpt = getPostExcerpt();
        var lastContent = getPostContent();
        var lastSlug = getPostSlug();
        wp.data.subscribe(function () {
            var nextTitle = getPostTitle();
            var nextExcerpt = getPostExcerpt();
            var nextContent = getPostContent();
            var nextSlug = getPostSlug();
            if (nextTitle !== lastTitle || nextExcerpt !== lastExcerpt || nextContent !== lastContent || nextSlug !== lastSlug) {
                lastTitle = nextTitle;
                lastExcerpt = nextExcerpt;
                lastContent = nextContent;
                lastSlug = nextSlug;
                debouncedCharCount();
                debouncedGooglePreview();
                debouncedOgPreview();
                debouncedFacebookPreview();
                debouncedTwitterPreview();
                debouncedSeoAnalysis();
                debouncedReadability();
                debouncedExtraSeo();
            }
        });
    }

    console.log('SEO plugin JS loaded');
    console.log('Edit buttons found:', $('.seo-fury-edit-snippet').length);
    console.log('Modals found:', $('#seo-fury-snippet-modal').length);
    console.log('Variables found:', $('.variable-item').length);

    $(document).on('click', '.shortcode-dropdown-btn', function (e) {
        e.preventDefault();
        e.stopPropagation();

        var $btn = $(this);
        var $menu = $btn.siblings('.shortcode-dropdown-menu');
        var targetId = $btn.data('target');

        $('.shortcode-dropdown-menu').not($menu).removeClass('active');
        $('.shortcode-dropdown-btn').not($btn).removeClass('active');

        $btn.toggleClass('active');
        $menu.toggleClass('active');

        $('#' + targetId).focus();

        console.log('Dropdown toggled for:', targetId);
    });

    $(document).on('click', '.shortcode-item', function (e) {
        e.preventDefault();
        e.stopPropagation();

        var variable = $(this).data('var');
        var $menu = $(this).closest('.shortcode-dropdown-menu');
        var $dropdown = $menu.closest('.input-with-dropdown');
        var $btn = $dropdown.find('.shortcode-dropdown-btn');
        var targetId = $btn.data('target');
        var $field = $('#' + targetId);

        console.log('Inserting variable:', variable, 'into field:', targetId);

        if ($field.length) {
            var cursorPos = $field[0].selectionStart || 0;
            var currentVal = $field.val();

            var valueToInsert = variable + ' ';
            var newVal = currentVal.substring(0, cursorPos) + valueToInsert + currentVal.substring(cursorPos);

            $field.val(newVal);

            var newCursorPos = cursorPos + valueToInsert.length;
            $field[0].setSelectionRange(newCursorPos, newCursorPos);

            $field.trigger('input');
            $field.focus();

            console.log('Variable inserted successfully with space');
        }
    });


    $(document).on('click', function (e) {
        if (!$(e.target).closest('.input-with-dropdown').length) {
            $('.shortcode-dropdown-menu').removeClass('active');
            $('.shortcode-dropdown-btn').removeClass('active');
        }
    });

    $(document).on('keydown', function (e) {
        if (e.key === 'Escape') {
            $('.shortcode-dropdown-menu').removeClass('active');
            $('.shortcode-dropdown-btn').removeClass('active');
        }
    });

    $('#seo-fury-scan-links').on('click', function () {
        var $btn = $(this);
        var $loading = $('#seo-fury-link-loading');
        var $suggestions = $('#seo-fury-link-suggestions');
        var $empty = $('#seo-fury-link-scan-empty');
        var $orphans = $('#seo-fury-orphan-pages');

        $btn.prop('disabled', true).text(seoFuryData.i18n.scanning);
        $loading.show();
        $suggestions.hide();
        $empty.hide();
        $orphans.hide();

        $.ajax({
            url: seoFuryData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'seo_fury_scan_internal_links',
                nonce: seoFuryData.internalLinksNonce,
                post_id: seoFuryData.postId
            },
            success: function (response) {
                $loading.hide();
                $btn.prop('disabled', false).text(seoFuryData.i18n.scanContent);

                if (!response.success) return;

                var data = response.data;
                $('#sf-stat-internal').text(data.stats.internal);
                $('#sf-stat-external').text(data.stats.external);
                $('#sf-stat-suggestions').text(data.suggestions.length);

                if (data.suggestions.length > 0) {
                    var html = '';
                    for (var i = 0; i < data.suggestions.length; i++) {
                        var s = data.suggestions[i];
                        var badge = s.match_type === 'exact' ? '<span style="background:#ECFDF5;color:#10B981;font-size:10px;padding:1px 6px;border-radius:4px;">exact</span>' : '<span style="background:#FEF3C7;color:#F59E0B;font-size:10px;padding:1px 6px;border-radius:4px;">partial</span>';
                        html += '<div style="padding:8px 10px;border:1px solid var(--sf-gray-200,#E2E8F0);border-radius:6px;margin-bottom:6px;font-size:12px;">';
                        html += '<div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:4px;">';
                        html += '<strong style="color:var(--sf-gray-900,#1E293B);">&ldquo;' + $('<span>').text(s.match_text).html() + '&rdquo;</strong> ' + badge;
                        html += '</div>';
                        html += '<div style="color:var(--sf-gray-500,#64748B);margin-bottom:4px;">' + $('<span>').text(s.target_title).html() + ' <span style="opacity:0.6;">(' + s.post_type + ')</span></div>';
                        html += '<div style="display:flex;justify-content:space-between;align-items:center;">';
                        html += '<a href="' + s.target_url + '" target="_blank" style="font-size:11px;color:var(--sf-primary,#4F46E5);text-decoration:none;">' + s.target_url.replace(location.origin, '') + '</a>';
                        html += '<button type="button" class="button button-small sf-insert-link" data-url="' + $('<span>').text(s.target_url).html() + '" data-text="' + $('<span>').text(s.match_text).html() + '" style="font-size:11px;padding:0 8px;line-height:24px;">' + seoFuryData.i18n.insertLink + '</button>';
                        html += '</div></div>';
                    }
                    $('#seo-fury-suggestions-list').html(html);
                    $suggestions.show();
                } else {
                    $empty.show();
                }

                if (data.orphan_pages && data.orphan_pages.length > 0) {
                    var ohtml = '';
                    for (var j = 0; j < data.orphan_pages.length; j++) {
                        var op = data.orphan_pages[j];
                        ohtml += '<div style="font-size:12px;padding:4px 0;border-bottom:1px solid var(--sf-gray-100,#F1F5F9);">';
                        ohtml += '<a href="' + op.url + '" target="_blank" style="color:var(--sf-primary,#4F46E5);text-decoration:none;">' + $('<span>').text(op.title).html() + '</a>';
                        ohtml += ' <span style="color:var(--sf-gray-400);font-size:10px;">(' + op.post_type + ')</span>';
                        ohtml += '</div>';
                    }
                    $('#seo-fury-orphan-list').html(ohtml);
                    $orphans.show();
                }
            },
            error: function () {
                $loading.hide();
                $btn.prop('disabled', false).text(seoFuryData.i18n.scanContent);
            }
        });
    });

    $(document).on('click', '.sf-insert-link', function () {
        var url = $(this).data('url');
        var text = $(this).data('text');
        var $btn = $(this);

        if (typeof wp !== 'undefined' && wp.data && wp.data.select('core/block-editor')) {
            var blocks = wp.data.select('core/block-editor').getBlocks();
            var updated = false;
            blocks.forEach(function (block) {
                if (updated) return;
                if (block.attributes && block.attributes.content) {
                    var content = block.attributes.content;
                    var regex = new RegExp('(?<!<a[^>]*>)(' + text.replace(/[.*+?^${}()|[\]\\]/g, '\\$&') + ')(?![^<]*</a>)', 'i');
                    if (regex.test(content)) {
                        var newContent = content.replace(regex, '<a href="' + url + '">$1</a>');
                        wp.data.dispatch('core/block-editor').updateBlockAttributes(block.clientId, { content: newContent });
                        updated = true;
                    }
                }
            });
            if (updated) {
                $btn.text(seoFuryData.i18n.linked).css('color', '#10B981').prop('disabled', true);
            } else {
                $btn.text(seoFuryData.i18n.notFound).css('color', '#d63638');
                setTimeout(function () { $btn.text(seoFuryData.i18n.insertLink).css('color', '').prop('disabled', false); }, 2000);
            }
        } else if (typeof tinymce !== 'undefined' && tinymce.activeEditor) {
            var editor = tinymce.activeEditor;
            var content = editor.getContent();
            var regex = new RegExp('(?<!<a[^>]*>)(' + text.replace(/[.*+?^${}()|[\]\\]/g, '\\$&') + ')(?![^<]*</a>)', 'i');
            if (regex.test(content)) {
                content = content.replace(regex, '<a href="' + url + '">$1</a>');
                editor.setContent(content);
                $btn.text(seoFuryData.i18n.linked).css('color', '#10B981').prop('disabled', true);
            } else {
                $btn.text(seoFuryData.i18n.notFound).css('color', '#d63638');
                setTimeout(function () { $btn.text(seoFuryData.i18n.insertLink).css('color', '').prop('disabled', false); }, 2000);
            }
        }
    });

});
