jQuery(document).ready(function($) {
    
    if (typeof seoRedirects === 'undefined') {
        console.warn('SEO Plugin: Redirects scripts not loaded properly');
        return;
    }
    
    loadRedirects();
    
    function loadRedirects() {
        $.ajax({
            url: seoRedirects.ajax_url,
            type: 'POST',
            data: {
                action: 'seo_load_redirects',
                nonce: seoRedirects.nonce
            },
            beforeSend: function() {
                $('#redirects-table tbody').html('<tr><td colspan="8" style="text-align: center;">' + seoRedirects.i18n.loading + '</td></tr>');
            },
            success: function(response) {
                if (response.success) {
                    var redirects = response.data.redirects;
                    var html = '';
                    
                    if (redirects.length === 0) {
                        html = '<tr><td colspan="8" style="text-align: center;">' + seoRedirects.i18n.noRedirectsFound + '</td></tr>';
                    } else {
                        $.each(redirects, function(index, redirect) {
                            var targetUrl = redirect.target_url || 'N/A';
                            var sourceUrl = redirect.source_url || 'N/A';
                            var redirectType = redirect.redirect_type || '301';
                            var hits = redirect.hits || '0';
                            
                            var isActive = (redirect.active === "1" || redirect.active === 1);
                            
                            var statusBadge = isActive
                                ? '<span style="color: #00a32a; font-weight: 600;">&#10003; ' + seoRedirects.i18n.active + '</span>'
                                : '<span style="color: #d63638; font-weight: 600;">&#10005; ' + seoRedirects.i18n.inactive + '</span>';
                            
                            var createdDate = 'N/A';
                            if (redirect.created && redirect.created !== '0000-00-00 00:00:00') {
                                try {
                                    var date = new Date(redirect.created.replace(' ', 'T'));
                                    createdDate = date.toLocaleString('en-US', {
                                        year: 'numeric',
                                        month: '2-digit',
                                        day: '2-digit',
                                        hour: '2-digit',
                                        minute: '2-digit'
                                    });
                                } catch(e) {
                                    createdDate = redirect.created;
                                }
                            }
                            
                            html += '<tr>';
                            html += '<td>' + redirect.id + '</td>';
                            html += '<td><code>' + sourceUrl + '</code></td>';
                            html += '<td><code>' + targetUrl + '</code></td>';
                            html += '<td>' + redirectType + '</td>';
                            html += '<td>' + hits + '</td>';
                            html += '<td>' + createdDate + '</td>';
                            html += '<td>' + statusBadge + '</td>';
                            html += '<td class="redirect-actions">';
                            html += '<button class="button button-small edit-redirect" data-id="' + redirect.id + '" data-source="' + sourceUrl + '" data-target="' + targetUrl + '" data-type="' + redirectType + '">' + seoRedirects.i18n.edit + '</button> ';
                            
                            var toggleText = isActive ? seoRedirects.i18n.deactivate : seoRedirects.i18n.activate;
                            var toggleValue = isActive ? 0 : 1;
                            html += '<button class="button button-small toggle-redirect" data-id="' + redirect.id + '" data-active="' + toggleValue + '">' + toggleText + '</button> ';
                            
                            html += '<button class="button button-small delete-redirect" data-id="' + redirect.id + '" style="color: #d63638;">' + seoRedirects.i18n.delete + '</button>';
                            html += '</td>';
                            html += '</tr>';
                        });
                    }
                    
                    $('#redirects-table tbody').html(html);
                } else {
                    var errorMsg = response.data && response.data.message ? response.data.message : seoRedirects.i18n.errorLoading;
                    $('#redirects-table tbody').html('<tr><td colspan="8" style="text-align: center; color: #d63638;">' + errorMsg + '</td></tr>');
                }
            },
            error: function() {
                $('#redirects-table tbody').html('<tr><td colspan="8" style="text-align: center; color: #d63638;">' + seoRedirects.i18n.errorLoadingData + '</td></tr>');
            }
        });
    }
    
    $('#add-redirect-btn').on('click', function() {
        var source = $('#redirect-source').val().trim();
        var target = $('#redirect-target').val().trim();
        var type = $('#redirect-type').val();
        
        if (!source || !target) {
            alert(seoRedirects.i18n.fillAllFields);
            return;
        }

        var btn = $(this);
        var originalText = btn.text();

        btn.prop('disabled', true).text(seoRedirects.i18n.adding);
        
        $.ajax({
            url: seoRedirects.ajax_url,
            type: 'POST',
            data: {
                action: 'seo_add_redirect',
                nonce: seoRedirects.nonce,
                source: source,
                target: target,
                type: type
            },
            success: function(response) {
                console.log('Full response:', response);
                console.log('Response data:', response.data);
                console.log('Type of data:', typeof response.data);
                
                if (response.success) {
                    var message = response.data && typeof response.data === 'string' ? response.data : seoRedirects.i18n.redirectAdded;
                    alert(message);
                    $('#redirect-source').val('');
                    $('#redirect-target').val('');
                    $('#redirect-type').val('301');
                    loadRedirects();
                } else {
                    var errorMessage = typeof response.data === 'string' ? response.data : (response.data && response.data.message ? response.data.message : seoRedirects.i18n.anErrorOccurred);
                    alert(seoRedirects.i18n.errorPrefix + ' ' + errorMessage);
                }
            },
            error: function() {
                alert(seoRedirects.i18n.errorAdding);
            },
            complete: function() {
                btn.prop('disabled', false).text(originalText);
            }
        });
    });

    $('#seo-export-redirects').on('click', function() {
        var url = seoRedirects.ajax_url + '?action=seo_export_redirects_csv&nonce=' + encodeURIComponent(seoRedirects.nonce);
        window.location.href = url;
    });

    $('#seo-import-redirects').on('click', function() {
        var fileInput = $('#seo-import-redirects-file')[0];
        var resultBox = $('#seo-import-redirects-result');
        if (!fileInput.files.length) {
            alert(seoRedirects.i18n.selectCsvFile);
            return;
        }

        var formData = new FormData();
        formData.append('action', 'seo_import_redirects_csv');
        formData.append('nonce', seoRedirects.nonce);
        formData.append('file', fileInput.files[0]);

        resultBox.html('');
        $.ajax({
            url: seoRedirects.ajax_url,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success) {
                    var msg = seoRedirects.i18n.imported + ': ' + response.data.imported +
                              ', ' + seoRedirects.i18n.updated + ': ' + response.data.updated +
                              ', ' + seoRedirects.i18n.skipped + ': ' + response.data.skipped;
                    resultBox.html('<div class="seo-alert success"><strong>✅ ' + seoRedirects.i18n.csvImportCompleted + '</strong><br>' + msg + '</div>');
                    loadRedirects();
                } else {
                    resultBox.html('<div class="seo-alert error"><strong>❌ ' + seoRedirects.i18n.error + ':</strong> ' + response.data + '</div>');
                }
            },
            error: function() {
                resultBox.html('<div class="seo-alert error"><strong>❌ ' + seoRedirects.i18n.error + ':</strong> ' + seoRedirects.i18n.uploadFailed + '</div>');
            }
        });
    });
    
    $(document).on('click', '.delete-redirect', function() {
        if (!confirm(seoRedirects.i18n.confirmDelete)) {
            return;
        }
        
        var id = $(this).data('id');
        var btn = $(this);
        
        btn.prop('disabled', true).text(seoRedirects.i18n.deleting);
        
        $.ajax({
            url: seoRedirects.ajax_url,
            type: 'POST',
            data: {
                action: 'seo_delete_redirect',
                nonce: seoRedirects.nonce,
                id: id
            },
            success: function(response) {
                if (response.success) {
                    alert(seoRedirects.i18n.redirectDeleted);
                    loadRedirects();
                } else {
                    alert(seoRedirects.i18n.errorPrefix + ' ' + response.data);
                    btn.prop('disabled', false).text(seoRedirects.i18n.delete);
                }
            },
            error: function() {
                alert(seoRedirects.i18n.errorDeleting);
                btn.prop('disabled', false).text(seoRedirects.i18n.delete);
            }
        });
    });
    
    $(document).on('click', '.toggle-redirect', function() {
        var id = $(this).data('id');
        var active = $(this).data('active');
        var btn = $(this);
        
        btn.prop('disabled', true).text(seoRedirects.i18n.updating);
        
        $.ajax({
            url: seoRedirects.ajax_url,
            type: 'POST',
            data: {
                action: 'seo_toggle_redirect',
                nonce: seoRedirects.nonce,
                id: id,
                active: active
            },
            success: function(response) {
                if (response.success) {
                    loadRedirects();
                } else {
                    alert(seoRedirects.i18n.errorPrefix + ' ' + response.data);
                    btn.prop('disabled', false);
                }
            },
            error: function() {
                alert(seoRedirects.i18n.errorUpdatingStatus);
                btn.prop('disabled', false);
            }
        });
    });
    
    $(document).on('click', '.edit-redirect', function() {
        var id = $(this).data('id');
        var source = $(this).data('source');
        var target = $(this).data('target');
        var type = $(this).data('type');
        
        $('#edit-redirect-id').val(id);
        $('#edit-redirect-source').val(source);
        $('#edit-redirect-target').val(target);
        $('#edit-redirect-type').val(type);
        
        $('#edit-redirect-modal').fadeIn();
    });
    
    $('.seo-modal-close, .seo-modal').on('click', function(e) {
        if (e.target === this) {
            $('#edit-redirect-modal').fadeOut();
        }
    });
    
    $('#update-redirect-btn').on('click', function() {
        var id = $('#edit-redirect-id').val();
        var source = $('#edit-redirect-source').val().trim();
        var target = $('#edit-redirect-target').val().trim();
        var type = $('#edit-redirect-type').val();
        
        if (!source || !target) {
            alert(seoRedirects.i18n.fillAllFields);
            return;
        }

        var btn = $(this);
        var originalText = btn.text();

        btn.prop('disabled', true).text(seoRedirects.i18n.saving);
        
        $.ajax({
            url: seoRedirects.ajax_url,
            type: 'POST',
            data: {
                action: 'seo_update_redirect',
                nonce: seoRedirects.nonce,
                id: id,
                source: source,
                target: target,
                type: type
            },
            success: function(response) {
                if (response.success) {
                    alert(seoRedirects.i18n.redirectUpdated);
                    $('#edit-redirect-modal').fadeOut();
                    loadRedirects();
                } else {
                    alert(seoRedirects.i18n.errorPrefix + ' ' + response.data);
                }
            },
            error: function() {
                alert(seoRedirects.i18n.errorUpdating);
            },
            complete: function() {
                btn.prop('disabled', false).text(originalText);
            }
        });
    });
    
});
