<?php

if (!defined('ABSPATH')) {
    exit;
}

class My_SEO_Plugin_Admin
{

    private $core;

    public function __construct($core)
    {
        $this->core = $core;
        $this->init_hooks();
    }

    private function init_hooks()
    {
        add_action('admin_menu', array($this, 'add_admin_menu'));

        add_action('admin_init', array($this, 'register_settings'));

        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

        add_action('update_option_my_seo_html_sitemap_slug', array($this, 'flush_on_slug_change'), 10, 2);
        add_action('update_option_my_seo_sitemap_enabled', array($this, 'flush_on_sitemap_toggle'), 10, 2);

        add_action('update_option_my_seo_sitemap_post_types', array($this, 'flush_on_post_types_change'), 10, 2);
        add_action('update_option_my_seo_html_sitemap_post_types', array($this, 'flush_on_html_post_types_change'), 10, 2);
    }

    public function add_admin_menu()
    {
        $plugin_name = function_exists('seo_fury_get_brand_name') ? seo_fury_get_brand_name() : 'SEO Fury';

        add_menu_page($plugin_name, $plugin_name, 'manage_options', 'my-seo-plugin', array($this, 'settings_page'), 'dashicons-search', 30);
    }

    private function should_hide_branding() {
        return function_exists('seo_fury_hide_branding') ? seo_fury_hide_branding() : false;
    }

    public function enqueue_admin_assets($hook)
    {
        if ($hook !== 'toplevel_page_my-seo-plugin') {
            return;
        }

        wp_enqueue_script('jquery');

        wp_enqueue_style(
            'my-seo-plugin-admin',
            MY_SEO_PLUGIN_URL . 'assets/admin-style.css',
            array(),
            MY_SEO_PLUGIN_VERSION
        );

        wp_enqueue_script(
            'my-seo-plugin-redirects',
            MY_SEO_PLUGIN_URL . 'assets/redirects.js',
            array('jquery'),
            '0.1.4',
            true
        );

        wp_localize_script('my-seo-plugin-redirects', 'seoRedirects', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('seo_redirects_nonce'),
            'i18n' => array(
                'loading' => __('Loading...', 'seo-fury'),
                'noRedirectsFound' => __('No redirects found', 'seo-fury'),
                'active' => __('Active', 'seo-fury'),
                'inactive' => __('Inactive', 'seo-fury'),
                'activate' => __('Activate', 'seo-fury'),
                'deactivate' => __('Deactivate', 'seo-fury'),
                'edit' => __('Edit', 'seo-fury'),
                'delete' => __('Delete', 'seo-fury'),
                'adding' => __('Adding...', 'seo-fury'),
                'deleting' => __('Deleting...', 'seo-fury'),
                'saving' => __('Saving...', 'seo-fury'),
                'updating' => __('Updating...', 'seo-fury'),
                'fillAllFields' => __('Please fill all fields', 'seo-fury'),
                'redirectAdded' => __('Redirect added successfully!', 'seo-fury'),
                'redirectDeleted' => __('Redirect deleted!', 'seo-fury'),
                'redirectUpdated' => __('Redirect updated!', 'seo-fury'),
                'confirmDelete' => __('Are you sure you want to delete this redirect?', 'seo-fury'),
                'errorLoading' => __('Error loading redirects', 'seo-fury'),
                'errorLoadingData' => __('Error loading data', 'seo-fury'),
                'errorAdding' => __('Error adding redirect', 'seo-fury'),
                'errorDeleting' => __('Error deleting redirect', 'seo-fury'),
                'errorUpdating' => __('Error updating redirect', 'seo-fury'),
                'errorUpdatingStatus' => __('Error updating status', 'seo-fury'),
                'selectCsvFile' => __('Please select a CSV file.', 'seo-fury'),
                'csvImportCompleted' => __('CSV import completed.', 'seo-fury'),
                'imported' => __('Imported', 'seo-fury'),
                'updated' => __('Updated', 'seo-fury'),
                'skipped' => __('Skipped', 'seo-fury'),
                'uploadFailed' => __('Upload failed.', 'seo-fury'),
                'error' => __('Error', 'seo-fury'),
                'anErrorOccurred' => __('An error occurred', 'seo-fury'),
                'errorPrefix' => __('Error:', 'seo-fury'),
            ),
        ));

        wp_enqueue_script(
            'my-seo-plugin-indexnow',
            MY_SEO_PLUGIN_URL . 'assets/indexnow.js',
            array('jquery'),
            MY_SEO_PLUGIN_VERSION,
            true
        );

        wp_localize_script('my-seo-plugin-indexnow', 'seoIndexNow', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('seo_redirects_nonce'),
            'i18n' => array(
                'enterUrl' => __('Please enter at least one URL', 'seo-fury'),
                'noValidUrls' => __('No valid URLs found', 'seo-fury'),
                'maxUrls' => __('Maximum 10,000 URLs per request. You have:', 'seo-fury'),
                'sending' => __('Sending...', 'seo-fury'),
                'sendingBatches' => __('Sending in batches...', 'seo-fury'),
                'saving' => __('Saving...', 'seo-fury'),
                'generating' => __('Generating...', 'seo-fury'),
                'checking' => __('Checking...', 'seo-fury'),
                'success' => __('Success!', 'seo-fury'),
                'urlsSent' => __('URLs sent:', 'seo-fury'),
                'error' => __('Error!', 'seo-fury'),
                'unknownError' => __('Unknown error', 'seo-fury'),
                'serverError' => __('Server connection error', 'seo-fury'),
                'settingsSaved' => __('Settings saved', 'seo-fury'),
                'saveError' => __('Save error', 'seo-fury'),
                'keyGenerated' => __('Key generated!', 'seo-fury'),
                'verifyKey' => __('Verify key', 'seo-fury'),
                'noHistoryYet' => __('No history yet', 'seo-fury'),
                'confirmClearHistory' => __('Are you sure you want to clear the entire history?', 'seo-fury'),
                'historyCleared' => __('History cleared', 'seo-fury'),
                'clearFailed' => __('Clear failed', 'seo-fury'),
                'sendingBatch' => __('Sending batch', 'seo-fury'),
                'of' => __('of', 'seo-fury'),
                'allBatchesSent' => __('All batches sent successfully!', 'seo-fury'),
                'totalUrls' => __('Total URLs:', 'seo-fury'),
                'batches' => __('Batches:', 'seo-fury'),
                'completedWithErrors' => __('Completed with errors', 'seo-fury'),
                'errors' => __('Errors:', 'seo-fury'),
                'auto' => __('AUTO', 'seo-fury'),
                'manual' => __('MANUAL', 'seo-fury'),
                'notAvailable' => __('N/A', 'seo-fury'),
            ),
        ));

        wp_enqueue_script(
            'my-seo-plugin-license',
            MY_SEO_PLUGIN_URL . 'assets/license.js',
            array('jquery'),
            MY_SEO_PLUGIN_VERSION,
            true
        );

        wp_localize_script('my-seo-plugin-license', 'seoLicense', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('sf_license_nonce'),
            'i18n' => array(
                'enterLicenseKey' => __('Please enter a license key.', 'seo-fury'),
                'invalidKeyFormat' => __('Invalid key format.', 'seo-fury'),
                'activating' => __('Activating...', 'seo-fury'),
                'licenseActivated' => __('License activated! The page will reload...', 'seo-fury'),
                'activationFailed' => __('License activation failed', 'seo-fury'),
                'activationServerError' => __('Activation server connection error', 'seo-fury'),
                'confirmDeactivate' => __("Are you sure you want to deactivate the license?\n\nAll PRO features will be disabled.", 'seo-fury'),
                'deactivating' => __('Deactivating...', 'seo-fury'),
                'licenseDeactivated' => __('License deactivated. The page will reload...', 'seo-fury'),
                'deactivationFailed' => __('Deactivation failed', 'seo-fury'),
                'serverError' => __('Server connection error', 'seo-fury'),
                'checking' => __('Checking...', 'seo-fury'),
                'licenseActiveValid' => __('License is active and valid!', 'seo-fury'),
                'plan' => __('Plan:', 'seo-fury'),
                'status' => __('Status:', 'seo-fury'),
                'expires' => __('Expires:', 'seo-fury'),
                'licenseInactive' => __('License is inactive or expired.', 'seo-fury'),
                'couldNotReach' => __('Could not reach the license server. Using local cache.', 'seo-fury'),
                'keyCopied' => __('Key copied to clipboard!', 'seo-fury'),
                'clickToCopy' => __('Click to copy', 'seo-fury'),
            ),
        ));

        wp_enqueue_script(
            'my-seo-plugin-404',
            MY_SEO_PLUGIN_URL . 'assets/404-monitor.js',
            array('jquery'),
            MY_SEO_PLUGIN_VERSION,
            true
        );

        wp_localize_script('my-seo-plugin-404', 'seo404Monitor', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('seo_redirects_nonce'),
            'i18n' => array(
                'loading' => __('Loading...', 'seo-fury'),
                'loadError' => __('Load error', 'seo-fury'),
                'serverError' => __('Server connection error', 'seo-fury'),
                'no404Found' => __('No 404 errors found', 'seo-fury'),
                'resolved' => __('Resolved', 'seo-fury'),
                'active' => __('Active', 'seo-fury'),
                'find' => __('Find', 'seo-fury'),
                'redirect' => __('Redirect', 'seo-fury'),
                'resolve' => __('Resolve', 'seo-fury'),
                'delete' => __('Delete', 'seo-fury'),
                'confirmClearAll' => __('Are you sure? All 404 logs will be deleted!', 'seo-fury'),
                'logsCleared' => __('Logs cleared', 'seo-fury'),
                'searchingSimilar' => __('Searching for similar pages...', 'seo-fury'),
                'foundSimilar' => __('Found %d similar pages:', 'seo-fury'),
                'useThisUrl' => __('Use this URL', 'seo-fury'),
                'noSimilarFound' => __('No similar pages found', 'seo-fury'),
                'provideTargetUrl' => __('Please provide a target URL', 'seo-fury'),
                'creating' => __('Creating...', 'seo-fury'),
                'redirectCreated' => __('Redirect created successfully!', 'seo-fury'),
                'errorCreatingRedirect' => __('Error creating redirect', 'seo-fury'),
                'confirmDeleteEntry' => __('Delete this entry?', 'seo-fury'),
            ),
        ));
    }

    public function register_settings()
    {
        if ($this->core) {
            $this->core->get_robots()->register_settings();
            $this->core->get_sitemap()->register_settings();
            $this->core->get_indexnow()->register_settings();
            $this->core->get_general()->register_settings();
            $this->core->get_meta_titles()->register_settings();
            if ($this->core->get_breadcrumbs()) {
                $this->core->get_breadcrumbs()->register_settings();
            }
            if ($this->core->get_local_seo()) {
                $this->core->get_local_seo()->register_settings();
            }

            if ($this->core->get_404_monitor()) {
                $this->core->get_404_monitor()->register_settings();
            }

            if ($this->core->get_woocommerce() && method_exists($this->core->get_woocommerce(), 'register_settings')) {
                $this->core->get_woocommerce()->register_settings();
            }
        }
    }

    public function settings_page()
    {
        ob_start();
        $tabs = array(
            'general' => array(
                'title' => __('General Settings', 'seo-fury'),
                'icon' => 'dashicons-admin-settings'
            ),

            'meta_titles' => array(
                'title' => __('Meta Titles', 'seo-fury'),
                'icon' => 'dashicons-editor-code',
            ),
            'robots' => array(
                'title' => __('Robots.txt', 'seo-fury'),
                'icon' => 'dashicons-admin-generic'
            ),
            'sitemap' => array(
                'title' => __('Sitemap', 'seo-fury'),
                'icon' => 'dashicons-networking'
            ),
            'breadcrumbs' => array(
                'title' => __('Breadcrumbs', 'seo-fury'),
                'icon' => 'dashicons-editor-ol'
            ),
            'local_seo' => array(
                'title' => __('Local SEO', 'seo-fury'),
                'icon' => 'dashicons-location-alt'
            ),
            'ssl' => array(
                'title' => __('SSL & Security', 'seo-fury'),
                'icon' => 'dashicons-shield'
            ),
        );

        if (class_exists('WooCommerce')) {
            $tabs['woocommerce'] = array(
                'title' => __('WooCommerce SEO', 'seo-fury'),
                'icon' => 'dashicons-cart'
            );
        }

        $tabs += array(
            'redirects' => array(
                'title' => __('Redirects', 'seo-fury'),
                'icon' => 'dashicons-randomize'
            ),
            'log404' => array(
                'title' => __('404 Monitor', 'seo-fury'),
                'icon' => 'dashicons-warning'
            ),
            'indexnow' => array(
                'title' => __('IndexNow', 'seo-fury'),
                'icon' => 'dashicons-update'
            ),

            'changelog' => array(
                'title' => __('Changelog', 'seo-fury'),
                'icon' => 'dashicons-text-page'
            ),
            'license' => array(
                'title' => __('License', 'seo-fury'),
                'icon' => 'dashicons-admin-network'
            ),
        );

        $current_tab = (isset($_GET['tab']) && isset($tabs[$_GET['tab']])) ? sanitize_text_field($_GET['tab']) : 'general';

?>
        <div class="wrap seo-plugin-wrap">
            <?php $brand = function_exists('seo_fury_get_brand_name') ? seo_fury_get_brand_name() : 'SEO Fury'; ?>
            <?php $hide_branding = $this->should_hide_branding(); ?>
            <h1 class="seo-plugin-title">
                <span class="dashicons dashicons-search"></span>
                <?php echo esc_html($brand); ?>
            </h1>

            <?php settings_errors(); ?>

            <div class="seo-plugin-container">
                <!-- Left sidebar menu -->
                <div class="seo-sidebar">
                    <nav class="seo-sidebar-nav">
                        <?php foreach ($tabs as $tab_key => $tab_data): ?>
                            <a href="?page=my-seo-plugin&tab=<?php echo esc_attr($tab_key); ?>"
                                class="seo-sidebar-item <?php echo ($current_tab === $tab_key) ? 'active' : ''; ?>">
                                <span class="dashicons <?php echo esc_attr($tab_data['icon']); ?>"></span>
                                <span class="seo-sidebar-item-text"><?php echo esc_html($tab_data['title']); ?></span>
                            </a>
                        <?php endforeach; ?>
                    </nav>

                    <?php if (!$hide_branding): ?>
                        <div class="seo-sidebar-footer">
                            <p><strong><?php echo esc_html($brand); ?></strong></p>
                            <p><?php printf(esc_html__('Version %s', 'seo-fury'), esc_html(MY_SEO_PLUGIN_VERSION)); ?></p>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Right content -->
                <div class="seo-content">
                    <div class="seo-content-header">
                        <h2>
                            <span class="dashicons <?php echo esc_attr($tabs[$current_tab]['icon']); ?>"></span>
                            <?php echo esc_html($tabs[$current_tab]['title']); ?>
                        </h2>
                    </div>

                    <div class="seo-content-body">
                        <?php
                        switch ($current_tab) {
                            case 'general':
                                $this->render_general_tab();
                                break;
                            case 'license':
                                $this->render_license_tab();
                                break;
                            case 'robots':
                                $this->render_robots_tab();
                                break;
                            case 'sitemap':
                                $this->render_sitemap_tab();
                                break;
                            case 'breadcrumbs':
                                $this->render_breadcrumbs_tab();
                                break;
                            case 'local_seo':
                                $this->render_local_seo_tab();
                                break;
                            case 'ssl':
                                $this->render_ssl_tab();
                                break;
                            case 'redirects':
                                $this->render_redirects_tab();
                                break;
                            case 'log404':
                                $this->render_404_tab();
                                break;
                            case 'indexnow':
                                $this->render_indexnow_tab();
                                break;
                            case 'meta_titles':
                                $this->render_meta_titles_tab();
                                break;
                            case 'woocommerce':
                                $this->render_woocommerce_tab();
                                break;
                            case 'changelog':
                                $this->render_changelog_tab();
                                break;
                            default:
                                $this->render_general_tab();
                        }
                        ?>
                    </div>
                </div>
            </div>
        </div>
    <?php
        $output = ob_get_clean();
        echo $this->translate_admin_output($output);
    }

    private function translate_admin_output($output) {
        return $output;
    }

    private function render_general_tab()
    {
        $core = SEO_Fury_Core::get_instance();
        $pro = $core->get_pro_features();
        $is_pro = $pro && $pro->is_pro();
        $hide_branding = $this->should_hide_branding();
    ?>
        <form method="post" action="options.php">
            <?php
            settings_fields('my_seo_general_settings');
            do_settings_sections('my_seo_general_settings');
            submit_button();
            ?>
        </form>

        <div class="seo-card" style="margin-top: 20px;">
            <h3 style="margin-top: 0;"><?php echo '🔗 ' . esc_html__('Internal Linking', 'seo-fury'); ?></h3>
            <p style="color: #64748B; font-size: 13px; margin-bottom: 12px;">
                <?php esc_html_e('Scan your posts and pages for internal linking opportunities. The module analyzes your content and suggests relevant links to other published pages based on keyword and title matching.', 'seo-fury'); ?>
            </p>
            <?php if ($is_pro): ?>
                <div style="background: #ECFDF5; border: 1px solid #10B981; border-radius: 6px; padding: 12px 15px; display: flex; align-items: center; gap: 10px;">
                    <span style="font-size: 20px;">✅</span>
                    <div>
                        <?php printf(
                            '<strong>%s</strong> — %s <strong>%s</strong> %s',
                            esc_html__('Active', 'seo-fury'),
                            esc_html__('Open any post or page in the editor and click the', 'seo-fury'),
                            esc_html__('Links', 'seo-fury'),
                            esc_html__('tab in the SEO Fury sidebar.', 'seo-fury')
                        ); ?>
                    </div>
                </div>
            <?php else: ?>
                <div style="background: #FEF3C7; border: 1px solid #F59E0B; border-radius: 6px; padding: 12px 15px; display: flex; align-items: center; gap: 10px;">
                    <span style="font-size: 20px;">🔒</span>
                    <div>
                        <strong><?php esc_html_e('PRO Feature', 'seo-fury'); ?></strong> — <?php esc_html_e('Available in all PRO plans. Upgrade to unlock smart internal linking suggestions in the post editor.', 'seo-fury'); ?>
                        <?php if (!$hide_branding): ?>
                            <a href="https://seo-fury.com/#pricing" target="_blank" style="font-weight: 600;"><?php esc_html_e('Upgrade', 'seo-fury'); ?> &rarr;</a>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    <?php
    }

    private function render_robots_tab()
    {
    ?>
        <form method="post" action="options.php">
            <?php
            settings_fields('my_seo_robots_group');
            do_settings_sections('my_seo_robots_group');
            submit_button();
            ?>
        </form>
    <?php
    }

    private function render_sitemap_tab()
    {
    ?>
        <form method="post" action="options.php">
            <?php
            settings_fields('my_seo_sitemap_group');
            do_settings_sections('my_seo_sitemap_group');
            submit_button();
            ?>
        </form>

        <?php
        $sitemap_enabled = get_option('my_seo_sitemap_enabled', 1);
        if ($sitemap_enabled) {
        ?>
            <div class="seo-sitemap-links">
                <h2><?php esc_html_e('Sitemap Links', 'seo-fury'); ?></h2>
                <p>
                    <strong><?php esc_html_e('Index XML Sitemap:', 'seo-fury'); ?></strong><br>
                    <a href="<?php echo esc_url(home_url('/sitemap.xml')); ?>" target="_blank" style="font-size: 14px;">
                        <?php echo esc_html(home_url('/sitemap.xml')); ?>
                    </a>
                </p>

                <?php
                $html_slug = get_option('my_seo_html_sitemap_slug', 'html-sitemap');
                ?>
                <p>
                    <strong><?php esc_html_e('HTML Sitemap:', 'seo-fury'); ?></strong><br>
                    <a href="<?php echo esc_url(home_url('/' . $html_slug . '/')); ?>" target="_blank" style="font-size: 14px;">
                        <?php echo esc_html(home_url('/' . $html_slug . '/')); ?>
                    </a>
                </p>

                <p class="description">
                    <?php printf(esc_html__('You can also use the %s shortcode to display the HTML sitemap on any page.', 'seo-fury'), '<code>[my_seo_html_sitemap]</code>'); ?>
                </p>
            </div>
        <?php
        }
    }

    private function render_breadcrumbs_tab()
    {
        $breadcrumbs = $this->core->get_breadcrumbs();
        if ($breadcrumbs) {
            $breadcrumbs->render_settings_page();
        }
    }

    private function render_local_seo_tab()
    {
        $local_seo = $this->core->get_local_seo();
        if ($local_seo) {
            $local_seo->render_settings_page();
        }
    }


    private function render_redirects_tab()
    {
        ?>
        <div class="seo-redirects-container">
            <p><?php esc_html_e('Create 301, 302 and 307 redirects. Track total hits.', 'seo-fury'); ?></p>

<?php
$core = SEO_Fury_Core::get_instance();
$pro = $core->get_pro_features();

if (!$pro->is_pro()) {
    global $wpdb;
    $redirects_table = $wpdb->prefix . 'seo_redirects';
    $current_count = $wpdb->get_var("SELECT COUNT(*) FROM {$redirects_table}");
    $limit = $pro->get_limit('max_redirects');
    $percentage = $limit > 0 ? ($current_count / $limit) * 100 : 0;

    $bar_class = '';
    if ($percentage > 80) {
        $bar_class = 'danger';
    } elseif ($percentage > 60) {
        $bar_class = 'warning';
    }
    ?>
    <div class="seo-fury-usage-indicator">
        <div class="usage-header">
            <span class="usage-label"><?php printf('📊 ' . esc_html__('Redirects used: %1$s of %2$s', 'seo-fury'), '<strong>' . $current_count . '</strong>', '<strong>' . $limit . '</strong>'); ?></span>
        </div>
        <div class="usage-progress">
            <div class="usage-progress-bar <?php echo $bar_class; ?>" style="width: <?php echo min($percentage, 100); ?>%;"></div>
        </div>
        <?php if ($percentage > 80 && !$this->should_hide_branding()): ?>
            <div class="seo-fury-upgrade-notice" style="margin-top: 15px;">
                <strong><?php echo '⚠️ ' . esc_html__('You are nearing the FREE plan limit.', 'seo-fury'); ?></strong>
                <?php esc_html_e('PRO provides unlimited redirects.', 'seo-fury'); ?>
                <a href="https://seo-fury.com/#pricing" target="_blank" style="color: white; text-decoration: underline;"><?php esc_html_e('Upgrade to PRO', 'seo-fury'); ?> →</a>
            </div>
        <?php endif; ?>
    </div>
    <?php
}
?>

            <div class="seo-card">
                <h3><?php esc_html_e('Add a new redirect', 'seo-fury'); ?></h3>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="redirect-source"><?php esc_html_e('Source URL', 'seo-fury'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="redirect-source" class="regular-text" placeholder="<?php esc_attr_e('/old-page', 'seo-fury'); ?>">
                            <p class="description"><?php printf(esc_html__('Relative path, e.g. %s', 'seo-fury'), '<code>/old-page</code>'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="redirect-target"><?php esc_html_e('Target URL', 'seo-fury'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="redirect-target" class="regular-text" placeholder="<?php esc_attr_e('/new-page', 'seo-fury'); ?>">
                            <p class="description"><?php esc_html_e('Full URL or relative path', 'seo-fury'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="redirect-type"><?php esc_html_e('Redirect Type', 'seo-fury'); ?></label>
                        </th>
                        <td>
                            <select id="redirect-type">
                                <option value="301"><?php esc_html_e('301 (Permanent)', 'seo-fury'); ?></option>
                                <option value="302"><?php esc_html_e('302 (Temporary)', 'seo-fury'); ?></option>
                                <option value="307"><?php esc_html_e('307 (Temporary, preserves method)', 'seo-fury'); ?></option>
                            </select>
                        </td>
                    </tr>
                </table>
                <p>
                    <button type="button" id="add-redirect-btn" class="button button-primary"><?php esc_html_e('Add Redirect', 'seo-fury'); ?></button>
                </p>
            </div>

            <div class="seo-card">
                <h3><?php esc_html_e('Import / Export CSV', 'seo-fury'); ?></h3>
                <p class="description"><?php printf(esc_html__('Export all redirects or import a CSV with columns: %s.', 'seo-fury'), '<code>source_url,target_url,redirect_type,active</code>'); ?></p>
                <p>
                    <button type="button" id="seo-export-redirects" class="button button-secondary"><?php esc_html_e('Export CSV', 'seo-fury'); ?></button>
                </p>
                <p>
                    <input type="file" id="seo-import-redirects-file" accept=".csv" />
                    <button type="button" id="seo-import-redirects" class="button button-primary"><?php esc_html_e('Import CSV', 'seo-fury'); ?></button>
                </p>
                <div id="seo-import-redirects-result"></div>
            </div>

            <div class="seo-card">
                <h3><?php esc_html_e('Redirects List', 'seo-fury'); ?></h3>
                <table class="wp-list-table widefat fixed striped" id="redirects-table">
                    <thead>
                        <tr>
                            <th style="width: 50px;"><?php esc_html_e('ID', 'seo-fury'); ?></th>
                            <th><?php esc_html_e('Source', 'seo-fury'); ?></th>
                            <th><?php esc_html_e('Target', 'seo-fury'); ?></th>
                            <th style="width: 80px;"><?php esc_html_e('Type', 'seo-fury'); ?></th>
                            <th style="width: 100px;"><?php esc_html_e('Hits', 'seo-fury'); ?></th>
                            <th style="width: 150px;"><?php esc_html_e('Created', 'seo-fury'); ?></th>
                            <th style="width: 100px;"><?php esc_html_e('Status', 'seo-fury'); ?></th>
                            <th style="width: 180px;"><?php esc_html_e('Actions', 'seo-fury'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td colspan="8" style="text-align: center;"><?php esc_html_e('Loading...', 'seo-fury'); ?></td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Edit modal -->
        <div id="edit-redirect-modal" class="seo-modal">
            <div class="seo-modal-content">
                <span class="seo-modal-close">&times;</span>
                <h2><?php esc_html_e('Edit Redirect', 'seo-fury'); ?></h2>
                <input type="hidden" id="edit-redirect-id">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="edit-redirect-source"><?php esc_html_e('Source', 'seo-fury'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="edit-redirect-source" class="regular-text">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="edit-redirect-target"><?php esc_html_e('Target', 'seo-fury'); ?></label>
                        </th>
                        <td>
                            <input type="text" id="edit-redirect-target" class="regular-text">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="edit-redirect-type"><?php esc_html_e('Type', 'seo-fury'); ?></label>
                        </th>
                        <td>
                            <select id="edit-redirect-type">
                                <option value="301">301</option>
                                <option value="302">302</option>
                                <option value="307">307</option>
                            </select>
                        </td>
                    </tr>
                </table>
                <p>
                    <button type="button" id="update-redirect-btn" class="button button-primary"><?php esc_html_e('Save Changes', 'seo-fury'); ?></button>
                </p>
            </div>
        </div>
    <?php
    }

    private function render_indexnow_tab()
    {
    ?>
        <div class="seo-indexnow-container">
            <p><?php esc_html_e('Submit URLs directly to the IndexNow API.', 'seo-fury'); ?> <a href="https://www.indexnow.org" target="_blank"><?php esc_html_e('Learn more', 'seo-fury'); ?></a></p>

            <form method="post" action="options.php">
                <?php
                settings_fields('my_seo_indexnow_group');
                do_settings_sections('my_seo_indexnow_group');
                submit_button(__('Save Settings', 'seo-fury'));
                ?>
            </form>

            <hr class="seo-separator">

            <div class="seo-card">
                <h3><?php esc_html_e('Submit URLs', 'seo-fury'); ?></h3>
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="indexnow-urls"><?php esc_html_e('URLs to submit', 'seo-fury'); ?></label>
                        </th>
                        <td>
                            <textarea id="indexnow-urls" rows="10" class="large-text" placeholder="<?php esc_attr_e('Enter one URL per line...', 'seo-fury'); ?>"></textarea>
                            <p class="description"><?php esc_html_e('Enter one URL per line. You can submit up to 10,000 URLs at once.', 'seo-fury'); ?></p>
                        </td>
                    </tr>
                </table>
                <p>
                    <button type="button" id="submit-indexnow-btn" class="button button-primary button-large"><?php esc_html_e('Submit to IndexNow', 'seo-fury'); ?></button>
                </p>

                <div id="indexnow-progress" style="display: none; margin-top: 20px;">
                    <div style="background: #f0f0f1; border-radius: 4px; height: 30px; position: relative; overflow: hidden;">
                        <div id="indexnow-progress-bar" style="background: #2271b1; height: 100%; width: 0; transition: width 0.3s;"></div>
                        <div id="indexnow-progress-text" style="position: absolute; left: 50%; top: 50%; transform: translate(-50%, -50%); font-weight: 600; color: #1d2327;"></div>
                    </div>
                    <div id="indexnow-progress-status" style="margin-top: 10px; text-align: center; font-size: 14px;"></div>
                </div>

                <div id="indexnow-submit-result"></div>
            </div>

            <hr class="seo-separator">

            <div class="seo-card">
                <h3><?php esc_html_e('Submission History', 'seo-fury'); ?></h3>
                <p><?php esc_html_e('Last 100 IndexNow API requests.', 'seo-fury'); ?> <a href="https://www.indexnow.org" target="_blank"><?php esc_html_e('Learn more', 'seo-fury'); ?></a></p>
                <p>
                    <button type="button" id="load-indexnow-history-btn" class="button"><?php esc_html_e('Load History', 'seo-fury'); ?></button>
                    <button type="button" id="clear-indexnow-history-btn" class="button" style="color: #d63638;"><?php esc_html_e('Clear History', 'seo-fury'); ?></button>
                </p>

                <table class="wp-list-table widefat fixed striped" id="indexnow-history-table" style="margin-top: 20px;">
                    <thead>
                        <tr>
                            <th style="width: 180px;"><?php esc_html_e('Time', 'seo-fury'); ?></th>
                            <th><?php esc_html_e('URL', 'seo-fury'); ?></th>
                            <th style="width: 300px;"><?php esc_html_e('Response', 'seo-fury'); ?></th>
                            <th style="width: 120px;"><?php esc_html_e('Method', 'seo-fury'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td colspan="4" style="text-align: center;"><?php esc_html_e('Loading...', 'seo-fury'); ?></td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>
    <?php
    }

    public function flush_on_slug_change($old_value, $new_value)
    {
        if ($old_value !== $new_value) {
            flush_rewrite_rules();
        }
    }

    public function flush_on_sitemap_toggle($old_value, $new_value)
    {
        if ($old_value !== $new_value) {
            flush_rewrite_rules();
        }
    }

    public function flush_on_post_types_change($old_value, $new_value)
    {
        if ($old_value !== $new_value) {
            flush_rewrite_rules();
        }
    }

    public function flush_on_html_post_types_change($old_value, $new_value)
    {
        if ($old_value !== $new_value) {
            flush_rewrite_rules();
        }
    }

    private function render_meta_titles_tab()
    {
        if (!current_user_can('manage_options')) {
            return;
        }
    ?>
        <?php
        $meta_titles = $this->core->get_meta_titles();
        if ($meta_titles && method_exists($meta_titles, 'render_settings_page')) {
            $meta_titles->render_settings_page();
        } else {
            ?>
            <form method="post" action="options.php">
                <?php
                settings_fields('my_seo_meta_group');
                do_settings_sections('my_seo_meta_group');
                submit_button();
                ?>
            </form>
            <?php
        }
        ?>
    <?php
    }

    private function render_404_tab()
    {
    ?>

    <?php
    $core = SEO_Fury_Core::get_instance();
    $pro = $core->get_pro_features();

    if ($pro && !$pro->is_pro()) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'seo_404_log';
        $current_count = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
        $limit = $pro->get_limit('max_404_logs');
        $percentage = $limit > 0 ? ($current_count / $limit) * 100 : 0;

        $bar_class = '';
        if ($percentage > 80) {
            $bar_class = 'danger';
        } elseif ($percentage > 60) {
            $bar_class = 'warning';
        }
        ?>
        <div class="seo-fury-usage-indicator">
            <div class="usage-header">
                <span class="usage-label"><?php printf('📊 ' . esc_html__('404 logs used: %1$s of %2$s', 'seo-fury'), '<strong>' . $current_count . '</strong>', '<strong>' . $limit . '</strong>'); ?></span>
            </div>
            <div class="usage-progress">
                <div class="usage-progress-bar <?php echo $bar_class; ?>" style="width: <?php echo min($percentage, 100); ?>%;"></div>
            </div>
            <?php if ($percentage > 80 && !$this->should_hide_branding()): ?>
                <div class="seo-fury-upgrade-notice" style="margin-top: 15px;">
                    <strong><?php echo '⚠️ ' . esc_html__('You are nearing the FREE plan limit.', 'seo-fury'); ?></strong>
                    <?php esc_html_e('PRO removes limits for 404 logging.', 'seo-fury'); ?>
                    <a href="https://seo-fury.com/#pricing" target="_blank" style="color: white; text-decoration: underline;"><?php esc_html_e('Upgrade to PRO', 'seo-fury'); ?> &rarr;</a>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
    ?>

        <div class="seo-404-container">
            <p><?php esc_html_e('Tracking 404 errors helps find broken links and create redirects.', 'seo-fury'); ?></p>

            <form method="post" action="options.php" style="margin-bottom: 30px;">
                <?php
                settings_fields('my_seo_404_group');
                do_settings_sections('my_seo_404_group');
                submit_button();
                ?>
            </form>

            <div class="seo-card">
                <h3><?php esc_html_e('404 Statistics', 'seo-fury'); ?></h3>
                <div id="404-stats" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 15px; margin-bottom: 20px;">
                    <div style="background: #f6f7f7; padding: 15px; border-radius: 4px; text-align: center;">
                        <div style="font-size: 24px; font-weight: 600; color: #2271b1;" id="stat-total">0</div>
                    <div style="font-size: 12px; color: #646970; margin-top: 5px;"><?php esc_html_e('Total URLs', 'seo-fury'); ?></div>
                    </div>
                    <div style="background: #fcf0f1; padding: 15px; border-radius: 4px; text-align: center;">
                        <div style="font-size: 24px; font-weight: 600; color: #d63638;" id="stat-active">0</div>
                        <div style="font-size: 12px; color: #646970; margin-top: 5px;"><?php esc_html_e('Active', 'seo-fury'); ?></div>
                    </div>
                    <div style="background: #edfaef; padding: 15px; border-radius: 4px; text-align: center;">
                        <div style="font-size: 24px; font-weight: 600; color: #00a32a;" id="stat-resolved">0</div>
                        <div style="font-size: 12px; color: #646970; margin-top: 5px;"><?php esc_html_e('Resolved', 'seo-fury'); ?></div>
                    </div>
                    <div style="background: #f0f6fc; padding: 15px; border-radius: 4px; text-align: center;">
                        <div style="font-size: 24px; font-weight: 600; color: #2271b1;" id="stat-hits">0</div>
                        <div style="font-size: 12px; color: #646970; margin-top: 5px;"><?php esc_html_e('Total Hits', 'seo-fury'); ?></div>
                    </div>
                </div>

                <p>
                    <button type="button" id="refresh-404-log" class="button"><?php echo '🔄 ' . esc_html__('Refresh', 'seo-fury'); ?></button>
                    <button type="button" id="seo-export-404-logs" class="button"><?php echo '📥 ' . esc_html__('Export CSV', 'seo-fury'); ?></button>
                    <button type="button" id="clear-404-log" class="button" style="color: #d63638;"><?php echo '🗑️ ' . esc_html__('Clear All', 'seo-fury'); ?></button>
                </p>

                <table class="wp-list-table widefat fixed striped" id="log-404-table">
                    <thead>
                        <tr>
                            <th style="width: 40px;"><?php esc_html_e('ID', 'seo-fury'); ?></th>
                            <th style="width: 35%;"><?php esc_html_e('URL', 'seo-fury'); ?></th>
                            <th style="width: 70px;"><?php esc_html_e('Hits', 'seo-fury'); ?></th>
                            <th style="width: 130px;"><?php esc_html_e('First Seen', 'seo-fury'); ?></th>
                            <th style="width: 130px;"><?php esc_html_e('Last Seen', 'seo-fury'); ?></th>
                            <th style="width: 90px;"><?php esc_html_e('Status', 'seo-fury'); ?></th>
                            <th style="width: 200px;"><?php esc_html_e('Actions', 'seo-fury'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td colspan="7" style="text-align: center;"><?php esc_html_e('Loading...', 'seo-fury'); ?></td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <!-- 404 table styles -->
            <style>
                #log-404-table td code {
                    word-break: break-word !important;
                    white-space: normal !important;
                    display: block;
                    max-width: 100%;
                }

                #log-404-table .404-actions {
                    white-space: nowrap;
                }

                #log-404-table .404-actions .button {
                    margin-bottom: 3px;
                }
            </style>

        </div>

        <!-- Similar pages modal -->
        <div id="find-similar-modal" class="seo-modal">
            <div class="seo-modal-content">
                <span class="seo-modal-close">&times;</span>
                <h2><?php esc_html_e('Find a Similar Page', 'seo-fury'); ?></h2>
                <input type="hidden" id="similar-404-id">
                <input type="hidden" id="similar-source-url">

                <p><strong><?php esc_html_e('404 URL:', 'seo-fury'); ?></strong> <code id="similar-url-display"></code></p>

                <div id="similar-pages-results" style="margin: 20px 0;">
                    <p><?php esc_html_e('Loading...', 'seo-fury'); ?></p>
                </div>
            </div>
        </div>

        <!-- Create redirect modal -->
        <div id="create-redirect-modal" class="seo-modal">
            <div class="seo-modal-content">
                <span class="seo-modal-close">&times;</span>
                <h2><?php esc_html_e('Create Redirect', 'seo-fury'); ?></h2>
                <input type="hidden" id="redirect-404-id">

                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="redirect-source"><?php esc_html_e('Source (404 URL)', 'seo-fury'); ?></label></th>
                        <td>
                            <input type="text" id="redirect-source" class="regular-text" readonly>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="redirect-target"><?php esc_html_e('Target (Target URL)', 'seo-fury'); ?></label></th>
                        <td>
                            <input type="text" id="redirect-target" class="regular-text" placeholder="https://">
                            <p class="description"><?php esc_html_e('Full URL or relative path', 'seo-fury'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="redirect-type-modal"><?php esc_html_e('Redirect Type', 'seo-fury'); ?></label></th>
                        <td>
                            <select id="redirect-type-modal">
                                <option value="301"><?php esc_html_e('301 (Permanent)', 'seo-fury'); ?></option>
                                <option value="302"><?php esc_html_e('302 (Temporary)', 'seo-fury'); ?></option>
                                <option value="307"><?php esc_html_e('307 (Temporary, preserves method)', 'seo-fury'); ?></option>
                            </select>
                        </td>
                    </tr>
                </table>

                <p>
                    <button type="button" id="save-redirect-btn" class="button button-primary"><?php echo '💾 ' . esc_html__('Create Redirect', 'seo-fury'); ?></button>
                </p>
            </div>
        </div>
    <?php
    }

    private function render_license_tab()
    {
        $license = $this->core->get_license();
        $status = $license->get_status();
        $is_active = $license->is_active();
        $plan = $license->get_plan();
        $license_key = get_option('_sf_' . substr(md5('seo_fury_2025key'), 0, 8));

    ?>
        <div class="seo-license-container">

            <?php if ($is_active): ?>
                <!-- ACTIVE LICENSE -->
                <div class="seo-card license-active">
                    <h3><?php echo '✅ ' . esc_html__('License Active', 'seo-fury'); ?></h3>

                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e('License Key:', 'seo-fury'); ?></th>
                            <td>
                                <code style="font-size: 14px; background: #f0f0f1; padding: 8px 12px; border-radius: 4px; display: inline-block;">
                                    <?php echo esc_html($license_key); ?>
                                </code>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Plan:', 'seo-fury'); ?></th>
                            <td>
                                <strong style="font-size: 16px; color: #2271b1;">
                                    <?php
                                    $plans = array(
                                        'free' => __('Free', 'seo-fury'),
                                        'pro_starter' => __('Pro Starter', 'seo-fury'),
                                        'pro_business' => __('Pro Business', 'seo-fury'),
                                        'pro_agency' => __('Pro Agency', 'seo-fury'),
                                        'lifetime' => __('Lifetime', 'seo-fury')
                                    );
                                    echo esc_html($plans[$plan] ?? __('Unknown', 'seo-fury'));
                                    ?>
                                </strong>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Status:', 'seo-fury'); ?></th>
                            <td>
                                <span style="background: #00a32a; color: white; padding: 4px 12px; border-radius: 12px; font-weight: 600;">
                                    <?php echo esc_html(strtoupper($status['status'] ?? 'active')); ?>
                                </span>
                            </td>
                        </tr>
                        <?php if (isset($status['expiresAt']) && $status['expiresAt']): ?>
                            <tr>
                                <th><?php esc_html_e('Expires:', 'seo-fury'); ?></th>
                                <td><?php echo esc_html(date('Y-m-d', strtotime($status['expiresAt']))); ?></td>
                            </tr>
                        <?php endif; ?>
                        <tr>
                            <th><?php esc_html_e('Domain:', 'seo-fury'); ?></th>
                            <td><code><?php echo esc_html(parse_url(home_url(), PHP_URL_HOST)); ?></code></td>
                        </tr>
                    </table>

                    <h4><?php esc_html_e('Available Features:', 'seo-fury'); ?></h4>
                    <ul style="list-style: none; padding: 0;">
                        <li style="margin: 8px 0;">
                            <?php if ($license->is_feature_enabled('advanced_seo')): ?>
                                <span style="color: #00a32a;">✅</span> <strong><?php esc_html_e('Advanced SEO', 'seo-fury'); ?></strong> - <?php esc_html_e('Advanced SEO features', 'seo-fury'); ?>
                            <?php else: ?>
                                <span style="color: #dba617;">⭕</span> <strong><?php esc_html_e('Advanced SEO', 'seo-fury'); ?></strong> - <?php esc_html_e('Not available on your plan', 'seo-fury'); ?>
                            <?php endif; ?>
                        </li>
                        <li style="margin: 8px 0;">
                            <?php if ($license->is_feature_enabled('unlimited_redirects')): ?>
                                <span style="color: #00a32a;">✅</span> <strong><?php esc_html_e('Unlimited redirects', 'seo-fury'); ?></strong>
                            <?php else: ?>
                                <span style="color: #dba617;">⭕</span> <strong><?php esc_html_e('Redirects', 'seo-fury'); ?></strong> - <?php esc_html_e('Limited to 50', 'seo-fury'); ?>
                            <?php endif; ?>
                        </li>
                        <li style="margin: 8px 0;">
                            <?php if ($license->is_feature_enabled('unlimited_404')): ?>
                                <span style="color: #00a32a;">✅</span> <strong><?php esc_html_e('Unlimited 404 logs', 'seo-fury'); ?></strong>
                            <?php else: ?>
                                <span style="color: #dba617;">⭕</span> <strong><?php esc_html_e('404 Logs', 'seo-fury'); ?></strong> - <?php esc_html_e('Limited to 100', 'seo-fury'); ?>
                            <?php endif; ?>
                        </li>
                        <li style="margin: 8px 0;">
                            <?php if ($license->is_feature_enabled('internal_linking')): ?>
                                <span style="color: #00a32a;">✅</span> <strong><?php esc_html_e('Internal Linking', 'seo-fury'); ?></strong> - <?php esc_html_e('Smart link suggestions', 'seo-fury'); ?>
                            <?php else: ?>
                                <span style="color: #dba617;">⭕</span> <strong><?php esc_html_e('Internal Linking', 'seo-fury'); ?></strong> - <?php esc_html_e('Not available on your plan', 'seo-fury'); ?>
                            <?php endif; ?>
                        </li>
                        <li style="margin: 8px 0;">
                            <?php if ($license->is_feature_enabled('white_label')): ?>
                                <span style="color: #00a32a;">✅</span> <strong><?php esc_html_e('White Label', 'seo-fury'); ?></strong> - <?php esc_html_e('Branding', 'seo-fury'); ?>
                            <?php else: ?>
                                <span style="color: #dba617;">⭕</span> <strong><?php esc_html_e('White Label', 'seo-fury'); ?></strong> - <?php esc_html_e('Not available', 'seo-fury'); ?>
                            <?php endif; ?>
                        </li>
                        <li style="margin: 8px 0;">
                            <?php if ($license->is_feature_enabled('priority_support')): ?>
                                <span style="color: #00a32a;">✅</span> <strong><?php esc_html_e('Priority Support', 'seo-fury'); ?></strong> - <?php esc_html_e('Priority support', 'seo-fury'); ?>
                            <?php else: ?>
                                <span style="color: #dba617;">⭕</span> <strong><?php esc_html_e('Support', 'seo-fury'); ?></strong> - <?php esc_html_e('Standard support', 'seo-fury'); ?>
                            <?php endif; ?>
                        </li>
                    </ul>

                    <?php if ($license->is_feature_enabled('priority_support')): ?>
                    <div style="background: linear-gradient(135deg, #00a32a 0%, #008a00 100%); color: white; padding: 15px; border-radius: 6px; margin-top: 20px;">
                        <div style="display: flex; align-items: center; gap: 12px;">
                            <div style="font-size: 32px;">💎</div>
                            <div>
                                <strong style="font-size: 14px;"><?php esc_html_e('Priority support is active', 'seo-fury'); ?></strong>
                                <p style="margin: 5px 0 0 0; font-size: 13px; opacity: 0.9;">
                                    &bull; <?php esc_html_e('Reply within 24 hours', 'seo-fury'); ?><br>
                                    &bull; <?php esc_html_e('Direct line to developers', 'seo-fury'); ?><br>
                                    &bull; <?php esc_html_e('Help with setup and optimization', 'seo-fury'); ?>
                                </p>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>

                    <p style="margin-top: 20px;">
                        <button type="button" id="deactivate-license-btn" class="button" style="color: #d63638;">
                            <?php esc_html_e('Deactivate License', 'seo-fury'); ?>
                        </button>
                        <button type="button" id="check-license-btn" class="button">
                            <?php esc_html_e('Check Status', 'seo-fury'); ?>
                        </button>
                    </p>

                    <div id="license-check-result" style="margin-top: 15px;"></div>
                </div>

            <?php else: ?>
                <div class="license-inactive">
    <h3><?php echo '🔓 ' . esc_html__('License Not Activated', 'seo-fury'); ?></h3>

    <p><?php esc_html_e('Enter a license key to activate plugin features.', 'seo-fury'); ?></p>
    <p class="description">
        <strong><?php esc_html_e('Supported formats:', 'seo-fury'); ?></strong><br>
        • PRO plans: <code>SEOFURY-XXXXX-XXXXXX-XXXXX-XXXXX</code><br>
        • FREE plan: <code>FREE-XXXXXXXXXXXXXXXX</code>
    </p>

    <table class="form-table">
        <tr>
            <th scope="row">
                <label for="license-key-input"><?php esc_html_e('License Key', 'seo-fury'); ?></label>
            </th>
            <td>
                <input type="text"
                       id="license-key-input"
                       class="regular-text"
                       placeholder="SEOFURY-XXXXX-XXXXXX-XXXXX-XXXXX or FREE-XXXXXXXXXXXXXXXX">

                <p class="description">
                    <?php printf(esc_html__('PRO format: %s', 'seo-fury'), 'SEOFURY-XXXXX-XXXXXX-XXXXX-XXXXX'); ?><br>
                    <?php printf(esc_html__('FREE format: %s', 'seo-fury'), 'FREE-XXXXXXXXXXXXXXXX'); ?>
                </p>
            </td>
        </tr>
    </table>

                    <p>
                        <button type="button" id="activate-license-btn" class="button button-primary">
                            <?php esc_html_e('Activate License', 'seo-fury'); ?>
                        </button>
                    </p>

                    <div id="license-activation-result" style="margin-top: 15px;"></div>

                    <hr style="margin: 30px 0;">

                    <h4><?php echo '📦 ' . esc_html__('Available plans:', 'seo-fury'); ?></h4>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin-top: 20px;">

                        <div style="border: 2px solid #ddd; padding: 20px; border-radius: 8px;">
                            <h4 style="margin-top: 0;"><?php esc_html_e('Free', 'seo-fury'); ?></h4>
                            <p style="font-size: 24px; font-weight: bold; color: #2271b1;">$0</p>
                            <ul style="list-style: none; padding: 0; font-size: 13px;">
                                <li><?php printf('✅ ' . esc_html__('%s domain', 'seo-fury'), '1'); ?></li>
                                <li><?php printf('✅ ' . esc_html__('%s redirects', 'seo-fury'), '50'); ?></li>
                                <li><?php printf('✅ ' . esc_html__('%s 404 logs', 'seo-fury'), '100'); ?></li>
                                <li><?php echo '❌ ' . esc_html__('Advanced SEO', 'seo-fury'); ?></li>
                            </ul>
                        </div>

                        <div style="border: 2px solid #2271b1; padding: 20px; border-radius: 8px; background: #f0f6fc;">
                            <h4 style="margin-top: 0; color: #2271b1;"><?php esc_html_e('Pro Starter', 'seo-fury'); ?></h4>
                            <p style="font-size: 24px; font-weight: bold; color: #2271b1;">$3<span style="font-size: 14px;">/<?php esc_html_e('month', 'seo-fury'); ?></span></p>
                            <ul style="list-style: none; padding: 0; font-size: 13px;">
                                <li><?php printf('✅ ' . esc_html__('%s domains', 'seo-fury'), '3'); ?></li>
                                <li><?php echo '✅ ' . esc_html__('Unlimited redirects', 'seo-fury'); ?></li>
                                <li><?php echo '✅ ' . esc_html__('Unlimited 404 logs', 'seo-fury'); ?></li>
                                <li><?php echo '✅ ' . esc_html__('Advanced SEO', 'seo-fury'); ?></li>
                            </ul>
                        </div>

                        <div style="border: 2px solid #00a32a; padding: 20px; border-radius: 8px; background: #edfaef;">
                            <h4 style="margin-top: 0; color: #00a32a;"><?php esc_html_e('Pro Business', 'seo-fury'); ?></h4>
                            <p style="font-size: 24px; font-weight: bold; color: #00a32a;">$12<span style="font-size: 14px;">/<?php esc_html_e('month', 'seo-fury'); ?></span></p>
                            <ul style="list-style: none; padding: 0; font-size: 13px;">
                                <li><?php printf('✅ ' . esc_html__('%s domains', 'seo-fury'), '10'); ?></li>
                                <li><?php echo '✅ ' . esc_html__('Everything in Starter +', 'seo-fury'); ?></li>
                                <li><?php echo '✅ ' . esc_html__('White Label', 'seo-fury'); ?></li>
                                <li><?php echo '✅ ' . esc_html__('Priority Support', 'seo-fury'); ?></li>
                            </ul>
                        </div>

                        <div style="border: 2px solid #d63638; padding: 20px; border-radius: 8px; background: #fcf0f1;">
                            <h4 style="margin-top: 0; color: #d63638;"><?php esc_html_e('Pro Agency', 'seo-fury'); ?></h4>
                            <p style="font-size: 24px; font-weight: bold; color: #d63638;">$27<span style="font-size: 14px;">/<?php esc_html_e('month', 'seo-fury'); ?></span></p>
                            <ul style="list-style: none; padding: 0; font-size: 13px;">
                                <li><?php echo '✅ ' . esc_html__('Unlimited domains', 'seo-fury'); ?></li>
                                <li><?php echo '✅ ' . esc_html__('Everything in Business +', 'seo-fury'); ?></li>
                                <li><?php echo '✅ ' . esc_html__('All pro features', 'seo-fury'); ?></li>
                                <li><?php echo '✅ ' . esc_html__('Priority Support', 'seo-fury'); ?></li>
                            </ul>
                        </div>

                        <div style="border: 2px solid #9333ea; padding: 20px; border-radius: 8px; background: #faf5ff;">
                            <h4 style="margin-top: 0; color: #9333ea;"><?php esc_html_e('Lifetime', 'seo-fury'); ?></h4>
                            <p style="font-size: 24px; font-weight: bold; color: #9333ea;">$300<span style="font-size: 14px;"> <?php esc_html_e('one-time', 'seo-fury'); ?></span></p>
                            <ul style="list-style: none; padding: 0; font-size: 13px;">
                                <li><?php echo '✅ ' . esc_html__('Unlimited domains', 'seo-fury'); ?></li>
                                <li><?php echo '✅ ' . esc_html__('All features forever', 'seo-fury'); ?></li>
                                <li><?php echo '✅ ' . esc_html__('Lifetime updates', 'seo-fury'); ?></li>
                                <li><?php echo '✅ ' . esc_html__('Priority Support', 'seo-fury'); ?></li>
                            </ul>
                        </div>

                    </div>

                    <?php if (!$this->should_hide_branding()): ?>
                        <p style="margin-top: 30px; text-align: center;">
                            <a href="https://seo-fury.com/#pricing" target="_blank" class="button button-primary button-large">
                                <?php esc_html_e('Buy a license', 'seo-fury'); ?>
                            </a>
                        </p>
                    <?php endif; ?>
                </div>
            <?php endif; ?>

        </div>
<?php
    }

    private function render_woocommerce_tab()
    {
        $woocommerce = $this->core->get_woocommerce();
        if ($woocommerce) {
            $woocommerce->render_settings_page();
        }
    }

    private function render_ssl_tab()
    {
        $general = $this->core->get_general();
        ?>
        <div class="seo-card" style="margin-bottom: 20px;">
            <div style="display: flex; align-items: center; gap: 15px; margin-bottom: 20px;">
                <div style="font-size: 40px;">🔒</div>
                <div>
                    <h3 style="margin: 0 0 5px;"><?php esc_html_e('SSL / HTTPS Security', 'seo-fury'); ?></h3>
                    <p class="description" style="margin: 0;"><?php esc_html_e('Protect your site with HTTPS enforcement, mixed content fixes, and security headers.', 'seo-fury'); ?></p>
                </div>
            </div>

            <?php
            $has_ssl = is_ssl();
            $force_https = get_option('my_seo_force_https', 0);
            $fix_mixed = get_option('my_seo_fix_mixed_content', 0);
            $hsts = get_option('my_seo_enable_hsts', 0);
            ?>

            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 25px;">
                <div style="background: <?php echo $has_ssl ? '#ECFDF5' : '#FEF2F2'; ?>; border-radius: 8px; padding: 15px; text-align: center;">
                    <div style="font-size: 28px;"><?php echo $has_ssl ? '✅' : '❌'; ?></div>
                    <div style="font-weight: 600; margin-top: 5px;"><?php esc_html_e('SSL Certificate', 'seo-fury'); ?></div>
                    <div style="font-size: 12px; color: #64748B;"><?php echo $has_ssl ? esc_html__('Active', 'seo-fury') : esc_html__('Not detected', 'seo-fury'); ?></div>
                </div>
                <div style="background: <?php echo $force_https ? '#ECFDF5' : '#FEF3C7'; ?>; border-radius: 8px; padding: 15px; text-align: center;">
                    <div style="font-size: 28px;"><?php echo $force_https ? '✅' : '⚠️'; ?></div>
                    <div style="font-weight: 600; margin-top: 5px;"><?php esc_html_e('HTTPS Redirect', 'seo-fury'); ?></div>
                    <div style="font-size: 12px; color: #64748B;"><?php echo $force_https ? esc_html__('Enabled', 'seo-fury') : esc_html__('Disabled', 'seo-fury'); ?></div>
                </div>
                <div style="background: <?php echo $fix_mixed ? '#ECFDF5' : '#F1F5F9'; ?>; border-radius: 8px; padding: 15px; text-align: center;">
                    <div style="font-size: 28px;"><?php echo $fix_mixed ? '✅' : '➖'; ?></div>
                    <div style="font-weight: 600; margin-top: 5px;"><?php esc_html_e('Mixed Content', 'seo-fury'); ?></div>
                    <div style="font-size: 12px; color: #64748B;"><?php echo $fix_mixed ? esc_html__('Auto-fix on', 'seo-fury') : esc_html__('Not active', 'seo-fury'); ?></div>
                </div>
                <div style="background: <?php echo $hsts ? '#ECFDF5' : '#F1F5F9'; ?>; border-radius: 8px; padding: 15px; text-align: center;">
                    <div style="font-size: 28px;"><?php echo $hsts ? '✅' : '➖'; ?></div>
                    <div style="font-weight: 600; margin-top: 5px;"><?php esc_html_e('HSTS Header', 'seo-fury'); ?></div>
                    <div style="font-size: 12px; color: #64748B;"><?php echo $hsts ? esc_html__('Active', 'seo-fury') : esc_html__('Not active', 'seo-fury'); ?></div>
                </div>
            </div>
        </div>

        <form method="post" action="options.php">
            <?php
            settings_fields('my_seo_ssl_settings');
            do_settings_sections('my_seo_ssl_settings');
            submit_button(__('Save Security Settings', 'seo-fury'));
            ?>
        </form>
        <?php
    }

    private function render_changelog_tab()
    {
        $changelog_data = $this->get_changelog_data();
    ?>
        <div class="seo-changelog-container">
            <div class="seo-card">
                <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 20px;">
                    <div>
                        <h2 style="margin: 0;"><?php echo '📝 ' . esc_html__('Changelog', 'seo-fury'); ?></h2>
                        <p class="description" style="margin: 5px 0 0 0;"><?php esc_html_e('All updates and improvements for the plugin', 'seo-fury'); ?></p>
                    </div>
                    <div style="text-align: right;">
                        <div style="background: #f0f6fc; padding: 10px 15px; border-radius: 6px; border: 1px solid #2271b1;">
                            <div style="font-size: 12px; color: #646970; margin-bottom: 3px;"><?php esc_html_e('Current version', 'seo-fury'); ?></div>
                            <div style="font-size: 20px; font-weight: 600; color: #2271b1;"><?php echo esc_html(SEO_FURY_VERSION); ?></div>
                        </div>
                    </div>
                </div>

                <?php if (empty($changelog_data)): ?>
                    <div style="text-align: center; padding: 40px 20px; color: #646970;">
                        <span class="dashicons dashicons-warning" style="font-size: 40px; width: 40px; height: 40px; margin-bottom: 10px;"></span>
                        <p><?php esc_html_e('Unable to load changelog. Please check your internet connection.', 'seo-fury'); ?></p>
                        <button type="button" class="button" onclick="location.reload();"><?php esc_html_e('Retry', 'seo-fury'); ?></button>
                    </div>
                <?php else: ?>
                    <?php foreach ($changelog_data as $version => $data): ?>
                        <div class="changelog-version" style="margin-bottom: 30px; padding: 20px; background: #f6f7f7; border-left: 4px solid #2271b1; border-radius: 4px;">
                            <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 15px;">
                                <h3 style="margin: 0; font-size: 18px; color: #2271b1;">
                                    <span class="dashicons dashicons-tag" style="font-size: 18px;"></span>
                                    <?php printf(esc_html__('Version %s', 'seo-fury'), esc_html($version)); ?>
                                </h3>
                                <span style="background: #fff; padding: 5px 12px; border-radius: 4px; font-size: 13px; color: #646970;">
                                    <?php echo esc_html(date('d.m.Y', strtotime($data['date']))); ?>
                                </span>
                            </div>

                            <?php if (isset($data['changes']['added']) && !empty($data['changes']['added'])): ?>
                                <div style="margin-bottom: 15px;">
                                    <h4 style="margin: 0 0 10px 0; font-size: 14px; color: #00a32a;">
                                        <span class="dashicons dashicons-plus" style="font-size: 16px;"></span>
                                        Added
                                    </h4>
                                    <ul style="margin: 0; padding-left: 25px;">
                                        <?php foreach ($data['changes']['added'] as $item): ?>
                                            <li style="margin-bottom: 5px;"><?php echo esc_html($item); ?></li>
                                        <?php endforeach; ?>
                                    </ul>
                                </div>
                            <?php endif; ?>

                            <?php if (isset($data['changes']['improved']) && !empty($data['changes']['improved'])): ?>
                                <div style="margin-bottom: 15px;">
                                    <h4 style="margin: 0 0 10px 0; font-size: 14px; color: #2271b1;">
                                        <span class="dashicons dashicons-arrow-up-alt" style="font-size: 16px;"></span>
                                        Improved
                                    </h4>
                                    <ul style="margin: 0; padding-left: 25px;">
                                        <?php foreach ($data['changes']['improved'] as $item): ?>
                                            <li style="margin-bottom: 5px;"><?php echo esc_html($item); ?></li>
                                        <?php endforeach; ?>
                                    </ul>
                                </div>
                            <?php endif; ?>

                            <?php if (isset($data['changes']['fixed']) && !empty($data['changes']['fixed'])): ?>
                                <div>
                                    <h4 style="margin: 0 0 10px 0; font-size: 14px; color: #d63638;">
                                        <span class="dashicons dashicons-dismiss" style="font-size: 16px;"></span>
                                        Fixed
                                    </h4>
                                    <ul style="margin: 0; padding-left: 25px;">
                                        <?php foreach ($data['changes']['fixed'] as $item): ?>
                                            <li style="margin-bottom: 5px;"><?php echo esc_html($item); ?></li>
                                        <?php endforeach; ?>
                                    </ul>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
<?php
    }

    private function get_changelog_data() {
        $cache_key = 'seo_fury_changelog_data';
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            return $cached;
        }

        $response = wp_remote_get('https://seo-fury.com/api/changelog.json', array(
            'timeout' => 10,
            'sslverify' => true,
        ));

        if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            if (is_array($data) && !empty($data)) {
                set_transient($cache_key, $data, 12 * HOUR_IN_SECONDS);
                update_option('seo_fury_changelog_fallback', $data, false);
                return $data;
            }
        }

        $fallback = get_option('seo_fury_changelog_fallback', array());
        if (!empty($fallback)) {
            return $fallback;
        }

        return $this->get_bundled_changelog();
    }

    private function get_bundled_changelog() {
        return array(
            '0.1.5' => array(
                'date' => '2026-02-08',
                'changes' => array(
                    'added' => array(
                        'Internal Linking module (PRO) — smart link suggestions in post editor',
                        'SSL / HTTPS Security module — separate admin tab with status dashboard',
                        'Force HTTPS redirect, mixed content fix, HSTS headers, WWW redirect',
                        '404 Monitor CSV export',
                        'Sitemap Additional URLs per-URL priority and changefreq (PRO)',
                    ),
                    'improved' => array(
                        'Internal linking: case-insensitive matching with Cyrillic support',
                        'Changelog now loads remotely from website API',
                        'Cleaned all code comments, translated Russian strings to English',
                    ),
                    'fixed' => array(
                        'Fixed parse_shortcodes() private method visibility error',
                        'Fixed Gutenberg sidebar metabox not displaying',
                        'Fixed internal linking not matching single-word Cyrillic titles',
                    ),
                ),
            ),
            '0.1.4' => array(
                'date' => '2026-01-17',
                'changes' => array(
                    'added' => array(
                        'Rebuilt Meta Titles module with Rank Math-style content type navigation',
                        'Added templates for home, blog, author, search, date, 404, taxonomies',
                    ),
                    'improved' => array(
                        'Translated remaining admin UI strings to English',
                        'Refined sitemap UI and XSL output',
                    ),
                    'fixed' => array(
                        'White label now hides pricing links when branding is disabled',
                    ),
                ),
            ),
            '0.1.3' => array(
                'date' => '2026-01-17',
                'changes' => array(
                    'added' => array(
                        'Rank Math-style Gutenberg sidebar with compact tabs and modals',
                        'Schema Generator modal with multi-item list',
                        'Social settings modal with nested tabs',
                    ),
                    'improved' => array(
                        'Google preview and snippet editor layout',
                        'Schema output: multiple items and custom JSON-LD',
                    ),
                    'fixed' => array(
                        'Schema settings persisting in Gutenberg sidebar',
                    ),
                ),
            ),
            '0.1.2' => array(
                'date' => '2026-01-15',
                'changes' => array(
                    'added' => array(
                        'Schema Markup module with 13+ structured data types',
                        'JSON-LD generation system for rich snippets',
                    ),
                ),
            ),
            '0.1.1' => array(
                'date' => '2026-01-10',
                'changes' => array(
                    'added' => array(
                        'Changelog page',
                        'PRO feature: block web archive bots',
                    ),
                ),
            ),
            '0.1.0' => array(
                'date' => '2026-01-01',
                'changes' => array(
                    'added' => array(
                        'First release: robots.txt, sitemap, redirects, IndexNow, 404 monitor, meta tags, licensing',
                    ),
                ),
            ),
        );
    }
}
