<?php

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Fury_Core
{

    private $db_version = '1.0.0';

    private $redirects_table;
    private $indexnow_table;
    private $log404_table;

    private $robots;
    private $sitemap;
    private $redirects;
    private $indexnow;
    private $general;
    private $admin;
    private $meta_titles;
    private $breadcrumbs;
    private $local_seo;
    private $log404_monitor;
    private $license;
    private $post_meta;
    private $fingerprint;
    private $integrity;
    private $pro_features;
    private $woocommerce;

    private static $instance = null;

public static function get_instance($redirects_table = null, $indexnow_table = null)
{
    if (self::$instance === null) {
        self::$instance = new self($redirects_table, $indexnow_table);
    }
    return self::$instance;
}

private function __construct($redirects_table = null, $indexnow_table = null)
{
    if ($redirects_table) {
        $this->redirects_table = $redirects_table;
    }
    if ($indexnow_table) {
        $this->indexnow_table = $indexnow_table;
    }

    if (!$this->redirects_table || !$this->indexnow_table || !$this->log404_table) {
        $this->setup_database_tables();
    }

    $this->load_dependencies();
    $this->init_modules();

    $this->remove_wp_default_tags();

$stored_version = get_option('seo_fury_installed_version', '');
if ($stored_version !== SEO_FURY_VERSION) {
    $this->integrity->clear_manifest();
    $this->integrity->generate_manifest();
    update_option('seo_fury_installed_version', SEO_FURY_VERSION);
}

if (!$this->integrity->verify_integrity()) {
    update_option('seo_fury_integrity_failed', 1, false);
    add_action('admin_notices', function() {
        $brand = function_exists('seo_fury_get_brand_name') ? seo_fury_get_brand_name() : 'SEO Fury';
        echo '<div class="notice notice-error"><p><strong>' . esc_html($brand) . ':</strong> ' . esc_html__('Plugin files have been modified. PRO features disabled for security.', 'seo-fury') . '</p></div>';
    });
} else {
    delete_option('seo_fury_integrity_failed');
}
}

private function remove_wp_default_tags() {
    remove_action('wp_head', 'rel_canonical');

    remove_action('wp_head', 'wp_generator');
    remove_action('wp_head', 'wlwmanifest_link');
    remove_action('wp_head', 'rsd_link');
    remove_action('wp_head', 'wp_shortlink_wp_head');

    remove_action('wp_head', 'rest_output_link_wp_head');
    remove_action('wp_head', 'wp_oembed_add_discovery_links');

    error_log('=== CORE: WordPress default tags removed ===');
}

private function setup_database_tables()
{
    global $wpdb;

    if (!$this->redirects_table) {
        $this->redirects_table = $wpdb->prefix . 'seo_redirects';
    }
    if (!$this->indexnow_table) {
        $this->indexnow_table = $wpdb->prefix . 'seo_indexnow_log';
    }

    if (!$this->log404_table) {
        $this->log404_table = $wpdb->prefix . 'seo_404_log';
    }
}

private function load_dependencies()
{
    require_once SEO_FURY_PATH . 'includes/class-robots.php';
    require_once SEO_FURY_PATH . 'includes/class-sitemap.php';
    require_once SEO_FURY_PATH . 'includes/class-redirects.php';
    require_once SEO_FURY_PATH . 'includes/class-indexnow.php';
    require_once SEO_FURY_PATH . 'includes/class-general.php';
    require_once SEO_FURY_PATH . 'includes/class-meta-titles.php';
    require_once SEO_FURY_PATH . 'includes/class-breadcrumbs.php';
    require_once SEO_FURY_PATH . 'includes/class-local-seo.php';
    require_once SEO_FURY_PATH . 'includes/class-404-monitor.php';
    require_once SEO_FURY_PATH . 'includes/class-admin.php';
    require_once SEO_FURY_PATH . 'includes/class-license.php';
    require_once SEO_FURY_PATH . 'includes/class-post-meta.php';
    require_once SEO_FURY_PATH . 'includes/class-fingerprint.php';
    require_once SEO_FURY_PATH . 'includes/class-integrity.php';
    require_once SEO_FURY_PATH . 'includes/class-migration.php';
    require_once SEO_FURY_PATH . 'includes/class-pro-features.php';
    require_once SEO_FURY_PATH . 'includes/class-post-meta-internal-links.php';

    if (class_exists('WooCommerce')) {
        require_once SEO_FURY_PATH . 'includes/class-woocommerce.php';
    }
}

private function init_modules()
{
    $this->robots = new My_SEO_Plugin_Robots();
    $this->sitemap = new My_SEO_Plugin_Sitemap();
    $this->redirects = new My_SEO_Plugin_Redirects($this->redirects_table);
    $this->indexnow = new My_SEO_Plugin_Indexnow($this->indexnow_table);
    $this->general = new My_SEO_Plugin_General();
    $this->meta_titles = new My_SEO_Plugin_Meta_Titles();
    $this->breadcrumbs = new My_SEO_Plugin_Breadcrumbs();
    $this->local_seo = new My_SEO_Plugin_Local_SEO();
    $this->license = new My_SEO_Plugin_License();
    $this->fingerprint = new My_SEO_Plugin_Fingerprint();
    $this->integrity = new SEO_Fury_Integrity();
    $this->pro_features = new SEO_Fury_Pro_Features($this->license);

    $this->log404_monitor = new My_SEO_Plugin_404_Monitor($this->log404_table, $this->redirects_table);
    $this->post_meta = new My_SEO_Plugin_Post_Meta();
    if (class_exists('WooCommerce')) {
        $this->woocommerce = new SEO_Fury_WooCommerce();
    }

    $this->admin = new My_SEO_Plugin_Admin($this);
}

public function get_integrity() {
    return $this->integrity;
}

public function get_404_monitor()
{
    return $this->log404_monitor;
}

    public function get_general()
    {
        return $this->general;
    }

    public function get_robots()
    {
        return $this->robots;
    }

    public function get_sitemap()
    {
        return $this->sitemap;
    }

    public function get_redirects()
    {
        return $this->redirects;
    }

    public function get_indexnow()
    {
        return $this->indexnow;
    }

public function get_meta_titles()
{
    return $this->meta_titles;
}

public function get_breadcrumbs()
{
    return $this->breadcrumbs;
}

public function get_local_seo()
{
    return $this->local_seo;
}

public function get_fingerprint() {
    return $this->fingerprint;
}

public function get_license()
{
    return $this->license;
}

public function get_post_meta() {
    return $this->post_meta;
}

public function get_pro_features() {
    return $this->pro_features;
}

public function get_woocommerce() {
    return $this->woocommerce;
}

    public static function activate()
    {
        $instance = new self();
        $migration = new SEO_Fury_Migration();
        $migration->run_migration();
        $instance->create_database_tables();
        $instance->set_default_options();

        $instance->sitemap->register_sitemap_rewrite();
        $instance->sitemap->register_xsl_rewrite();

        flush_rewrite_rules();
    }

    public static function deactivate()
    {
        flush_rewrite_rules();
    }

private function create_database_tables()
{
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();

    $sql_redirects = "CREATE TABLE IF NOT EXISTS {$this->redirects_table} (
        id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        source_url varchar(500) NOT NULL,
        target_url varchar(500) NOT NULL,
        redirect_type int(3) NOT NULL DEFAULT 301,
        hits bigint(20) UNSIGNED NOT NULL DEFAULT 0,
        active tinyint(1) NOT NULL DEFAULT 1,
        created datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY source_url (source_url(191)),
        KEY active (active)
    ) $charset_collate;";

    $sql_indexnow = "CREATE TABLE IF NOT EXISTS {$this->indexnow_table} (
        id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        url varchar(500) NOT NULL,
        response_code int(3) DEFAULT NULL,
        response_message text DEFAULT NULL,
        submitted_at datetime DEFAULT CURRENT_TIMESTAMP,
        method varchar(50) DEFAULT 'manual',
        PRIMARY KEY (id),
        KEY url (url(191)),
        KEY submitted_at (submitted_at)
    ) $charset_collate;";

    $sql_404 = "CREATE TABLE IF NOT EXISTS {$this->log404_table} (
    id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
    url varchar(500) NOT NULL,
    referer varchar(500) DEFAULT NULL,
    user_agent varchar(255) DEFAULT NULL,
    ip_address varchar(45) DEFAULT NULL,
    hits bigint(20) UNSIGNED NOT NULL DEFAULT 1,
    first_seen datetime DEFAULT CURRENT_TIMESTAMP,
    last_seen datetime DEFAULT CURRENT_TIMESTAMP,
    resolved tinyint(1) NOT NULL DEFAULT 0,
    PRIMARY KEY (id),
    KEY url (url(191)),
    KEY last_seen (last_seen)
) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql_redirects);
    dbDelta($sql_indexnow);
    dbDelta($sql_404);

    update_option('seo_fury_db_version', $this->db_version);
}

    private function set_default_options()
    {
        if (get_option('my_seo_robots_enabled') === false) {
            update_option('my_seo_robots_enabled', 1);
        }

        if (get_option('my_seo_robots_use_dynamic_vars') === false) {
            update_option('my_seo_robots_use_dynamic_vars', 1);
        }

        if (get_option('my_seo_sitemap_enabled') === false) {
            update_option('my_seo_sitemap_enabled', 0);
        }

        if (get_option('my_seo_sitemap_post_types') === false) {
            update_option('my_seo_sitemap_post_types', array('post', 'page'));
        }

        if (get_option('my_seo_sitemap_include_images') === false) {
            update_option('my_seo_sitemap_include_images', 0);
        }

        if (get_option('my_seo_indexnow_auto_submit') === false) {
            update_option('my_seo_indexnow_auto_submit', 0);
        }

        if (get_option('my_seo_indexnow_auto_delete') === false) {
            update_option('my_seo_indexnow_auto_delete', 0);
        }

        if (get_option('my_seo_indexnow_search_engine') === false) {
            update_option('my_seo_indexnow_search_engine', 'api.indexnow.org');
        }

        if (get_option('my_seo_robots_meta_index') === false) {
            update_option('my_seo_robots_meta_index', 1);
        }

        if (get_option('my_seo_robots_meta_max_snippet') === false) {
            update_option('my_seo_robots_meta_max_snippet', -1);
        }
        if (get_option('my_seo_robots_meta_max_video_preview') === false) {
            update_option('my_seo_robots_meta_max_video_preview', -1);
        }
        if (get_option('my_seo_robots_meta_max_image_preview') === false) {
            update_option('my_seo_robots_meta_max_image_preview', 'standard');
        }
        if (get_option('my_seo_title_separator') === false) {
            update_option('my_seo_title_separator', '-');
        }
        if (get_option('my_seo_twitter_card_type') === false) {
            update_option('my_seo_twitter_card_type', 'summary_large_image');
        }

        if (get_option('my_seo_404_monitoring_enabled') === false) {
            update_option('my_seo_404_monitoring_enabled', 1);
        }
        if (get_option('my_seo_404_ignore_bots') === false) {
            update_option('my_seo_404_ignore_bots', 1);
        }
        if (get_option('my_seo_404_auto_cleanup_days') === false) {
            update_option('my_seo_404_auto_cleanup_days', 30);
        }
    }
}
