<?php

if (!defined('ABSPATH')) {
    exit;
}

class My_SEO_Plugin_General {

    public function __construct() {
        $this->init_hooks();
    }

private function init_hooks() {
    add_filter('sanitize_title', array($this, 'transliterate_slug'), 10, 3);
    add_action('wp_ajax_seo_bulk_transliterate', array($this, 'ajax_bulk_transliterate'));
    add_action('wp_ajax_seo_get_url_stats', array($this, 'ajax_get_url_stats'));

    add_action('template_redirect', array($this, 'handle_security_redirects'), 1);
    add_action('send_headers', array($this, 'send_security_headers'));

    if (get_option('my_seo_fix_mixed_content', 0) && !is_admin()) {
        add_action('template_redirect', array($this, 'start_mixed_content_buffer'), 2);
    }
}

public function register_settings() {
    register_setting('my_seo_general_settings', 'my_seo_transliterate_urls', array(
        'type' => 'boolean',
        'default' => 0,
        'sanitize_callback' => array($this, 'sanitize_checkbox')
    ));

    register_setting('my_seo_ssl_settings', 'my_seo_force_https', array('type' => 'boolean', 'default' => 0, 'sanitize_callback' => array($this, 'sanitize_checkbox')));
    register_setting('my_seo_ssl_settings', 'my_seo_fix_mixed_content', array('type' => 'boolean', 'default' => 0, 'sanitize_callback' => array($this, 'sanitize_checkbox')));
    register_setting('my_seo_ssl_settings', 'my_seo_enable_hsts', array('type' => 'boolean', 'default' => 0, 'sanitize_callback' => array($this, 'sanitize_checkbox')));
    register_setting('my_seo_ssl_settings', 'my_seo_hsts_max_age', array('type' => 'integer', 'default' => 31536000));
    register_setting('my_seo_ssl_settings', 'my_seo_www_redirect', array('type' => 'string', 'default' => ''));

    register_setting('my_seo_general_settings', 'my_seo_plugin_language', array(
        'type' => 'string',
        'default' => '',
        'sanitize_callback' => 'sanitize_text_field'
    ));

    register_setting('my_seo_general_settings', 'my_seo_white_label_enabled');
    register_setting('my_seo_general_settings', 'my_seo_white_label_name');
    register_setting('my_seo_general_settings', 'my_seo_white_label_hide_links');

    add_settings_section('my_seo_general_section', __('General SEO Settings', 'seo-fury'), array($this, 'general_section_callback'), 'my_seo_general_settings');

    add_settings_section('my_seo_security_section', __('Settings', 'seo-fury'), array($this, 'security_section_callback'), 'my_seo_ssl_settings');

    add_settings_section('my_seo_white_label_section', __('White Label', 'seo-fury'), array($this, 'white_label_section_callback'), 'my_seo_general_settings');

    add_settings_field(
        'my_seo_plugin_language',
        __('Plugin Language', 'seo-fury'),
        array($this, 'plugin_language_callback'),
        'my_seo_general_settings',
        'my_seo_general_section'
    );

    add_settings_field(
        'my_seo_transliterate_urls',
        __('URL Transliteration', 'seo-fury'),
        array($this, 'transliterate_urls_callback'),
        'my_seo_general_settings',
        'my_seo_general_section'
    );

    add_settings_field(
        'my_seo_white_label_enabled',
        __('Enable White Label', 'seo-fury'),
        array($this, 'white_label_enabled_callback'),
        'my_seo_general_settings',
        'my_seo_white_label_section'
    );

    add_settings_field(
        'my_seo_white_label_name',
        __('Plugin Name', 'seo-fury'),
        array($this, 'white_label_name_callback'),
        'my_seo_general_settings',
        'my_seo_white_label_section'
    );

        add_settings_field(
            'my_seo_white_label_hide_links',
            __('Hide Branding Links', 'seo-fury'),
        array($this, 'white_label_hide_links_callback'),
        'my_seo_general_settings',
        'my_seo_white_label_section'
    );

    add_settings_field('my_seo_bulk_transliterate', __('Bulk URL Conversion', 'seo-fury'), array($this, 'bulk_transliterate_callback'), 'my_seo_general_settings', 'my_seo_general_section');

    add_settings_field('my_seo_force_https', __('Force HTTPS', 'seo-fury'), array($this, 'force_https_callback'), 'my_seo_ssl_settings', 'my_seo_security_section');
    add_settings_field('my_seo_fix_mixed_content', __('Fix Mixed Content', 'seo-fury'), array($this, 'fix_mixed_content_callback'), 'my_seo_ssl_settings', 'my_seo_security_section');
    add_settings_field('my_seo_enable_hsts', __('HSTS Header', 'seo-fury'), array($this, 'enable_hsts_callback'), 'my_seo_ssl_settings', 'my_seo_security_section');
    add_settings_field('my_seo_www_redirect', __('WWW Redirect', 'seo-fury'), array($this, 'www_redirect_callback'), 'my_seo_ssl_settings', 'my_seo_security_section');
}

    public function general_section_callback() {
        echo '<p>' . esc_html__('Configure the main SEO settings for your site.', 'seo-fury') . '</p>';
    }

    public function plugin_language_callback() {
        $current = get_option('my_seo_plugin_language', '');
        $languages = array(
            ''      => '🌐 ' . __('Auto (WordPress language)', 'seo-fury'),
            'en_US' => '🇬🇧 English',
            'ru_RU' => '🇷🇺 Русский',
            'de_DE' => '🇩🇪 Deutsch',
            'fr_FR' => '🇫🇷 Français',
            'es_ES' => '🇪🇸 Español',
        );
        ?>
        <select name="my_seo_plugin_language" onchange="HTMLFormElement.prototype.submit.call(this.form);">
            <?php foreach ($languages as $code => $label) : ?>
                <option value="<?php echo esc_attr($code); ?>" <?php selected($current, $code); ?>>
                    <?php echo esc_html($label); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <p class="description">
            <?php esc_html_e('Choose the language for the plugin admin interface. "Auto" follows your WordPress site language.', 'seo-fury'); ?>
        </p>
        <?php
    }

    public function transliterate_urls_callback() {
        $enabled = get_option('my_seo_transliterate_urls', 0);
        ?>
        <label>
            <input type="checkbox" name="my_seo_transliterate_urls" value="1" <?php checked($enabled, 1); ?>>
            <?php esc_html_e('Enable automatic transliteration for non-Latin URLs', 'seo-fury'); ?>
        </label>
        <p class="description">
            <?php esc_html_e('Automatically converts non-Latin characters to Latin in post and page URLs.', 'seo-fury'); ?>
        </p>
        <?php
    }

    public function sanitize_checkbox($input) {
        return ($input == 1) ? 1 : 0;
    }

public function transliterate_slug($slug, $raw_title = '', $context = 'save', $force = false) {
    if (!$force && ((defined('REST_REQUEST') && REST_REQUEST) || wp_doing_ajax())) {
        return $slug;
    }

    if (!$force && !get_option('my_seo_transliterate_urls', 0)) {
        return $slug;
    }

    $text = !empty($raw_title) ? $raw_title : $slug;

    if (!preg_match('/[А-Яа-яЁёІіЇїЄєҐґ]/u', $text)) {
        return $slug;
    }

    $transliteration = array(
        'а' => 'a', 'б' => 'b', 'в' => 'v', 'г' => 'g', 'д' => 'd',
        'е' => 'e', 'ё' => 'yo', 'ж' => 'zh', 'з' => 'z', 'и' => 'i',
        'й' => 'y', 'к' => 'k', 'л' => 'l', 'м' => 'm', 'н' => 'n',
        'о' => 'o', 'п' => 'p', 'р' => 'r', 'с' => 's', 'т' => 't',
        'у' => 'u', 'ф' => 'f', 'х' => 'h', 'ц' => 'ts', 'ч' => 'ch',
        'ш' => 'sh', 'щ' => 'sch', 'ъ' => '', 'ы' => 'y', 'ь' => '',
        'э' => 'e', 'ю' => 'yu', 'я' => 'ya',
        'А' => 'A', 'Б' => 'B', 'В' => 'V', 'Г' => 'G', 'Д' => 'D',
        'Е' => 'E', 'Ё' => 'Yo', 'Ж' => 'Zh', 'З' => 'Z', 'И' => 'I',
        'Й' => 'Y', 'К' => 'K', 'Л' => 'L', 'М' => 'M', 'Н' => 'N',
        'О' => 'O', 'П' => 'P', 'Р' => 'R', 'С' => 'S', 'Т' => 'T',
        'У' => 'U', 'Ф' => 'F', 'Х' => 'H', 'Ц' => 'Ts', 'Ч' => 'Ch',
        'Ш' => 'Sh', 'Щ' => 'Sch', 'Ъ' => '', 'Ы' => 'Y', 'Ь' => '',
        'Э' => 'E', 'Ю' => 'Yu', 'Я' => 'Ya',
        'і' => 'i', 'ї' => 'yi', 'є' => 'ye', 'ґ' => 'g',
        'І' => 'I', 'Ї' => 'Yi', 'Є' => 'Ye', 'Ґ' => 'G',
    );

    $transliterated = strtr($text, $transliteration);

    $transliterated = mb_strtolower($transliterated, 'UTF-8');

    $transliterated = preg_replace('/[^a-z0-9]+/i', '-', $transliterated);

    $transliterated = trim($transliterated, '-');

    $transliterated = preg_replace('/-+/', '-', $transliterated);

    return $transliterated;
}

public function ajax_bulk_transliterate() {
    check_ajax_referer('seo_bulk_transliterate_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Insufficient permissions.', 'seo-fury'));
    }

    $post_types = isset($_POST['post_types']) ? array_map('sanitize_text_field', $_POST['post_types']) : array('post', 'page');

    $args = array(
        'post_type' => $post_types,
        'post_status' => 'any',
        'posts_per_page' => -1,
        'fields' => 'ids'
    );

    $posts = get_posts($args);
    $updated = 0;
    $skipped = 0;

    foreach ($posts as $post_id) {
        $post = get_post($post_id);
        $current_slug = $post->post_name;

        $decoded_slug = urldecode($current_slug);

        $has_cyrillic = preg_match('/[А-Яа-яЁёІіЇїЄєҐґ]/u', $decoded_slug);

        $has_url_encoded = preg_match('/%[0-9A-F]{2}/', $current_slug);

        if ($has_cyrillic || $has_url_encoded) {
            $new_slug = $this->transliterate_slug($decoded_slug, $post->post_title, 'save', true);

            if (!empty($new_slug) && !preg_match('/[А-Яа-яЁёІіЇїЄєҐґ%]/u', $new_slug)) {
                $result = wp_update_post(array(
                    'ID' => $post_id,
                    'post_name' => $new_slug
                ), true);

                if (!is_wp_error($result)) {
                    $updated++;
                } else {
                    $skipped++;
                }
            } else {
                $skipped++;
            }
        } else {
            $skipped++;
        }
    }

    wp_send_json_success(array(
        'updated' => $updated,
        'skipped' => $skipped,
        'total' => count($posts)
    ));
}

public function ajax_get_url_stats() {
    check_ajax_referer('seo_bulk_transliterate_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Insufficient permissions.', 'seo-fury'));
    }

    $post_types = isset($_POST['post_types']) ? array_map('sanitize_text_field', $_POST['post_types']) : array('post', 'page');

    $args = array(
        'post_type' => $post_types,
        'post_status' => 'any',
        'posts_per_page' => -1,
        'fields' => 'ids'
    );

    $posts = get_posts($args);
    $cyrillic_count = 0;
    $latin_count = 0;

    foreach ($posts as $post_id) {
        $post = get_post($post_id);
        $slug = $post->post_name;

        $decoded_slug = urldecode($slug);

        if (preg_match('/[А-Яа-яЁёІіЏїЄєҐґ]/u', $decoded_slug) || preg_match('/%[0-9A-F]{2}/', $slug)) {
            $cyrillic_count++;
        } else {
            $latin_count++;
        }
    }

    wp_send_json_success(array(
        'total' => count($posts),
        'cyrillic' => $cyrillic_count,
        'latin' => $latin_count
    ));
}

public function bulk_transliterate_callback() {
    ?>
    <div class="seo-card" style="max-width: 800px;">
        <p><?php esc_html_e('Convert existing non-Latin URLs to Latin slugs.', 'seo-fury'); ?></p>

        <div id="url-stats" style="background: #f6f7f7; padding: 15px; border-radius: 4px; margin-bottom: 15px;">
            <p style="margin: 0;"><strong><?php esc_html_e('Loading stats...', 'seo-fury'); ?></strong></p>
        </div>

        <p>
            <label style="display: block; margin-bottom: 10px;">
                <input type="checkbox" name="bulk_post_types[]" value="post" checked>
                <?php esc_html_e('Posts', 'seo-fury'); ?>
            </label>
            <label style="display: block; margin-bottom: 10px;">
                <input type="checkbox" name="bulk_post_types[]" value="page" checked>
                <?php esc_html_e('Pages', 'seo-fury'); ?>
            </label>
        </p>

        <p>
            <button type="button" id="refresh-url-stats" class="button"><?php echo '🔄 ' . esc_html__('Refresh Stats', 'seo-fury'); ?></button>
            <button type="button" id="bulk-transliterate-btn" class="button button-primary"><?php echo '🚀 ' . esc_html__('Convert URLs', 'seo-fury'); ?></button>
        </p>

        <div id="bulk-transliterate-result" style="margin-top: 15px;"></div>
    </div>

    <script>
    jQuery(document).ready(function($) {
        loadUrlStats();

        $('#refresh-url-stats').on('click', function() {
            loadUrlStats();
        });

        function loadUrlStats() {
            var postTypes = [];
            $('input[name="bulk_post_types[]"]:checked').each(function() {
                postTypes.push($(this).val());
            });

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'seo_get_url_stats',
                    nonce: '<?php echo wp_create_nonce('seo_bulk_transliterate_nonce'); ?>',
                    post_types: postTypes
                },
                success: function(response) {
                    if (response.success) {
                        var html = '<p style="margin: 0;"><strong><?php echo esc_js(__('URL stats:', 'seo-fury')); ?></strong></p>';
                        html += '<p style="margin: 5px 0;"><?php echo esc_js(__('Total:', 'seo-fury')); ?> <strong>' + response.data.total + '</strong></p>';
                        html += '<p style="margin: 5px 0;"><?php echo esc_js(__('Non-Latin:', 'seo-fury')); ?> <strong style="color: #d63638;">' + response.data.cyrillic + '</strong></p>';
                        html += '<p style="margin: 5px 0;"><?php echo esc_js(__('Latin:', 'seo-fury')); ?> <strong style="color: #00a32a;">' + response.data.latin + '</strong></p>';
                        $('#url-stats').html(html);
                    }
                }
            });
        }

        $('#bulk-transliterate-btn').on('click', function() {
            if (!confirm('<?php echo esc_js(__('Are you sure? This will change URLs for all posts with non-Latin slugs.', 'seo-fury')); ?>\n\n<?php echo esc_js(__('We recommend backing up the database first.', 'seo-fury')); ?>')) {
                return;
            }

            var postTypes = [];
            $('input[name="bulk_post_types[]"]:checked').each(function() {
                postTypes.push($(this).val());
            });

            var btn = $(this);
            var originalText = btn.text();
            btn.prop('disabled', true).text('⏳ <?php echo esc_js(__('Converting...', 'seo-fury')); ?>');

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'seo_bulk_transliterate',
                    nonce: '<?php echo wp_create_nonce('seo_bulk_transliterate_nonce'); ?>',
                    post_types: postTypes
                },
                success: function(response) {
                    if (response.success) {
                        var html = '<div class="seo-alert success">';
                        html += '<strong>✅ <?php echo esc_js(__('Conversion completed!', 'seo-fury')); ?></strong><br>';
                        html += '<?php echo esc_js(__('Updated:', 'seo-fury')); ?> <strong>' + response.data.updated + '</strong><br>';
                        html += '<?php echo esc_js(__('Skipped (already Latin):', 'seo-fury')); ?> <strong>' + response.data.skipped + '</strong><br>';
                        html += '<?php echo esc_js(__('Total checked:', 'seo-fury')); ?> <strong>' + response.data.total + '</strong>';
                        html += '</div>';
                        $('#bulk-transliterate-result').html(html);
                        loadUrlStats();
                    } else {
                        $('#bulk-transliterate-result').html('<div class="seo-alert error"><strong>❌ <?php echo esc_js(__('Error:', 'seo-fury')); ?></strong> ' + response.data + '</div>');
                    }
                },
                error: function() {
                    $('#bulk-transliterate-result').html('<div class="seo-alert error"><strong>❌ <?php echo esc_js(__('Server connection error', 'seo-fury')); ?></strong></div>');
                },
                complete: function() {
                    btn.prop('disabled', false).text(originalText);
                }
            });
        });
    });
    </script>
    <?php
}

    public function security_section_callback() {
        echo '<p>' . esc_html__('Configure HTTPS enforcement, HSTS headers, and WWW redirect.', 'seo-fury') . '</p>';
    }

    public function force_https_callback() {
        $enabled = get_option('my_seo_force_https', 0);
        ?>
        <label>
            <input type="checkbox" name="my_seo_force_https" value="1" <?php checked($enabled, 1); ?>>
            <?php esc_html_e('Redirect all HTTP requests to HTTPS (301)', 'seo-fury'); ?>
        </label>
        <p class="description"><?php esc_html_e('Requires a valid SSL certificate installed on your server.', 'seo-fury'); ?></p>
        <?php
    }

    public function fix_mixed_content_callback() {
        $enabled = get_option('my_seo_fix_mixed_content', 0);
        ?>
        <label>
            <input type="checkbox" name="my_seo_fix_mixed_content" value="1" <?php checked($enabled, 1); ?>>
            <?php esc_html_e('Automatically replace HTTP with HTTPS in page output', 'seo-fury'); ?>
        </label>
        <p class="description"><?php esc_html_e('Fixes mixed content warnings by rewriting internal HTTP URLs to HTTPS on the frontend.', 'seo-fury'); ?></p>
        <?php
    }

    public function enable_hsts_callback() {
        $enabled = get_option('my_seo_enable_hsts', 0);
        $max_age = get_option('my_seo_hsts_max_age', 31536000);
        ?>
        <label>
            <input type="checkbox" name="my_seo_enable_hsts" value="1" <?php checked($enabled, 1); ?>>
            <?php esc_html_e('Send Strict-Transport-Security header', 'seo-fury'); ?>
        </label>
        <div style="margin-top: 8px;">
            <label><?php esc_html_e('Max-Age (seconds):', 'seo-fury'); ?>
                <input type="number" name="my_seo_hsts_max_age" value="<?php echo esc_attr($max_age); ?>" min="0" style="width: 120px;">
            </label>
            <span class="description" style="margin-left: 8px;"><?php esc_html_e('Default: 31536000 (1 year)', 'seo-fury'); ?></span>
        </div>
        <p class="description"><?php esc_html_e('Tells browsers to always use HTTPS for this domain. Only enable if SSL is fully configured.', 'seo-fury'); ?></p>
        <?php
    }

    public function www_redirect_callback() {
        $value = get_option('my_seo_www_redirect', '');
        ?>
        <select name="my_seo_www_redirect">
            <option value="" <?php selected($value, ''); ?>><?php esc_html_e('No redirect', 'seo-fury'); ?></option>
            <option value="force_www" <?php selected($value, 'force_www'); ?>><?php esc_html_e('Force www (redirect non-www to www)', 'seo-fury'); ?></option>
            <option value="force_no_www" <?php selected($value, 'force_no_www'); ?>><?php esc_html_e('Force non-www (redirect www to non-www)', 'seo-fury'); ?></option>
        </select>
        <p class="description"><?php esc_html_e('Enforce a canonical www or non-www version of your domain (301 redirect).', 'seo-fury'); ?></p>
        <?php
    }

    public function handle_security_redirects() {
        if (is_admin() || wp_doing_ajax() || (defined('REST_REQUEST') && REST_REQUEST) || (defined('WP_CLI') && WP_CLI)) {
            return;
        }

        $force_https = get_option('my_seo_force_https', 0);
        if ($force_https && !is_ssl()) {
            $redirect_url = 'https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
            wp_redirect($redirect_url, 301);
            exit;
        }

        $www_redirect = get_option('my_seo_www_redirect', '');
        if (!empty($www_redirect)) {
            $host = $_SERVER['HTTP_HOST'];
            $scheme = is_ssl() ? 'https' : 'http';

            if ($www_redirect === 'force_www' && strpos($host, 'www.') !== 0) {
                wp_redirect($scheme . '://www.' . $host . $_SERVER['REQUEST_URI'], 301);
                exit;
            } elseif ($www_redirect === 'force_no_www' && strpos($host, 'www.') === 0) {
                $host_no_www = substr($host, 4);
                wp_redirect($scheme . '://' . $host_no_www . $_SERVER['REQUEST_URI'], 301);
                exit;
            }
        }
    }

    public function send_security_headers() {
        if (get_option('my_seo_enable_hsts', 0) && is_ssl()) {
            $max_age = intval(get_option('my_seo_hsts_max_age', 31536000));
            header('Strict-Transport-Security: max-age=' . $max_age . '; includeSubDomains');
        }
    }

    public function start_mixed_content_buffer() {
        if (is_admin()) {
            return;
        }
        ob_start(array($this, 'fix_mixed_content_output'));
    }

    public function fix_mixed_content_output($buffer) {
        $site_url = site_url();
        $parsed = parse_url($site_url);
        $domain = isset($parsed['host']) ? $parsed['host'] : '';
        if (empty($domain)) {
            return $buffer;
        }
        $buffer = str_replace('http://' . $domain, 'https://' . $domain, $buffer);
        $buffer = str_replace('http://www.' . $domain, 'https://www.' . $domain, $buffer);
        return $buffer;
    }

    public function white_label_section_callback() {
        $core = SEO_Fury_Core::get_instance();
        $pro = $core->get_pro_features();
        $is_pro = $pro->is_pro();
        ob_start();

        $white_label_enabled = get_option('my_seo_white_label_enabled', 0);
        $hide_links = get_option('my_seo_white_label_hide_links', 0);
        $should_hide = $white_label_enabled && $hide_links;

        if (!$is_pro && !$should_hide) {
            ?>
            <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 8px; margin: 20px 0;">
                <div style="display: flex; align-items: center; gap: 20px;">
                    <div style="font-size: 48px;">🎭</div>
                    <div>
                        <h3 style="color: white; margin: 0 0 8px 0;"><?php esc_html_e('White Label is a PRO feature', 'seo-fury'); ?></h3>
                        <p style="margin: 0 0 12px 0; opacity: 0.95;">
                            <?php esc_html_e('Rename the plugin and remove branding. Perfect for agencies and resellers.', 'seo-fury'); ?>
                        </p>
                        <a href="https://seo-fury.com/#pricing" target="_blank" class="button button-primary" style="background: white; color: #667eea; border: none; font-weight: 600;">
                            <?php echo '🚀 ' . esc_html__('Upgrade to PRO', 'seo-fury'); ?>
                        </a>
                    </div>
                </div>
            </div>
            <div style="opacity: 0.4; pointer-events: none;">
            <?php
        } else {
            echo '<p>' . esc_html__('Customize the plugin branding for your product.', 'seo-fury') . '</p>';
        }
        echo ob_get_clean();
    }

    public function white_label_enabled_callback() {
        $enabled = get_option('my_seo_white_label_enabled', 0);
        ?>
        <label>
            <input type="checkbox" name="my_seo_white_label_enabled" value="1" <?php checked($enabled, 1); ?>>
            <?php esc_html_e('Enable white label', 'seo-fury'); ?>
        </label>
        <p class="description"><?php esc_html_e('When enabled, the plugin name changes across the admin UI.', 'seo-fury'); ?></p>
        <?php
    }

    public function white_label_name_callback() {
        $brand = function_exists('seo_fury_get_brand_name') ? seo_fury_get_brand_name() : 'SEO Fury';
        $name = get_option('my_seo_white_label_name', $brand);
        ?>
        <input type="text"
               name="my_seo_white_label_name"
               value="<?php echo esc_attr($name); ?>"
               class="regular-text"
               placeholder="<?php echo esc_attr($brand); ?>">
        <p class="description"><?php esc_html_e('New plugin name used across the UI.', 'seo-fury'); ?></p>
        <?php
    }

    public function white_label_hide_links_callback() {
        $hide = get_option('my_seo_white_label_hide_links', 0);

        $core = SEO_Fury_Core::get_instance();
        $pro = $core->get_pro_features();
        $is_pro = $pro->is_pro();

        ?>
        <label>
            <input type="checkbox" name="my_seo_white_label_hide_links" value="1" <?php checked($hide, 1); ?>>
            <?php esc_html_e('Hide all branding links', 'seo-fury'); ?>
        </label>
        <p class="description"><?php esc_html_e('Removes upgrade links and branding references.', 'seo-fury'); ?></p>
        <?php

        if (!$is_pro) {
            echo '</div>';
        }
    }

}
