<?php

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Fury_Integrity {

    private $critical_files = array(
        'includes/class-license.php',
        'includes/class-fingerprint.php',
        'includes/class-core.php',
        'includes/class-pro-features.php',
        'seo-fury.php'
    );

    private $manifest = array();
    private $api_url = 'https://api.seo-fury.com/api/v1';
    private $fallback_api_url = 'https://api-ru.seo-fury.com/api/v1';

    public function verify_integrity() {
        if (empty($this->manifest)) {
            $this->manifest = $this->load_manifest();

            if (empty($this->manifest)) {
                $this->generate_manifest();
                return true;
            }
        }

        $tampered = false;

        foreach ($this->critical_files as $file) {
            $full_path = MY_SEO_PLUGIN_PATH . $file;

            if (!file_exists($full_path)) {
                $tampered = true;
                continue;
            }

            $current_hash = $this->get_file_hash($full_path);
            $stored_hash = isset($this->manifest[$file]) ? $this->manifest[$file] : '';

            if (empty($stored_hash)) {
                continue;
            }

            if ($current_hash !== $stored_hash) {
                $tampered = true;
            }
        }

        if ($tampered) {
            $server_check = $this->verify_with_server();
            if ($server_check === false) {
                return false;
            }
            if ($server_check === true) {
                $this->generate_manifest();
                return true;
            }
            return false;
        }

        return true;
    }

    private function verify_with_server() {
        $hashes = $this->get_current_hashes();
        $license_salt = 'seo_fury_2025';
        $key_option = '_sf_' . substr(md5($license_salt . 'key'), 0, 8);
        $license_key = get_option($key_option, '');

        if (empty($license_key)) {
            return null;
        }

        $domain = $this->get_domain();

        $request_body = json_encode(array(
            'licenseKey' => $license_key,
            'domain' => $domain,
            'pluginVersion' => defined('SEO_FURY_VERSION') ? SEO_FURY_VERSION : '0.0.0',
            'fileHashes' => $hashes,
        ));

        $request_args = array(
            'timeout' => 15,
            'headers' => array('Content-Type' => 'application/json'),
            'body' => $request_body,
        );

        $response = wp_remote_post($this->api_url . '/plugin/verify-integrity', $request_args);

        if (is_wp_error($response)) {
            $response = wp_remote_post($this->fallback_api_url . '/plugin/verify-integrity', $request_args);
        }

        if (is_wp_error($response)) {
            return null;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code < 200 || $code >= 300) {
            return null;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (!is_array($body)) {
            return null;
        }

        if (!empty($body['success']) && !empty($body['data']['verified'])) {
            if (!empty($body['data']['signature'])) {
                update_option('seo_fury_server_signature', array(
                    'sig' => $body['data']['signature'],
                    'ts' => time(),
                    'version' => defined('SEO_FURY_VERSION') ? SEO_FURY_VERSION : '0.0.0',
                ), false);
            }
            return true;
        }

        return false;
    }

    public function verify_server_signature() {
        $sig_data = get_option('seo_fury_server_signature', array());
        if (empty($sig_data) || !isset($sig_data['sig'])) {
            return null;
        }

        $hashes = $this->get_current_hashes();
        $version = defined('SEO_FURY_VERSION') ? SEO_FURY_VERSION : '0.0.0';

        $payload = $version . '|' . implode('|', array_values($hashes));
        $expected = hash_hmac('sha256', $payload, 'sf_integrity_' . $this->get_domain());

        return hash_equals($sig_data['sig'], $expected);
    }

    private function get_current_hashes() {
        $hashes = array();
        foreach ($this->critical_files as $file) {
            $full_path = MY_SEO_PLUGIN_PATH . $file;
            if (file_exists($full_path)) {
                $hashes[$file] = $this->get_file_hash($full_path);
            }
        }
        return $hashes;
    }

    private function get_domain() {
        $url = home_url();
        $parsed = parse_url($url);
        $domain = $parsed['host'] ?? 'localhost';
        return strtolower(preg_replace('/^www\./', '', $domain));
    }

    private function get_file_hash($file_path) {
        if (!file_exists($file_path)) {
            return '';
        }
        return hash_file('sha256', $file_path);
    }

    public function generate_manifest() {
        $manifest = array();

        foreach ($this->critical_files as $file) {
            $full_path = MY_SEO_PLUGIN_PATH . $file;
            if (!file_exists($full_path)) {
                continue;
            }
            $manifest[$file] = $this->get_file_hash($full_path);
        }

        $this->save_manifest($manifest);
        return $manifest;
    }

    private function save_manifest($manifest) {
        update_option('seo_fury_integrity_manifest', $manifest);
    }

    private function load_manifest() {
        return get_option('seo_fury_integrity_manifest', array());
    }

    public function clear_manifest() {
        delete_option('seo_fury_integrity_manifest');
        delete_option('seo_fury_server_signature');
    }
}
