<?php

if (!defined('ABSPATH')) {
    exit;
}

class My_SEO_Plugin_License {

    private $api_url = 'https://api.seo-fury.com/api/v1';

    private $fallback_servers = [
        'https://api.seo-fury.com/api/v1',
        'https://api-ru.seo-fury.com/api/v1',
    ];

    private $cache_ttl = 259200;

    private $option_key;
    private $option_status;
    private $option_token;
    private $option_cache;
    private $fingerprint;

    public function __construct() {
        $salt = 'seo_fury_2025';
        $this->option_key = '_sf_' . substr(md5($salt . 'key'), 0, 8);
        $this->option_status = '_sf_' . substr(md5($salt . 'status'), 0, 8);
        $this->option_token = '_sf_' . substr(md5($salt . 'token'), 0, 8);
        $this->option_cache = '_sf_' . substr(md5($salt . 'cache'), 0, 8);
        $this->fingerprint = new My_SEO_Plugin_Fingerprint();

        $this->init_hooks();
    }

    private function init_hooks() {
        add_action('sf_license_heartbeat', array($this, 'validate_license'));

        if (!wp_next_scheduled('sf_license_heartbeat')) {
            wp_schedule_event(time(), 'daily', 'sf_license_heartbeat');
        }

        add_action('wp_ajax_sf_activate_license', array($this, 'ajax_activate_license'));
        add_action('wp_ajax_sf_deactivate_license', array($this, 'ajax_deactivate_license'));
        add_action('wp_ajax_sf_check_license', array($this, 'ajax_check_license'));
    }

public function activate_license($license_key) {
    $license_key = sanitize_text_field(trim($license_key));

    error_log('=== LICENSE ACTIVATION START ===');
    error_log('License key: ' . $license_key);

    if (!$this->validate_key_format($license_key)) {
        error_log('LICENSE: Invalid key format');
        return array(
            'success' => false,
            'message' => __('Invalid license key format', 'seo-fury')
        );
    }

    error_log('LICENSE: Key format valid');

$domain = $this->get_domain();
$site_info = $this->fingerprint->get_site_info();

error_log('LICENSE: Domain: ' . $domain);
error_log('LICENSE: Site hash: ' . $site_info['siteHash']);

$data = array(
    'licenseKey' => $license_key,
    'domain' => $domain,
    'siteUrl' => $site_info['siteUrl'],
    'homeUrl' => $site_info['homeUrl'],
    'serverIp' => $_SERVER['SERVER_ADDR'] ?? 'unknown',
    'phpVersion' => $site_info['phpVersion'],
    'wpVersion' => $site_info['wpVersion'],
    'siteHash' => $site_info['siteHash'],
    'dbPrefix' => $site_info['dbPrefix']
);

    error_log('LICENSE: Request data: ' . print_r($data, true));

    $response = $this->api_request('/plugin/activate', 'POST', $data);

    error_log('LICENSE: API Response: ' . print_r($response, true));
    error_log('LICENSE: Response is_array: ' . (is_array($response) ? 'YES' : 'NO'));

    if (!is_array($response)) {
        error_log('LICENSE ERROR: Response is not array!');
        return array(
            'success' => false,
            'message' => __('Error: invalid API response format', 'seo-fury')
        );
    }

    if (empty($response['success'])) {
        error_log('LICENSE ERROR: Response success is false');
        return array(
            'success' => false,
            'message' => $response['message'] ?? __('License activation failed', 'seo-fury')
        );
    }

    error_log('LICENSE: Activation successful - preparing data');

    update_option($this->option_key, $license_key);
    error_log('LICENSE: Key saved');

    $token = $response['data']['token'] ?? $response['token'] ?? null;
    if ($token) {
        update_option($this->option_token, $token);
        error_log('LICENSE: Token saved');
    }

    $license_data = $response['data']['license'] ?? $response['data'] ?? array();

    error_log('LICENSE: Extracted license_data: ' . print_r($license_data, true));

    update_option($this->option_status, $license_data);
    update_option($this->option_cache, array(
        'data' => $license_data,
        'timestamp' => time()
    ));

    error_log('LICENSE: All options saved successfully');

    $result = array(
        'success' => true,
        'message' => __('License activated successfully!', 'seo-fury'),
        'data' => $license_data
    );

    error_log('LICENSE: Returning result: ' . print_r($result, true));

    return $result;
}

public function validate_license() {
    $license_key = get_option($this->option_key);

    if (!$license_key) {
        error_log('LICENSE: No license key found');
        return false;
    }

$domain = $this->get_domain();
$site_info = $this->fingerprint->get_site_info();

$data = array(
    'licenseKey' => $license_key,
    'domain' => $domain,
    'siteHash' => $site_info['siteHash']
);

    error_log('LICENSE: Validating with domain: ' . $domain);

    $response = $this->api_request('/plugin/validate', 'POST', $data);

    error_log('LICENSE: API response: ' . print_r($response, true));

    $is_valid = false;

    if ($response['success']) {
        if (isset($response['valid']) && $response['valid']) {
            $is_valid = true;
        }

        if (isset($response['data']['valid']) && $response['data']['valid']) {
            $is_valid = true;
        }

        if (isset($response['data']['status']) && $response['data']['status'] === 'active') {
            $is_valid = true;
        }
    }

    error_log('LICENSE: Is valid: ' . ($is_valid ? 'YES' : 'NO'));

        if ($is_valid) {
        error_log('LICENSE: Validation successful');

        $license_data = $response['data']['license'] ?? $response['data'] ?? $response;

        update_option($this->option_status, $license_data);
        update_option($this->option_cache, array(
            'data' => $license_data,
            'timestamp' => time()
        ));

        error_log('LICENSE: Cache updated successfully');
        return true;
    } else {
        error_log('LICENSE: Validation failed, deactivating features');
        $this->deactivate_features();
        return false;
    }
}

public function deactivate_license() {
    $license_key = get_option($this->option_key);

    if ($license_key) {
        $domain = $this->get_domain();

        $data = array(
            'licenseKey' => $license_key,
            'domain' => $domain
        );

        error_log('=== LICENSE DEACTIVATION ===');
        error_log('Key: ' . $license_key);
        error_log('Domain: ' . $domain);

        $response = $this->api_request('/plugin/deactivate', 'POST', $data);

        error_log('Deactivation API response: ' . print_r($response, true));
    }

    delete_option($this->option_key);
    delete_option($this->option_token);
    delete_option($this->option_status);
    delete_option($this->option_cache);

    error_log('LICENSE: Local data cleared');

    return array(
        'success' => true,
        'message' => __('License deactivated', 'seo-fury')
    );
}

    public function get_status() {
        $status = get_option($this->option_status);
        $cache = get_option($this->option_cache);

        if (!$status && $cache) {
            $age = time() - $cache['timestamp'];
            if ($age < $this->cache_ttl) {
                return $cache['data'];
            }
        }

        return $status;
    }

    public function is_active() {
        $status = $this->get_status();

        if (!$status) {
            return false;
        }

        return isset($status['status']) && $status['status'] === 'active';
    }

    public function get_plan() {
        $status = $this->get_status();
        return $status['plan'] ?? 'free';
    }

    public function is_feature_enabled($feature) {
        if (!$this->is_active()) {
            return false;
        }

        $status = $this->get_status();
        $features = $status['features'] ?? array();

        switch ($feature) {
            case 'advanced_seo':
                return !empty($features['advancedSeo']);
            case 'unlimited_redirects':
                return isset($features['maxRedirects']) && $features['maxRedirects'] > 100;
            case 'unlimited_404':
                return isset($features['max404Logs']) && $features['max404Logs'] > 100;
            case 'white_label':
                return !empty($features['whiteLabel']);
            case 'priority_support':
                return !empty($features['prioritySupport']);
            default:
                return false;
        }
    }

private function api_request($endpoint, $method = 'GET', $data = array()) {
    $url = $this->api_url . $endpoint;

    error_log('=== API REQUEST ===');
    error_log('URL: ' . $url);
    error_log('Method: ' . $method);
    error_log('Data: ' . print_r($data, true));

    $args = array(
        'method' => $method,
        'timeout' => 30,
        'headers' => array(
            'Content-Type' => 'application/json',
        ),
        'body' => $method !== 'GET' ? json_encode($data) : null
    );

        $token = get_option($this->option_token);
        if ($token) {
            $args['headers']['Authorization'] = 'Bearer ' . $token;
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            foreach ($this->fallback_servers as $fallback_url) {
                if ($fallback_url === $this->api_url) continue;

                $url = $fallback_url . $endpoint;
                $response = wp_remote_request($url, $args);

                if (!is_wp_error($response)) {
                    break;
                }
            }
        }

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }

        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);

        return $result ?? array('success' => false, 'message' => __('Invalid response', 'seo-fury'));
    }

private function validate_key_format($key) {
    $pro_pattern = '/^SEOFURY-[A-Z0-9]{5}-[A-Z0-9]{6}-[A-Z0-9]{5}-[A-Z0-9]{5}$/';

    $free_pattern = '/^FREE-[A-F0-9]{16}$/';

    return preg_match($pro_pattern, $key) || preg_match($free_pattern, $key);
}

    private function get_domain() {
        $url = home_url();
        $parsed = parse_url($url);
        $domain = $parsed['host'] ?? 'localhost';

        $domain = preg_replace('/^www\./', '', $domain);

        return $domain;
    }

    private function deactivate_features() {
        update_option($this->option_status, array(
            'status' => 'inactive',
            'plan' => 'free'
        ));
    }

public function ajax_activate_license() {
    check_ajax_referer('sf_license_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Insufficient permissions', 'seo-fury'));
        return;
    }

    $license_key = isset($_POST['license_key']) ? sanitize_text_field($_POST['license_key']) : '';

    if (empty($license_key)) {
        wp_send_json_error(__('License key is required', 'seo-fury'));
        return;
    }

    error_log('=== AJAX ACTIVATE START ===');
    error_log('License key from POST: ' . $license_key);

    $result = $this->activate_license($license_key);

    error_log('Activate result: ' . print_r($result, true));
    error_log('Result is array: ' . (is_array($result) ? 'YES' : 'NO'));
    error_log('Result success isset: ' . (isset($result['success']) ? 'YES' : 'NO'));

    if (!is_array($result)) {
        error_log('ERROR: Result is not array!');
        wp_send_json_error(__('Invalid response format', 'seo-fury'));
        return;
    }

    if (!empty($result['success'])) {
        error_log('SUCCESS: Sending success response');
        wp_send_json_success(array(
            'message' => $result['message'] ?? __('License activated', 'seo-fury'),
            'data' => $result['data'] ?? array()
        ));
    } else {
        error_log('ERROR: Sending error response');
        $error_message = !empty($result['message']) ? $result['message'] : __('Activation failed', 'seo-fury');
        wp_send_json_error($error_message);
    }
}

    public function ajax_deactivate_license() {
        check_ajax_referer('sf_license_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'seo-fury'));
        }

        $result = $this->deactivate_license();
        wp_send_json_success($result);
    }

public function ajax_check_license() {
    check_ajax_referer('sf_license_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Insufficient permissions', 'seo-fury'));
    }

    $cached_status = $this->get_status();
    $is_active = $this->is_active();
    $plan = $this->get_plan();

    error_log('=== LICENSE CHECK ===');
    error_log('Cached status: ' . print_r($cached_status, true));
    error_log('Is active: ' . ($is_active ? 'YES' : 'NO'));
    error_log('Plan: ' . $plan);

    $validation_result = $this->validate_license();
    error_log('Validation result: ' . ($validation_result ? 'SUCCESS' : 'FAILED'));

    if ($validation_result) {
        $cached_status = $this->get_status();
        $is_active = $this->is_active();
        $plan = $this->get_plan();

        error_log('Updated status: ' . print_r($cached_status, true));
        error_log('Updated is_active: ' . ($is_active ? 'YES' : 'NO'));
        error_log('Updated plan: ' . $plan);
    }

wp_send_json_success(array(
        'status' => $cached_status,
        'is_active' => $is_active,
        'plan' => $plan,
        'validation_performed' => $validation_result
    ));
}

public function get_cached_license_data() {
    $status = $this->get_status();
    if ($status) {
        return $status;
    }
    $cache = get_option($this->option_cache);
    return isset($cache["data"]) ? $cache["data"] : array();
}

}
