<?php

if (!defined('ABSPATH')) {
    exit;
}

class My_SEO_Plugin_Meta_Titles
{

    public function __construct()
    {
        $this->init_hooks();
    }

    private function init_hooks()
    {
        if ($this->is_rest_request()) {
            return;
        }

        add_filter('wp_robots', array($this, 'disable_wp_robots'), 999);

        add_action('wp_head', array($this, 'output_meta_tags'), 0);

        add_filter('wp_title', array($this, 'modify_title'), 10, 3);
        add_filter('document_title_parts', array($this, 'modify_document_title_parts'));
    }

    private function is_rest_request()
    {
        if (defined('REST_REQUEST') && REST_REQUEST) {
            return true;
        }

        if (isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], '/wp-json/') !== false) {
            return true;
        }

        return false;
    }

    public function disable_wp_robots($robots)
    {
        if (isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], '/wp-json/') !== false) {
            return $robots;
        }

        if (is_admin()) {
            return $robots;
        }

        if (wp_doing_ajax()) {
            return $robots;
        }

        return array();
    }

    public function register_settings()
    {
        register_setting('my_seo_meta_group', 'my_seo_robots_meta_index');
        register_setting('my_seo_meta_group', 'my_seo_robots_meta_noindex');
        register_setting('my_seo_meta_group', 'my_seo_robots_meta_nofollow');
        register_setting('my_seo_meta_group', 'my_seo_robots_meta_noarchive');
        register_setting('my_seo_meta_group', 'my_seo_robots_meta_noimageindex');
        register_setting('my_seo_meta_group', 'my_seo_robots_meta_nosnippet');

        register_setting('my_seo_meta_group', 'my_seo_robots_meta_max_snippet');
        register_setting('my_seo_meta_group', 'my_seo_robots_meta_max_video_preview');
        register_setting('my_seo_meta_group', 'my_seo_robots_meta_max_image_preview');

        register_setting('my_seo_meta_group', 'my_seo_noindex_empty_archives');

        register_setting('my_seo_meta_group', 'my_seo_title_separator');
        register_setting('my_seo_meta_group', 'my_seo_title_capitalize');

        register_setting('my_seo_meta_group', 'my_seo_twitter_card_type');

        register_setting('my_seo_meta_group', 'my_seo_title_template_home');
        register_setting('my_seo_meta_group', 'my_seo_description_template_home');
        register_setting('my_seo_meta_group', 'my_seo_title_template_author');
        register_setting('my_seo_meta_group', 'my_seo_description_template_author');
        register_setting('my_seo_meta_group', 'my_seo_title_template_search');
        register_setting('my_seo_meta_group', 'my_seo_description_template_search');
        register_setting('my_seo_meta_group', 'my_seo_title_template_404');
        register_setting('my_seo_meta_group', 'my_seo_description_template_404');
        register_setting('my_seo_meta_group', 'my_seo_title_template_date');
        register_setting('my_seo_meta_group', 'my_seo_description_template_date');
        register_setting('my_seo_meta_group', 'my_seo_title_template_blog');
        register_setting('my_seo_meta_group', 'my_seo_description_template_blog');

        $post_types = get_post_types(array('public' => true), 'objects');
        foreach ($post_types as $post_type) {
            register_setting('my_seo_meta_group', 'my_seo_title_template_' . $post_type->name);
            register_setting('my_seo_meta_group', 'my_seo_description_template_' . $post_type->name);
        }

        $taxonomies = get_taxonomies(array('public' => true), 'objects');
        foreach ($taxonomies as $taxonomy) {
            register_setting('my_seo_meta_group', 'my_seo_title_template_tax_' . $taxonomy->name);
            register_setting('my_seo_meta_group', 'my_seo_description_template_tax_' . $taxonomy->name);
        }

        add_settings_section(
            'my_seo_robots_meta_section',
            __('Robots Meta', 'seo-fury'),
            array($this, 'robots_meta_section_callback'),
            'my_seo_meta_group'
        );

        add_settings_field(
            'my_seo_robots_meta_tags',
            __('Robots Meta Tags', 'seo-fury'),
            array($this, 'robots_meta_tags_callback'),
            'my_seo_meta_group',
            'my_seo_robots_meta_section'
        );

        add_settings_section(
            'my_seo_advanced_robots_section',
            __('Advanced Robots Meta', 'seo-fury'),
            array($this, 'advanced_robots_section_callback'),
            'my_seo_meta_group'
        );

        add_settings_field(
            'my_seo_robots_snippet',
            __('Snippet', 'seo-fury'),
            array($this, 'robots_snippet_callback'),
            'my_seo_meta_group',
            'my_seo_advanced_robots_section'
        );

        add_settings_field(
            'my_seo_robots_video',
            __('Video Preview', 'seo-fury'),
            array($this, 'robots_video_callback'),
            'my_seo_meta_group',
            'my_seo_advanced_robots_section'
        );

        add_settings_field(
            'my_seo_robots_image',
            __('Image Preview Size', 'seo-fury'),
            array($this, 'robots_image_callback'),
            'my_seo_meta_group',
            'my_seo_advanced_robots_section'
        );

        add_settings_section(
            'my_seo_noindex_archives_section',
            __('Noindex Empty Archives', 'seo-fury'),
            array($this, 'noindex_archives_section_callback'),
            'my_seo_meta_group'
        );

        add_settings_field(
            'my_seo_noindex_empty',
            __('Empty Archives', 'seo-fury'),
            array($this, 'noindex_empty_callback'),
            'my_seo_meta_group',
            'my_seo_noindex_archives_section'
        );

        add_settings_section(
            'my_seo_title_settings_section',
            __('Title Settings', 'seo-fury'),
            array($this, 'title_settings_section_callback'),
            'my_seo_meta_group'
        );

        add_settings_field(
            'my_seo_title_sep',
            __('Title Separator', 'seo-fury'),
            array($this, 'title_separator_callback'),
            'my_seo_meta_group',
            'my_seo_title_settings_section'
        );

        add_settings_field(
            'my_seo_title_cap',
            __('Capitalize Titles', 'seo-fury'),
            array($this, 'title_capitalize_callback'),
            'my_seo_meta_group',
            'my_seo_title_settings_section'
        );

        add_settings_section(
            'my_seo_twitter_card_section',
            __('Twitter Card', 'seo-fury'),
            array($this, 'twitter_card_section_callback'),
            'my_seo_meta_group'
        );

        add_settings_section(
            'my_seo_content_types_section',
            __('Content Type Templates', 'seo-fury'),
            array($this, 'content_types_section_callback'),
            'my_seo_meta_group'
        );

        add_settings_field(
            'my_seo_content_types_templates',
            __('Templates', 'seo-fury'),
            array($this, 'content_types_templates_callback'),
            'my_seo_meta_group',
            'my_seo_content_types_section'
        );

        add_settings_field(
            'my_seo_twitter_card_type_field',
            __('Twitter Card Type', 'seo-fury'),
            array($this, 'twitter_card_type_callback'),
            'my_seo_meta_group',
            'my_seo_twitter_card_section'
        );
    }

    public function robots_meta_section_callback()
    {
        echo '<p>' . esc_html__('Default values for the robots meta tag.', 'seo-fury') . '</p>';
    }

    public function robots_meta_tags_callback()
    {
        $index = get_option('my_seo_robots_meta_index', 0);
        $noindex = get_option('my_seo_robots_meta_noindex', 0);
        $nofollow = get_option('my_seo_robots_meta_nofollow', 0);
        $noarchive = get_option('my_seo_robots_meta_noarchive', 0);
        $noimageindex = get_option('my_seo_robots_meta_noimageindex', 0);
        $nosnippet = get_option('my_seo_robots_meta_nosnippet', 0);
?>
        <div style="padding: 10px; background: #fff3cd; border-left: 4px solid #ffc107; margin-bottom: 15px;">
            <strong><?php esc_html_e('Important:', 'seo-fury'); ?></strong> <?php
            /* translators: Do not translate the <code> tags or their contents (Index, NoIndex). */
            printf(
                esc_html__('Do not enable %1$s and %2$s at the same time. If both are enabled, %2$s takes priority.', 'seo-fury'),
                '<code>Index</code>',
                '<code>NoIndex</code>'
            ); ?>
        </div>

        <fieldset>
            <label style="display: block; margin-bottom: 10px;">
                <input type="checkbox" name="my_seo_robots_meta_index" value="1" <?php checked($index, 1); ?>>
                <strong><?php esc_html_e('Index', 'seo-fury'); ?></strong> - <?php esc_html_e('Allow search engines to index and show these pages in results', 'seo-fury'); ?>
            </label>

            <label style="display: block; margin-bottom: 10px;">
                <input type="checkbox" name="my_seo_robots_meta_noindex" value="1" <?php checked($noindex, 1); ?>>
                <strong><?php esc_html_e('NoIndex', 'seo-fury'); ?></strong> - <?php esc_html_e('Prevent indexing and showing pages in search results', 'seo-fury'); ?>
            </label>

            <label style="display: block; margin-bottom: 10px;">
                <input type="checkbox" name="my_seo_robots_meta_nofollow" value="1" <?php checked($nofollow, 1); ?>>
                <strong><?php esc_html_e('NoFollow', 'seo-fury'); ?></strong> - <?php esc_html_e('Prevent search engines from following links on the page', 'seo-fury'); ?>
            </label>

            <label style="display: block; margin-bottom: 10px;">
                <input type="checkbox" name="my_seo_robots_meta_noarchive" value="1" <?php checked($noarchive, 1); ?>>
                <strong><?php esc_html_e('NoArchive', 'seo-fury'); ?></strong> - <?php esc_html_e('Prevent cached links in search results', 'seo-fury'); ?>
            </label>

            <label style="display: block; margin-bottom: 10px;">
                <input type="checkbox" name="my_seo_robots_meta_noimageindex" value="1" <?php checked($noimageindex, 1); ?>>
                <strong><?php esc_html_e('NoImageIndex', 'seo-fury'); ?></strong> - <?php esc_html_e('Prevent indexing page images by search engines', 'seo-fury'); ?>
            </label>

            <label style="display: block; margin-bottom: 10px;">
                <input type="checkbox" name="my_seo_robots_meta_nosnippet" value="1" <?php checked($nosnippet, 1); ?>>
                <strong><?php esc_html_e('NoSnippet', 'seo-fury'); ?></strong> - <?php esc_html_e('Prevents displaying snippets in search results', 'seo-fury'); ?>
            </label>
        </fieldset>
    <?php
    }


    public function advanced_robots_section_callback()
    {
        echo '<p>' . esc_html__('Additional settings to control content appearance in search results.', 'seo-fury') . '</p>';
    }

    public function robots_snippet_callback()
    {
        $value = get_option('my_seo_robots_meta_max_snippet', -1);
    ?>
        <input type="number"
            name="my_seo_robots_meta_max_snippet"
            value="<?php echo esc_attr($value); ?>"
            min="-1" max="500"
            class="small-text">
        <span><?php esc_html_e('characters', 'seo-fury'); ?></span>

        <p class="description">
            <?php esc_html_e('Set the maximum snippet length in characters.', 'seo-fury'); ?><br>
            <code>-1</code> <?php esc_html_e('means unlimited.', 'seo-fury'); ?><br>
            <code>0</code> <?php esc_html_e('disables snippets.', 'seo-fury'); ?>
        </p>
    <?php
    }



    public function robots_video_callback()
    {
        $value = get_option('my_seo_robots_meta_max_video_preview', -1);
    ?>
        <input type="number"
            name="my_seo_robots_meta_max_video_preview"
            value="<?php echo esc_attr($value); ?>"
            min="-1" max="999"
            class="small-text">
        <span><?php esc_html_e('seconds', 'seo-fury'); ?></span>

        <p class="description">
            <?php esc_html_e('Set the maximum video preview length in seconds.', 'seo-fury'); ?><br>
            <code>-1</code> <?php esc_html_e('means unlimited.', 'seo-fury'); ?><br>
            <code>0</code> <?php esc_html_e('disables video previews.', 'seo-fury'); ?>
        </p>
    <?php
    }


    public function robots_image_callback()
    {
        $value = get_option('my_seo_robots_meta_max_image_preview', 'standard');
    ?>
        <select name="my_seo_robots_meta_max_image_preview">
            <option value="none" <?php selected($value, 'none'); ?>><?php esc_html_e('None', 'seo-fury'); ?></option>
            <option value="standard" <?php selected($value, 'standard'); ?>><?php esc_html_e('Standard', 'seo-fury'); ?></option>
            <option value="large" <?php selected($value, 'large'); ?>><?php esc_html_e('Large', 'seo-fury'); ?></option>
        </select>

        <p class="description">
            <?php esc_html_e('Set the maximum image preview size for images on this page.', 'seo-fury'); ?><br>
            <code>None</code> <?php esc_html_e('disables image previews.', 'seo-fury'); ?><br>
            <code>Standard</code> / <code>Large</code> <?php esc_html_e('are standard limits.', 'seo-fury'); ?>
        </p>
    <?php
    }


    public function noindex_archives_section_callback()
    {
        echo '<p>' . esc_html__('Adding noindex for empty archives helps avoid indexing empty pages.', 'seo-fury') . '</p>';
    }

    public function noindex_empty_callback()
    {
        $value = get_option('my_seo_noindex_empty_archives', 0);
    ?>
        <label>
            <input type="checkbox" name="my_seo_noindex_empty_archives" value="1" <?php checked($value, 1); ?>>
            <?php
            /* translators: %s is the <code>noindex</code> HTML tag, do not translate. */
            printf(esc_html__('Add %s for empty category and tag archives', 'seo-fury'), '<code>noindex</code>'); ?>
        </label>
        <p class="description">
            <?php
            /* translators: %s is the <code>index</code> HTML tag, do not translate. */
            printf(esc_html__('When a post is added, the archive switches back to %s.', 'seo-fury'), '<code>index</code>'); ?>
        </p>
    <?php
    }

    public function title_settings_section_callback()
    {
        echo '<p>' . esc_html__('Configure how page titles are displayed on your site.', 'seo-fury') . '</p>';
    }

    public function title_separator_callback()
    {
        $separator = get_option('my_seo_title_separator', '-');
        $separators = array(
            '-' => '-',
            '–' => '–',
            '—' => '—',
            '»' => '»',
            '|' => '|',
            '•' => '•'
        );
    ?>
        <select name="my_seo_title_separator" id="title-separator-select">
            <?php foreach ($separators as $value => $label): ?>
                <option value="<?php echo esc_attr($value); ?>" <?php selected($separator, $value); ?>>
                    <?php echo esc_html($label); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <p class="description">
            <?php
            /* translators: %1$s and %2$s are template variable names wrapped in <code> tags, do not translate. */
            printf(esc_html__('Use the separator in title templates with %1$s or %2$s.', 'seo-fury'), '<code>%separator%</code>', '<code>%sep%</code>'); ?>
        </p>

        <div style="margin-top: 15px; padding: 15px; background: #f5f5f5; border-left: 4px solid #0073aa;">
            <strong><?php esc_html_e('Preview:', 'seo-fury'); ?></strong>
            <div id="separator-preview" style="margin-top: 10px; font-size: 16px;">
                <?php esc_html_e('Page title', 'seo-fury'); ?> <span id="preview-sep"><?php echo esc_html($separator); ?></span> <?php esc_html_e('Site name', 'seo-fury'); ?>
            </div>
        </div>

        <script>
            jQuery(document).ready(function($) {
                $('#title-separator-select').on('change', function() {
                    var sep = $(this).val();
                    $('#preview-sep').text(sep);
                });
            });
        </script>
    <?php
    }

    public function title_capitalize_callback()
    {
        $value = get_option('my_seo_title_capitalize', 0);
    ?>
        <label>
            <input type="checkbox" name="my_seo_title_capitalize" value="1" <?php checked($value, 1); ?>>
            <?php esc_html_e('Automatically capitalize each word in titles', 'seo-fury'); ?>
        </label>
        <p class="description">
            <?php
            /* translators: %1$s is the before example, %2$s is the after example. */
            printf(esc_html__('Example: "%1$s" → "%2$s"', 'seo-fury'), 'page title example', 'Page Title Example'); ?>
        </p>
    <?php
    }

    public function twitter_card_section_callback()
    {
        echo '<p>' . esc_html__('Configure the Twitter card type for your pages.', 'seo-fury') . '</p>';
    }

    public function twitter_card_type_callback()
    {
        $value = get_option('my_seo_twitter_card_type', 'summary_large_image');
    ?>
        <select name="my_seo_twitter_card_type">
            <option value="summary_large_image" <?php selected($value, 'summary_large_image'); ?>>
                <?php esc_html_e('Summary with large image', 'seo-fury'); ?>
            </option>
            <option value="summary" <?php selected($value, 'summary'); ?>>
                <?php esc_html_e('Summary', 'seo-fury'); ?>
            </option>
        </select>

        <p class="description">
            <?php esc_html_e('This card type is used for new posts and for posts without a selected card type.', 'seo-fury'); ?>
        </p>
<?php
    }

    public function content_types_section_callback()
    {
        echo '<p>' . esc_html__('Set default title and description templates for each content type.', 'seo-fury') . '</p>';
        echo '<p class="description">' . esc_html__('Available variables:', 'seo-fury') . ' <code>%title%</code>, <code>%excerpt%</code>, <code>%sitename%</code>, <code>%sep%</code></p>';
    }

    public function content_types_templates_callback()
    {
        $post_types = get_post_types(array('public' => true), 'objects');
        ?>
        <table class="widefat striped" style="max-width: 100%;">
            <thead>
                <tr>
                    <th><?php esc_html_e('Content Type', 'seo-fury'); ?></th>
                    <th><?php esc_html_e('Title Template', 'seo-fury'); ?></th>
                    <th><?php esc_html_e('Description Template', 'seo-fury'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($post_types as $post_type): ?>
                    <?php
                    $title_key = 'my_seo_title_template_' . $post_type->name;
                    $desc_key = 'my_seo_description_template_' . $post_type->name;
                    $title_val = get_option($title_key, '%title% %sep% %sitename%');
                    $desc_val = get_option($desc_key, '%excerpt%');
                    ?>
                    <tr>
                        <td><?php echo esc_html($post_type->label); ?> <span class="description">(<?php echo esc_html($post_type->name); ?>)</span></td>
                        <td>
                            <input type="text" name="<?php echo esc_attr($title_key); ?>" value="<?php echo esc_attr($title_val); ?>" class="regular-text" />
                        </td>
                        <td>
                            <input type="text" name="<?php echo esc_attr($desc_key); ?>" value="<?php echo esc_attr($desc_val); ?>" class="regular-text" />
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php
    }

    private function get_available_variables_note() {
        return esc_html__('Available variables:', 'seo-fury') . ' <code>%title%</code>, <code>%excerpt%</code>, <code>%sitename%</code>, <code>%sitedesc%</code>, <code>%sep%</code>';
    }

    private function render_template_pair($title_key, $desc_key, $title_label, $desc_label, $default_title = '', $default_desc = '')
    {
        $title_val = get_option($title_key, $default_title);
        $desc_val = get_option($desc_key, $default_desc);
        ?>
        <div class="seo-meta-template-row">
            <div class="seo-meta-template-field">
                <label for="<?php echo esc_attr($title_key); ?>"><?php echo esc_html($title_label); ?></label>
                <input type="text" id="<?php echo esc_attr($title_key); ?>" name="<?php echo esc_attr($title_key); ?>" value="<?php echo esc_attr($title_val); ?>" class="regular-text" />
            </div>
            <div class="seo-meta-template-field">
                <label for="<?php echo esc_attr($desc_key); ?>"><?php echo esc_html($desc_label); ?></label>
                <input type="text" id="<?php echo esc_attr($desc_key); ?>" name="<?php echo esc_attr($desc_key); ?>" value="<?php echo esc_attr($desc_val); ?>" class="regular-text" />
            </div>
        </div>
        <?php
    }

    public function render_settings_page()
    {
        $post_types = get_post_types(array('public' => true), 'objects');
        $taxonomies = get_taxonomies(array('public' => true), 'objects');
        ?>
        <form method="post" action="options.php">
            <?php settings_fields('my_seo_meta_group'); ?>
            <div class="seo-meta-titles-layout">
                <div class="seo-meta-titles-nav">
                    <button type="button" class="seo-meta-nav-item active" data-panel="general">
                        <span class="dashicons dashicons-admin-generic"></span>
                        <?php esc_html_e('General Meta', 'seo-fury'); ?>
                    </button>
                    <button type="button" class="seo-meta-nav-item" data-panel="home">
                        <span class="dashicons dashicons-admin-home"></span>
                        <?php esc_html_e('Home', 'seo-fury'); ?>
                    </button>
                    <button type="button" class="seo-meta-nav-item" data-panel="authors">
                        <span class="dashicons dashicons-admin-users"></span>
                        <?php esc_html_e('Authors', 'seo-fury'); ?>
                    </button>
                    <button type="button" class="seo-meta-nav-item" data-panel="misc">
                        <span class="dashicons dashicons-admin-page"></span>
                        <?php esc_html_e('Misc Pages', 'seo-fury'); ?>
                    </button>
                    <?php if (!empty($taxonomies)): ?>
                        <div class="seo-meta-nav-heading"><?php esc_html_e('Taxonomies', 'seo-fury'); ?></div>
                        <?php foreach ($taxonomies as $taxonomy): ?>
                            <button type="button" class="seo-meta-nav-item" data-panel="tax-<?php echo esc_attr($taxonomy->name); ?>">
                                <span class="dashicons dashicons-tag"></span>
                                <?php echo esc_html($taxonomy->label); ?>
                            </button>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    <div class="seo-meta-nav-heading"><?php esc_html_e('Post Types', 'seo-fury'); ?></div>
                    <?php foreach ($post_types as $post_type): ?>
                        <button type="button" class="seo-meta-nav-item" data-panel="post-<?php echo esc_attr($post_type->name); ?>">
                            <span class="dashicons dashicons-media-document"></span>
                            <?php echo esc_html($post_type->label); ?>
                        </button>
                    <?php endforeach; ?>
                </div>
                <div class="seo-meta-titles-panels">
                    <div class="seo-meta-panel active" data-panel="general">
                        <h3><?php esc_html_e('General Meta', 'seo-fury'); ?></h3>
                        <p class="description"><?php echo $this->get_available_variables_note(); ?></p>
                        <div class="seo-meta-section">
                            <h4><?php esc_html_e('Robots Meta', 'seo-fury'); ?></h4>
                            <?php $this->robots_meta_tags_callback(); ?>
                        </div>
                        <div class="seo-meta-section">
                            <h4><?php esc_html_e('Advanced Robots Meta', 'seo-fury'); ?></h4>
                            <?php $this->robots_snippet_callback(); ?>
                            <?php $this->robots_video_callback(); ?>
                            <?php $this->robots_image_callback(); ?>
                        </div>
                        <div class="seo-meta-section">
                            <h4><?php esc_html_e('Noindex Empty Archives', 'seo-fury'); ?></h4>
                            <?php $this->noindex_empty_callback(); ?>
                        </div>
                        <div class="seo-meta-section">
                            <h4><?php esc_html_e('Title Settings', 'seo-fury'); ?></h4>
                            <?php $this->title_separator_callback(); ?>
                            <?php $this->title_capitalize_callback(); ?>
                        </div>
                        <div class="seo-meta-section">
                            <h4><?php esc_html_e('Twitter Card', 'seo-fury'); ?></h4>
                            <?php $this->twitter_card_type_callback(); ?>
                        </div>
                    </div>

                    <div class="seo-meta-panel" data-panel="home">
                        <h3><?php esc_html_e('Home', 'seo-fury'); ?></h3>
                        <p class="description"><?php echo $this->get_available_variables_note(); ?></p>
                        <?php $this->render_template_pair('my_seo_title_template_home', 'my_seo_description_template_home', __('Homepage Title', 'seo-fury'), __('Homepage Description', 'seo-fury'), '%sitename% %sep% %sitedesc%', '%sitedesc%'); ?>
                    </div>

                    <div class="seo-meta-panel" data-panel="authors">
                        <h3><?php esc_html_e('Authors', 'seo-fury'); ?></h3>
                        <p class="description"><?php echo $this->get_available_variables_note(); ?></p>
                        <?php $this->render_template_pair('my_seo_title_template_author', 'my_seo_description_template_author', __('Author Archive Title', 'seo-fury'), __('Author Archive Description', 'seo-fury'), '%title% %sep% %sitename%', '%excerpt%'); ?>
                    </div>

                    <div class="seo-meta-panel" data-panel="misc">
                        <h3><?php esc_html_e('Misc Pages', 'seo-fury'); ?></h3>
                        <p class="description"><?php echo $this->get_available_variables_note(); ?></p>
                        <?php $this->render_template_pair('my_seo_title_template_blog', 'my_seo_description_template_blog', __('Blog Page Title', 'seo-fury'), __('Blog Page Description', 'seo-fury'), '%title% %sep% %sitename%', '%excerpt%'); ?>
                        <?php $this->render_template_pair('my_seo_title_template_search', 'my_seo_description_template_search', __('Search Results Title', 'seo-fury'), __('Search Results Description', 'seo-fury'), 'Search Results for %title% %sep% %sitename%', 'Search results for %title%'); ?>
                        <?php $this->render_template_pair('my_seo_title_template_date', 'my_seo_description_template_date', __('Date Archive Title', 'seo-fury'), __('Date Archive Description', 'seo-fury'), '%title% %sep% %sitename%', '%excerpt%'); ?>
                        <?php $this->render_template_pair('my_seo_title_template_404', 'my_seo_description_template_404', __('404 Title', 'seo-fury'), __('404 Description', 'seo-fury'), 'Page Not Found %sep% %sitename%', 'The page you are looking for could not be found.'); ?>
                    </div>

                    <?php foreach ($taxonomies as $taxonomy): ?>
                        <div class="seo-meta-panel" data-panel="tax-<?php echo esc_attr($taxonomy->name); ?>">
                            <h3><?php echo esc_html($taxonomy->label); ?></h3>
                            <p class="description"><?php echo $this->get_available_variables_note(); ?></p>
                            <?php
                            $title_key = 'my_seo_title_template_tax_' . $taxonomy->name;
                            $desc_key = 'my_seo_description_template_tax_' . $taxonomy->name;
                            /* translators: %s is the taxonomy name (e.g. Categories, Tags). */
                            $this->render_template_pair($title_key, $desc_key, sprintf(__('%s Title', 'seo-fury'), $taxonomy->label), sprintf(__('%s Description', 'seo-fury'), $taxonomy->label), '%title% %sep% %sitename%', '%excerpt%');
                            ?>
                        </div>
                    <?php endforeach; ?>

                    <?php foreach ($post_types as $post_type): ?>
                        <div class="seo-meta-panel" data-panel="post-<?php echo esc_attr($post_type->name); ?>">
                            <h3><?php echo esc_html($post_type->label); ?></h3>
                            <p class="description"><?php echo $this->get_available_variables_note(); ?></p>
                            <?php
                            $title_key = 'my_seo_title_template_' . $post_type->name;
                            $desc_key = 'my_seo_description_template_' . $post_type->name;
                            /* translators: %s is the post type name (e.g. Posts, Pages). */
                            $this->render_template_pair($title_key, $desc_key, sprintf(__('%s Title', 'seo-fury'), $post_type->label), sprintf(__('%s Description', 'seo-fury'), $post_type->label), '%title% %sep% %sitename%', '%excerpt%');
                            ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php submit_button(); ?>
        </form>

        <script>
            jQuery(document).ready(function($) {
                $('.seo-meta-nav-item').on('click', function() {
                    var target = $(this).data('panel');
                    $('.seo-meta-nav-item').removeClass('active');
                    $(this).addClass('active');
                    $('.seo-meta-panel').removeClass('active');
                    $('.seo-meta-panel[data-panel="' + target + '"]').addClass('active');
                });
            });
        </script>
        <?php
    }


    public function output_meta_tags()
    {
        remove_action('wp_head', 'noindex', 1);
        remove_action('wp_head', 'wp_no_robots');

        error_log('=== META-TITLES DEBUG ===');
        error_log('is_singular: ' . (is_singular() ? 'YES' : 'NO'));

        if (is_singular()) {
            $current_post = get_post();
            error_log('Current post ID: ' . ($current_post ? $current_post->ID : 'NULL'));

            if ($current_post && isset($current_post->ID)) {
                $has_custom_robots = metadata_exists('post', $current_post->ID, '_my_seo_page_robots_noindex')
                    || metadata_exists('post', $current_post->ID, '_my_seo_page_robots_nofollow')
                    || metadata_exists('post', $current_post->ID, '_my_seo_page_robots_noarchive')
                    || metadata_exists('post', $current_post->ID, '_my_seo_page_robots_nosnippet')
                    || metadata_exists('post', $current_post->ID, '_my_seo_page_robots_noimageindex');

                error_log('has_custom_robots: ' . ($has_custom_robots ? 'YES' : 'NO'));

                error_log('noindex exists: ' . (metadata_exists('post', $current_post->ID, '_my_seo_page_robots_noindex') ? 'YES' : 'NO'));
                error_log('nofollow exists: ' . (metadata_exists('post', $current_post->ID, '_my_seo_page_robots_nofollow') ? 'YES' : 'NO'));
                error_log('noarchive exists: ' . (metadata_exists('post', $current_post->ID, '_my_seo_page_robots_noarchive') ? 'YES' : 'NO'));

                if ($has_custom_robots) {
                    error_log('SKIPPING global robots - using page-specific settings');
                    $twitter_card_type = get_option('my_seo_twitter_card_type', 'summary_large_image');
                    echo '<meta name="twitter:card" content="' . esc_attr($twitter_card_type) . '">' . "\n";
                    return;
                } else {
                    error_log('NO custom robots found - will output global settings');
                }
            }
        }

        error_log('Outputting GLOBAL robots meta tags');

        $robots_parts = array();

        $index_set = get_option('my_seo_robots_meta_index', 0);
        $noindex_set = get_option('my_seo_robots_meta_noindex', 0);

        if ($noindex_set) {
            $robots_parts[] = 'noindex';
        } elseif ($index_set) {
            $robots_parts[] = 'index';
        }

        if (get_option('my_seo_robots_meta_nofollow', 0)) {
            $robots_parts[] = 'nofollow';
        } else {
            $robots_parts[] = 'follow';
        }

        if (get_option('my_seo_robots_meta_noarchive', 0)) {
            $robots_parts[] = 'noarchive';
        }
        if (get_option('my_seo_robots_meta_noimageindex', 0)) {
            $robots_parts[] = 'noimageindex';
        }
        if (get_option('my_seo_robots_meta_nosnippet', 0)) {
            $robots_parts[] = 'nosnippet';
        }

        if (get_option('my_seo_noindex_empty_archives', 0)) {
            if ((is_category() || is_tag() || is_tax()) && !have_posts()) {
                $robots_parts = array_diff($robots_parts, array('index'));
                if (!in_array('noindex', $robots_parts)) {
                    $robots_parts[] = 'noindex';
                }
            }
        }

        $max_snippet = get_option('my_seo_robots_meta_max_snippet', -1);
        if ($max_snippet !== '') {
            $robots_parts[] = 'max-snippet:' . intval($max_snippet);
        }

        $max_video_preview = get_option('my_seo_robots_meta_max_video_preview', -1);
        if ($max_video_preview !== '') {
            $robots_parts[] = 'max-video-preview:' . intval($max_video_preview);
        }

        $max_image_preview = get_option('my_seo_robots_meta_max_image_preview', 'standard');
        if ($max_image_preview) {
            $robots_parts[] = 'max-image-preview:' . $max_image_preview;
        }

        if (!empty($robots_parts)) {
            echo '<meta name="robots" content="' . esc_attr(implode(', ', $robots_parts)) . '">' . "\n";
        }

        if (!is_singular()) {
            $desc_template = '';
            if (is_front_page()) {
                $desc_template = get_option('my_seo_description_template_home', '');
            } elseif (is_home()) {
                $desc_template = get_option('my_seo_description_template_blog', '');
            } elseif (is_author()) {
                $desc_template = get_option('my_seo_description_template_author', '');
            } elseif (is_search()) {
                $desc_template = get_option('my_seo_description_template_search', '');
            } elseif (is_404()) {
                $desc_template = get_option('my_seo_description_template_404', '');
            } elseif (is_date()) {
                $desc_template = get_option('my_seo_description_template_date', '');
            } elseif (is_category() || is_tag() || is_tax()) {
                $term = get_queried_object();
                if ($term && !empty($term->taxonomy)) {
                    $desc_template = get_option('my_seo_description_template_tax_' . $term->taxonomy, '');
                }
            } elseif (is_post_type_archive()) {
                $post_type = get_query_var('post_type');
                if ($post_type) {
                    $desc_template = get_option('my_seo_description_template_' . $post_type, '');
                }
            }

            if ($desc_template !== '') {
                $description = $this->apply_template($desc_template, $this->get_context_replacements());
                if ($description !== '') {
                    echo '<meta name="description" content="' . esc_attr($description) . '">' . "\n";
                }
            }
        }

        $twitter_card_type = get_option('my_seo_twitter_card_type', 'summary_large_image');
        echo '<meta name="twitter:card" content="' . esc_attr($twitter_card_type) . '">' . "\n";
    }

    private function apply_template($template, array $replacements) {
        $separator = get_option('my_seo_title_separator', '—');
        $defaults = array(
            '%sitename%' => get_bloginfo('name'),
            '%sitedesc%' => get_bloginfo('description'),
            '%sep%' => $separator,
            '%separator%' => $separator,
            '%title%' => '',
            '%excerpt%' => '',
        );

        return strtr($template, array_merge($defaults, $replacements));
    }

    private function get_post_replacements($post) {
        $content = $post ? $post->post_content : '';
        $content = strip_shortcodes($content);
        $content = wp_strip_all_tags($content);
        $excerpt = wp_trim_words($content, 20, '...');

        return array(
            '%title%' => $post ? get_the_title($post) : '',
            '%excerpt%' => $excerpt,
        );
    }

    private function get_context_replacements() {
        if (is_front_page()) {
            return array(
                '%title%' => get_bloginfo('name'),
                '%excerpt%' => get_bloginfo('description')
            );
        }

        if (is_home()) {
            $title = get_the_title(get_option('page_for_posts')) ?: get_bloginfo('name');
            return array(
                '%title%' => $title,
                '%excerpt%' => get_bloginfo('description')
            );
        }

        if (is_author()) {
            $author = get_queried_object();
            return array(
                '%title%' => $author ? $author->display_name : '',
                '%excerpt%' => $author ? get_the_author_meta('description', $author->ID) : ''
            );
        }

        if (is_search()) {
            return array(
                '%title%' => get_search_query(),
                '%excerpt%' => ''
            );
        }

        if (is_404()) {
            return array(
                '%title%' => __('Page Not Found', 'seo-fury'),
                '%excerpt%' => ''
            );
        }

        if (is_date()) {
            return array(
                '%title%' => get_the_archive_title(),
                '%excerpt%' => get_the_archive_description()
            );
        }

        if (is_category() || is_tag() || is_tax()) {
            $term = get_queried_object();
            return array(
                '%title%' => $term ? $term->name : '',
                '%excerpt%' => $term ? wp_strip_all_tags(term_description($term->term_id, $term->taxonomy)) : ''
            );
        }

        return array(
            '%title%' => get_the_archive_title(),
            '%excerpt%' => get_the_archive_description()
        );
    }



    public function modify_title($title, $sep, $seplocation)
    {
        if (is_admin() || wp_doing_ajax()) {
            return $title;
        }

        if (defined('REST_REQUEST') && REST_REQUEST) {
            return $title;
        }

        if (strpos($_SERVER['REQUEST_URI'], '/wp-json/') !== false) {
            return $title;
        }

        $separator  = get_option('my_seo_title_separator', '—');
        $capitalize = get_option('my_seo_title_capitalize', 0);

        $title = str_replace($sep, $separator, $title);

        if ($capitalize) {
            $title = mb_convert_case($title, MB_CASE_TITLE, 'UTF-8');
        }

        return $title;
    }


    public function modify_document_title_parts($title_parts)
    {
        if (is_admin() || wp_doing_ajax()) {
            return $title_parts;
        }

        if (defined('REST_REQUEST') && REST_REQUEST) {
            return $title_parts;
        }

        if (strpos($_SERVER['REQUEST_URI'], '/wp-json/') !== false) {
            return $title_parts;
        }

        $separator  = get_option('my_seo_title_separator', '—');
        $capitalize = get_option('my_seo_title_capitalize', 0);

        $custom_title = '';

        if (is_front_page()) {
            $template = get_option('my_seo_title_template_home', '');
            if ($template) {
                $custom_title = $this->apply_template($template, $this->get_context_replacements());
            }
        } elseif (is_home()) {
            $template = get_option('my_seo_title_template_blog', '');
            if ($template) {
                $custom_title = $this->apply_template($template, $this->get_context_replacements());
            }
        } elseif (is_author()) {
            $template = get_option('my_seo_title_template_author', '');
            if ($template) {
                $custom_title = $this->apply_template($template, $this->get_context_replacements());
            }
        } elseif (is_search()) {
            $template = get_option('my_seo_title_template_search', '');
            if ($template) {
                $custom_title = $this->apply_template($template, $this->get_context_replacements());
            }
        } elseif (is_404()) {
            $template = get_option('my_seo_title_template_404', '');
            if ($template) {
                $custom_title = $this->apply_template($template, $this->get_context_replacements());
            }
        } elseif (is_date()) {
            $template = get_option('my_seo_title_template_date', '');
            if ($template) {
                $custom_title = $this->apply_template($template, $this->get_context_replacements());
            }
        } elseif (is_category() || is_tag() || is_tax()) {
            $term = get_queried_object();
            if ($term && !empty($term->taxonomy)) {
                $template = get_option('my_seo_title_template_tax_' . $term->taxonomy, '');
                if ($template) {
                    $custom_title = $this->apply_template($template, $this->get_context_replacements());
                }
            }
        } elseif (is_post_type_archive()) {
            $post_type = get_query_var('post_type');
            if ($post_type) {
                $template = get_option('my_seo_title_template_' . $post_type, '');
                if ($template) {
                    $custom_title = $this->apply_template($template, $this->get_context_replacements());
                }
            }
        } elseif (is_singular()) {
            $post = get_post();
            if ($post) {
                $meta_title = get_post_meta($post->ID, '_my_seo_title', true);
                if (!$meta_title) {
                    $meta_title = get_option('my_seo_title_template_' . $post->post_type, '');
                }
                if (!empty($meta_title)) {
                    $custom_title = $this->apply_template($meta_title, $this->get_post_replacements($post));
                }
            }
        }

        if ($custom_title !== '') {
            if ($capitalize) {
                $custom_title = mb_convert_case($custom_title, MB_CASE_TITLE, 'UTF-8');
            }
            return array('title' => $custom_title);
        }

        $site = isset($title_parts['site']) ? $title_parts['site'] : get_bloginfo('name');
        $page = isset($title_parts['title']) ? $title_parts['title'] : '';

        if ($page !== '') {
            $title = $page . ' ' . $separator . ' ' . $site;
        } else {
            $title = $site;
        }

        if ($capitalize) {
            $title = mb_convert_case($title, MB_CASE_TITLE, 'UTF-8');
        }

        return array(
            'title' => $title,
        );
    }
}
