<?php

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Fury_Post_Meta_Internal_Links {

    public function __construct() {
        add_action('wp_ajax_seo_fury_scan_internal_links', array($this, 'ajax_scan'));
    }

    public function render_tab($post) {
        $core = SEO_Fury_Core::get_instance();
        $pro = $core->get_pro_features();
        $is_pro = $pro && $pro->is_pro();
        $hide_branding = function_exists('seo_fury_hide_branding') ? seo_fury_hide_branding() : false;

        if (!$is_pro) {
            ?>
            <div style="padding: 20px; text-align: center;">
                <div style="background: var(--sf-primary-light, #EEF2FF); border-radius: 12px; padding: 30px 20px;">
                    <div style="font-size: 36px; margin-bottom: 10px;">🔗</div>
                    <h3 style="margin: 0 0 8px; color: var(--sf-gray-900, #1E293B);">Internal Linking Suggestions</h3>
                    <p style="color: var(--sf-gray-500, #64748B); font-size: 13px; margin: 0 0 15px;">
                        Scan your content to find internal linking opportunities. Get smart suggestions based on your existing pages, posts, and products.
                    </p>
                    <span style="background: linear-gradient(135deg, #4F46E5, #7C3AED); color: #fff; font-size: 11px; padding: 4px 12px; border-radius: 12px; font-weight: 600;">PRO Feature</span>
                    <?php if (!$hide_branding): ?>
                        <br><br>
                        <a href="https://seo-fury.com/#pricing" target="_blank" style="color: var(--sf-primary, #4F46E5); font-weight: 600; text-decoration: none;">Upgrade to PRO &rarr;</a>
                    <?php endif; ?>
                </div>
            </div>
            <?php
            return;
        }
        ?>
        <div class="seo-fury-internal-links" style="padding: 12px 0;">
            <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 12px;">
                <h4 style="margin: 0; font-size: 13px; font-weight: 600; color: var(--sf-gray-900, #1E293B);">Internal Linking</h4>
                <button type="button" id="seo-fury-scan-links" class="button button-small" style="font-size: 12px;">Scan Content</button>
            </div>

            <div id="seo-fury-link-stats" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 8px; margin-bottom: 15px;">
                <div style="background: var(--sf-primary-light, #EEF2FF); border-radius: 8px; padding: 10px; text-align: center;">
                    <div id="sf-stat-internal" style="font-size: 20px; font-weight: 700; color: var(--sf-primary, #4F46E5);">—</div>
                    <div style="font-size: 10px; color: var(--sf-gray-500, #64748B);">Internal</div>
                </div>
                <div style="background: var(--sf-success-light, #ECFDF5); border-radius: 8px; padding: 10px; text-align: center;">
                    <div id="sf-stat-external" style="font-size: 20px; font-weight: 700; color: var(--sf-success, #10B981);">—</div>
                    <div style="font-size: 10px; color: var(--sf-gray-500, #64748B);">External</div>
                </div>
                <div style="background: var(--sf-accent-light, #FEF3C7); border-radius: 8px; padding: 10px; text-align: center;">
                    <div id="sf-stat-suggestions" style="font-size: 20px; font-weight: 700; color: var(--sf-accent, #F59E0B);">—</div>
                    <div style="font-size: 10px; color: var(--sf-gray-500, #64748B);">Suggestions</div>
                </div>
            </div>

            <div id="seo-fury-link-suggestions" style="display: none;">
                <h5 style="margin: 0 0 8px; font-size: 12px; font-weight: 600; color: var(--sf-gray-700, #334155);">Suggested Links</h5>
                <div id="seo-fury-suggestions-list" style="max-height: 300px; overflow-y: auto;"></div>
            </div>

            <div id="seo-fury-link-scan-empty" style="display: none; text-align: center; padding: 20px; color: var(--sf-gray-500, #64748B);">
                <div style="font-size: 24px; margin-bottom: 8px;">✅</div>
                <p style="margin: 0; font-size: 13px;">No linking opportunities found. Your content looks well-linked!</p>
            </div>

            <div id="seo-fury-link-loading" style="display: none; text-align: center; padding: 20px;">
                <div style="font-size: 24px; margin-bottom: 8px;">⏳</div>
                <p style="margin: 0; font-size: 13px; color: var(--sf-gray-500, #64748B);">Scanning content...</p>
            </div>

            <div id="seo-fury-orphan-pages" style="display: none; margin-top: 15px; border-top: 1px solid var(--sf-gray-200, #E2E8F0); padding-top: 12px;">
                <h5 style="margin: 0 0 8px; font-size: 12px; font-weight: 600; color: var(--sf-gray-700, #334155);">Orphan Pages <span style="font-weight: 400; color: var(--sf-gray-400);">(no inbound links)</span></h5>
                <div id="seo-fury-orphan-list"></div>
            </div>
        </div>
        <?php
    }

    public function ajax_scan() {
        check_ajax_referer('seo_fury_internal_links_nonce', 'nonce');
        if (!current_user_can('edit_posts')) {
            wp_send_json_error('Insufficient permissions');
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        if (!$post_id) {
            wp_send_json_error('Invalid post ID');
        }

        $post = get_post($post_id);
        if (!$post) {
            wp_send_json_error('Post not found');
        }

        $content = $post->post_content;
        $plain_content = wp_strip_all_tags(strip_shortcodes($content));
        $plain_lower = mb_strtolower($plain_content, 'UTF-8');

        $link_stats = $this->get_link_stats($content);

        $all_posts = get_posts(array(
            'post_type' => get_post_types(array('public' => true), 'names'),
            'post_status' => 'publish',
            'posts_per_page' => 500,
            'exclude' => array($post_id),
            'orderby' => 'title',
            'order' => 'ASC',
        ));

        $existing_links = $this->get_existing_links($content);
        $suggestions = array();

        foreach ($all_posts as $candidate) {
            $target_url = get_permalink($candidate->ID);

            $already_linked = false;
            foreach ($existing_links as $link) {
                if (rtrim($link, '/') === rtrim($target_url, '/')) {
                    $already_linked = true;
                    break;
                }
            }
            if ($already_linked) continue;

            $title = $candidate->post_title;
            $title_lower = mb_strtolower($title, 'UTF-8');
            $title_len = mb_strlen($title_lower, 'UTF-8');

            if ($title_len < 3) continue;

            if (mb_strpos($plain_lower, $title_lower) !== false) {
                $suggestions[] = array(
                    'match_text' => $title,
                    'target_title' => $candidate->post_title,
                    'target_url' => $target_url,
                    'post_type' => $candidate->post_type,
                    'score' => 100,
                    'match_type' => 'exact',
                );
                continue;
            }

            $words = preg_split('/[\s\-_,.:;!?()]+/u', $title_lower);
            $words = array_filter($words, function($w) {
                $stop = array('the','a','an','and','or','but','in','on','at','to','for','of','is','it','by','as','с','и','в','на','по','для','из','от','до','не','что','как','это','все','его','она','они','мы','вы','об','о');
                return mb_strlen($w, 'UTF-8') > 2 && !in_array($w, $stop);
            });
            $words = array_values($words);

            if (count($words) === 1 && mb_strlen($words[0], 'UTF-8') >= 4) {
                if (preg_match('/(?<=\s|^)' . preg_quote($words[0], '/') . '(?=\s|$|[.,;:!?\)\]\-])/ui', $plain_lower)) {
                    $suggestions[] = array(
                        'match_text' => $title,
                        'target_title' => $candidate->post_title,
                        'target_url' => $target_url,
                        'post_type' => $candidate->post_type,
                        'score' => 80,
                        'match_type' => 'word',
                    );
                    continue;
                }
            }

            if (count($words) >= 2) {
                $matched = false;
                for ($i = 0; $i <= count($words) - 2; $i++) {
                    $phrase = $words[$i] . ' ' . $words[$i+1];
                    if (mb_strpos($plain_lower, $phrase) !== false) {
                        $suggestions[] = array(
                            'match_text' => $phrase,
                            'target_title' => $candidate->post_title,
                            'target_url' => $target_url,
                            'post_type' => $candidate->post_type,
                            'score' => 50 + count($words) * 5,
                            'match_type' => 'partial',
                        );
                        $matched = true;
                        break;
                    }
                }

                if (!$matched) {
                    foreach ($words as $word) {
                        if (mb_strlen($word, 'UTF-8') >= 5 && preg_match('/(?<=\s|^)' . preg_quote($word, '/') . '(?=\s|$|[.,;:!?\)\]\-])/ui', $plain_lower)) {
                            $suggestions[] = array(
                                'match_text' => $word,
                                'target_title' => $candidate->post_title,
                                'target_url' => $target_url,
                                'post_type' => $candidate->post_type,
                                'score' => 40,
                                'match_type' => 'word',
                            );
                            break;
                        }
                    }
                }
            }
        }

        usort($suggestions, function($a, $b) {
            return $b['score'] - $a['score'];
        });
        $suggestions = array_slice($suggestions, 0, 20);

        $orphan_pages = $this->get_orphan_pages($post_id, $all_posts);

        wp_send_json_success(array(
            'stats' => $link_stats,
            'suggestions' => $suggestions,
            'orphan_pages' => array_slice($orphan_pages, 0, 10),
        ));
    }

    private function get_link_stats($content) {
        $internal = 0;
        $external = 0;
        $home = home_url();

        preg_match_all('/<a\s[^>]*href=["\']([^"\']+)["\'][^>]*>/i', $content, $matches);
        if (!empty($matches[1])) {
            foreach ($matches[1] as $url) {
                if (strpos($url, '#') === 0 || strpos($url, 'mailto:') === 0 || strpos($url, 'tel:') === 0) continue;
                if (strpos($url, $home) === 0 || strpos($url, '/') === 0) {
                    $internal++;
                } else {
                    $external++;
                }
            }
        }

        return array(
            'internal' => $internal,
            'external' => $external,
            'total' => $internal + $external,
        );
    }

    private function get_existing_links($content) {
        $urls = array();
        preg_match_all('/<a\s[^>]*href=["\']([^"\']+)["\'][^>]*>/i', $content, $matches);
        if (!empty($matches[1])) {
            $urls = $matches[1];
        }
        return $urls;
    }

    private function get_orphan_pages($current_id, $all_posts) {
        global $wpdb;
        $home = home_url();
        $orphans = array();

        foreach ($all_posts as $candidate) {
            $url = get_permalink($candidate->ID);
            $relative = str_replace($home, '', $url);

            $found = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'publish' AND ID != %d AND ID != %d AND (post_content LIKE %s OR post_content LIKE %s)",
                $candidate->ID,
                $current_id,
                '%' . $wpdb->esc_like($url) . '%',
                '%' . $wpdb->esc_like($relative) . '%'
            ));

            if (intval($found) === 0) {
                $orphans[] = array(
                    'title' => $candidate->post_title,
                    'url' => $url,
                    'post_type' => $candidate->post_type,
                );
            }

            if (count($orphans) >= 10) break;
        }

        return $orphans;
    }
}
