<?php

if (!defined('ABSPATH')) exit;

class MySEO_Plugin_Post_Meta_Schema {

    private $schema_types = array(
        'none' => 'None',
        'article' => 'Article',
        'blog_posting' => 'Blog Posting',
        'news_article' => 'News Article',
        'product' => 'Product',
        'review' => 'Review',
        'recipe' => 'Recipe',
        'faq' => 'FAQ Page',
        'how_to' => 'How To',
        'breadcrumb' => 'BreadcrumbList',
        'table' => 'Table',
        'service' => 'Service',
        'job_posting' => 'Job Posting',
        'course' => 'Course',
        'event' => 'Event',
        'local_business' => 'Local Business',
        'organization' => 'Organization',
        'person' => 'Person',
        'video' => 'Video Object',
        'book' => 'Book',
        'software' => 'Software Application',
    );

    private $schema_types_translated = null;

    private $schema_icons = array(
        'article' => 'dashicons-media-document',
        'blog_posting' => 'dashicons-welcome-write-blog',
        'news_article' => 'dashicons-megaphone',
        'product' => 'dashicons-cart',
        'review' => 'dashicons-star-filled',
        'recipe' => 'dashicons-carrot',
        'faq' => 'dashicons-editor-help',
        'how_to' => 'dashicons-welcome-learn-more',
        'breadcrumb' => 'dashicons-menu',
        'table' => 'dashicons-editor-table',
        'service' => 'dashicons-hammer',
        'job_posting' => 'dashicons-id-alt',
        'course' => 'dashicons-welcome-learn-more',
        'event' => 'dashicons-calendar-alt',
        'local_business' => 'dashicons-location',
        'organization' => 'dashicons-building',
        'person' => 'dashicons-admin-users',
        'video' => 'dashicons-video-alt3',
        'book' => 'dashicons-book-alt',
        'software' => 'dashicons-desktop',
        'custom' => 'dashicons-media-code',
    );

    public function __construct() {
    }

    public function get_schema_types() {
        if ($this->schema_types_translated === null) {
            $this->schema_types_translated = array(
                'none'           => __('None', 'seo-fury'),
                'article'        => __('Article', 'seo-fury'),
                'blog_posting'   => __('Blog Posting', 'seo-fury'),
                'news_article'   => __('News Article', 'seo-fury'),
                'product'        => __('Product', 'seo-fury'),
                'review'         => __('Review', 'seo-fury'),
                'recipe'         => __('Recipe', 'seo-fury'),
                'faq'            => __('FAQ Page', 'seo-fury'),
                'how_to'         => __('How To', 'seo-fury'),
                'breadcrumb'     => __('BreadcrumbList', 'seo-fury'),
                'table'          => __('Table', 'seo-fury'),
                'service'        => __('Service', 'seo-fury'),
                'job_posting'    => __('Job Posting', 'seo-fury'),
                'course'         => __('Course', 'seo-fury'),
                'event'          => __('Event', 'seo-fury'),
                'local_business' => __('Local Business', 'seo-fury'),
                'organization'   => __('Organization', 'seo-fury'),
                'person'         => __('Person', 'seo-fury'),
                'video'          => __('Video Object', 'seo-fury'),
                'book'           => __('Book', 'seo-fury'),
                'software'       => __('Software Application', 'seo-fury'),
            );
        }
        return apply_filters('seo_fury_schema_types', $this->schema_types_translated);
    }

    private function get_schema_icon_class($type, $mode = 'generator') {
        if ($mode === 'custom') {
            return $this->schema_icons['custom'];
        }
        return $this->schema_icons[$type] ?? 'dashicons-tag';
    }

    public function render_schema_tab($post, $meta_data) {
        $schema_type = get_post_meta($post->ID, '_my_seo_schema_type', true) ?: 'none';
        $schema_data = get_post_meta($post->ID, '_my_seo_schema_data', true) ?: array();
        $schema_mode = get_post_meta($post->ID, '_my_seo_schema_mode', true) ?: 'generator';
        $custom_schema = get_post_meta($post->ID, '_my_seo_schema_custom', true) ?: '';

        $schema_items = get_post_meta($post->ID, '_my_seo_schema_items', true);
        if (!is_array($schema_items)) {
            $schema_items = array();
            if (!empty($custom_schema)) {
                $schema_items[] = array(
                    'id' => uniqid('schema_', true),
                    'type' => 'custom',
                    'mode' => 'custom',
                    'data' => array(),
                    'custom' => $custom_schema,
                );
            } elseif (!empty($schema_type) && $schema_type !== 'none') {
                $schema_items[] = array(
                    'id' => uniqid('schema_', true),
                    'type' => $schema_type,
                    'mode' => 'generator',
                    'data' => $schema_data,
                    'custom' => '',
                );
            }
        }

        $schema_active = get_post_meta($post->ID, '_my_seo_schema_active', true);
        if (empty($schema_active) && !empty($schema_items)) {
            $schema_active = $schema_items[0]['id'];
        }

        $active_item = null;
        foreach ($schema_items as $item) {
            if (isset($item['id']) && $item['id'] === $schema_active) {
                $active_item = $item;
                break;
            }
        }

        if (!$active_item && !empty($schema_items)) {
            $active_item = $schema_items[0];
            $schema_active = $active_item['id'];
        }

        if ($active_item) {
            $schema_mode = $active_item['mode'] ?? $schema_mode;
            $schema_type = $active_item['type'] ?? $schema_type;
            $schema_data = $active_item['data'] ?? $schema_data;
            $custom_schema = $active_item['custom'] ?? $custom_schema;
        }

        if ($schema_mode === 'custom') {
            $schema_type = 'none';
        }
        ?>
        <div class="seo-fury-tab-content" id="tab-schema">
            <div class="seo-fury-section seo-fury-schema-used">
                <h4><?php esc_html_e('Used Schema', 'seo-fury'); ?></h4>
                <input type="hidden" id="my_seo_schema_items" name="my_seo_schema_items" value="<?php echo esc_attr(wp_json_encode($schema_items)); ?>">
                <input type="hidden" id="my_seo_schema_active" name="my_seo_schema_active" value="<?php echo esc_attr($schema_active); ?>">
                <div class="seo-fury-schema-list">
                    <?php if (!empty($schema_items)) : ?>
                        <?php foreach ($schema_items as $item) : ?>
                            <?php
                            $item_id = $item['id'] ?? '';
                            $item_mode = $item['mode'] ?? 'generator';
                            $item_type = $item['type'] ?? 'none';
                            $item_label = $item_mode === 'custom' ? __('Custom JSON-LD', 'seo-fury') : ($item['label'] ?? ($this->get_schema_types()[$item_type] ?? $item_type));
                            ?>
                            <div class="seo-fury-schema-card <?php echo $item_id === $schema_active ? 'is-active' : ''; ?>" data-schema-id="<?php echo esc_attr($item_id); ?>" data-schema-mode="<?php echo esc_attr($item_mode); ?>">
                                <div class="seo-fury-schema-label">
                                    <span class="seo-fury-schema-icon dashicons <?php echo esc_attr($this->get_schema_icon_class($item_type, $item_mode)); ?>" aria-hidden="true"></span>
                                    <span class="seo-fury-schema-badge"><?php echo esc_html($item_label); ?></span>
                                    <span class="seo-fury-schema-name seo-fury-sr-only"><?php echo esc_html($item_label); ?></span>
                                </div>
                                <div class="seo-fury-schema-actions">
                                    <a href="#" class="seo-fury-icon-link seo-fury-schema-preview" data-schema-id="<?php echo esc_attr($item_id); ?>" title="<?php echo esc_attr__('Preview', 'seo-fury'); ?>" aria-label="<?php echo esc_attr__('Preview', 'seo-fury'); ?>">
                                        <span class="dashicons dashicons-visibility"></span>
                                    </a>
                                    <a href="#" class="seo-fury-icon-link seo-fury-schema-edit" data-schema-id="<?php echo esc_attr($item_id); ?>" title="<?php echo esc_attr__('Edit', 'seo-fury'); ?>" aria-label="<?php echo esc_attr__('Edit', 'seo-fury'); ?>">
                                        <span class="dashicons dashicons-edit"></span>
                                    </a>
                                    <a href="#" class="seo-fury-icon-link seo-fury-schema-remove" data-schema-id="<?php echo esc_attr($item_id); ?>" title="<?php echo esc_attr__('Remove', 'seo-fury'); ?>" aria-label="<?php echo esc_attr__('Remove', 'seo-fury'); ?>">
                                        <span class="dashicons dashicons-trash"></span>
                                    </a>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php else : ?>
                        <p class="description"><?php esc_html_e('No schema selected yet. Use Schema Generator to apply.', 'seo-fury'); ?></p>
                    <?php endif; ?>
                </div>

                <button type="button" class="button button-primary seo-fury-open-schema-modal"><?php esc_html_e('Schema Generator', 'seo-fury'); ?></button>
            </div>
        </div>

        <div id="seo-fury-schema-modal" class="seo-fury-modal" style="display: none;">
            <div class="seo-fury-modal-overlay"></div>
            <div class="seo-fury-modal-content">
                <div class="seo-fury-modal-header">
                    <h2><?php esc_html_e('Schema Generator', 'seo-fury'); ?></h2>
                    <button type="button" class="seo-fury-modal-close seo-fury-schema-modal-close">
                        <span class="dashicons dashicons-no"></span>
                    </button>
                </div>

                <div class="seo-fury-modal-body">
                    <div class="seo-fury-schema-modal-tabs">
                        <button type="button" class="seo-fury-schema-modal-tab active" data-schema-tab="templates"><?php esc_html_e('Schema Templates', 'seo-fury'); ?></button>
                        <button type="button" class="seo-fury-schema-modal-tab" data-schema-tab="import"><?php esc_html_e('Import', 'seo-fury'); ?></button>
                    </div>

                    <div class="seo-fury-schema-modal-panels">
                        <div class="seo-fury-schema-modal-panel active" data-schema-panel="templates">
                            <div class="seo-fury-section">
                                <h4><?php esc_html_e('Used Schema', 'seo-fury'); ?></h4>
                                <div class="seo-fury-schema-list">
                                    <?php if (!empty($schema_items)) : ?>
                                        <?php foreach ($schema_items as $item) : ?>
                                            <?php
                                            $item_id = $item['id'] ?? '';
                                            $item_mode = $item['mode'] ?? 'generator';
                                            $item_type = $item['type'] ?? 'none';
                                            $item_label = $item_mode === 'custom' ? __('Custom JSON-LD', 'seo-fury') : ($item['label'] ?? ($this->get_schema_types()[$item_type] ?? $item_type));
                                            ?>
                                            <div class="seo-fury-schema-card <?php echo $item_id === $schema_active ? 'is-active' : ''; ?>" data-schema-id="<?php echo esc_attr($item_id); ?>" data-schema-mode="<?php echo esc_attr($item_mode); ?>">
                                                <div class="seo-fury-schema-label">
                                                    <span class="seo-fury-schema-icon dashicons <?php echo esc_attr($this->get_schema_icon_class($item_type, $item_mode)); ?>" aria-hidden="true"></span>
                                                    <span class="seo-fury-schema-badge"><?php echo esc_html($item_label); ?></span>
                                                    <span class="seo-fury-schema-name seo-fury-sr-only"><?php echo esc_html($item_label); ?></span>
                                                </div>
                                                <div class="seo-fury-schema-actions">
                                                    <a href="#" class="seo-fury-icon-link seo-fury-schema-preview" data-schema-id="<?php echo esc_attr($item_id); ?>" title="<?php echo esc_attr__('Preview', 'seo-fury'); ?>" aria-label="<?php echo esc_attr__('Preview', 'seo-fury'); ?>">
                                                        <span class="dashicons dashicons-visibility"></span>
                                                    </a>
                                                    <a href="#" class="seo-fury-icon-link seo-fury-schema-edit" data-schema-id="<?php echo esc_attr($item_id); ?>" title="<?php echo esc_attr__('Edit', 'seo-fury'); ?>" aria-label="<?php echo esc_attr__('Edit', 'seo-fury'); ?>">
                                                        <span class="dashicons dashicons-edit"></span>
                                                    </a>
                                                    <a href="#" class="seo-fury-icon-link seo-fury-schema-remove" data-schema-id="<?php echo esc_attr($item_id); ?>" title="<?php echo esc_attr__('Remove', 'seo-fury'); ?>" aria-label="<?php echo esc_attr__('Remove', 'seo-fury'); ?>">
                                                        <span class="dashicons dashicons-trash"></span>
                                                    </a>
                                                </div>
                                            </div>
                                        <?php endforeach; ?>
                                    <?php else : ?>
                                        <p class="description"><?php esc_html_e('No schema selected. Choose a template or import JSON-LD.', 'seo-fury'); ?></p>
                                    <?php endif; ?>
                                </div>
                            </div>

                            <div class="seo-fury-section">
                                <h4><?php esc_html_e('Available Schema Types', 'seo-fury'); ?></h4>
                                <div class="seo-fury-schema-type-grid">
                                    <?php foreach ($this->get_schema_types() as $key => $label): ?>
                                        <?php if ($key === 'none') continue; ?>
                                        <div class="seo-fury-schema-type-card <?php echo $schema_type === $key ? 'is-active' : ''; ?>" data-schema-type="<?php echo esc_attr($key); ?>">
                                            <span class="seo-fury-schema-type-name"><?php echo esc_html($label); ?></span>
                                            <button type="button" class="button seo-fury-schema-apply" data-schema-type="<?php echo esc_attr($key); ?>"><?php esc_html_e('Apply', 'seo-fury'); ?></button>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>

                            <div class="seo-fury-section" id="seo-fury-schema-settings">
                                <h4><?php esc_html_e('Schema Settings', 'seo-fury'); ?></h4>
                                <input type="hidden" id="my_seo_schema_mode" name="my_seo_schema_mode" value="<?php echo esc_attr($schema_mode); ?>">
                                <select id="my_seo_schema_type" name="my_seo_schema_type" class="seo-fury-hidden">
                                    <?php foreach ($this->get_schema_types() as $key => $label): ?>
                                        <option value="<?php echo esc_attr($key); ?>" <?php selected($schema_type, $key); ?>>
                                            <?php echo esc_html($label); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>

                                <div class="seo-fury-schema-fields-wrap">
                                    <?php foreach ($this->get_schema_types() as $key => $label): ?>
                                        <?php if ($key === 'none') continue; ?>
                                        <div class="seo-fury-schema-fields" data-schema-type="<?php echo esc_attr($key); ?>" style="<?php echo $schema_type === $key ? '' : 'display:none;'; ?>">
                                            <?php $this->render_schema_fields($key, $schema_data, $post); ?>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>

                            <div class="seo-fury-section" id="schema-preview-section" style="<?php echo ($schema_mode === 'custom' || $schema_type === 'none') ? 'display:none;' : ''; ?>">
                                <h4><?php esc_html_e('Schema JSON-LD Preview', 'seo-fury'); ?></h4>
                                <div class="schema-json-preview">
                                    <pre id="schema-json-output"><?php echo esc_html($this->generate_schema_preview($schema_type, $schema_data, $post)); ?></pre>
                                </div>
                            </div>
                        </div>

                        <div class="seo-fury-schema-modal-panel" data-schema-panel="import">
                            <div class="seo-fury-section">
                                <h4><?php esc_html_e('Import JSON-LD', 'seo-fury'); ?></h4>
                                <div class="seo-fury-field">
                                    <label for="my_seo_schema_custom"><?php esc_html_e('Paste JSON-LD', 'seo-fury'); ?></label>
                                    <textarea id="my_seo_schema_custom" name="my_seo_schema_custom" rows="8" placeholder='{"@context":"https://schema.org","@type":"Article"}'><?php echo esc_textarea($custom_schema); ?></textarea>
                                    <p class="description"><?php
                                        printf(
                                            /* translators: %s: <script> HTML tag */
                                            esc_html__('Paste valid JSON-LD without the %s tag', 'seo-fury'),
                                            '<code>&lt;script&gt;</code>'
                                        );
                                    ?></p>
                                </div>
                                <button type="button" class="button button-primary seo-fury-schema-import-apply"><?php esc_html_e('Apply JSON-LD', 'seo-fury'); ?></button>
                            </div>

                            <div class="seo-fury-section" id="schema-custom-preview-section" style="<?php echo $schema_mode === 'custom' ? '' : 'display:none;'; ?>">
                                <h4><?php esc_html_e('Custom JSON-LD Preview', 'seo-fury'); ?></h4>
                                <p class="description seo-fury-schema-error" id="seo-fury-schema-error" style="display:none;"></p>
                                <div class="schema-json-preview">
                                    <pre id="schema-custom-json-output"><?php echo esc_html($this->get_custom_schema_preview($custom_schema)); ?></pre>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="seo-fury-modal-footer">
                    <button type="button" class="button button-primary seo-fury-schema-modal-close"><?php esc_html_e('Done', 'seo-fury'); ?></button>
                </div>
            </div>
        </div>
        <?php
    }

    private function render_schema_fields($schema_type, $schema_data, $post) {
        if ($schema_type === 'none') {
            return;
        }

        echo '<div id="schema-fields-container">';

        switch ($schema_type) {
            case 'article':
            case 'blog_posting':
            case 'news_article':
                $this->render_article_fields($schema_data, $post);
                break;

            case 'product':
                $this->render_product_fields($schema_data, $post);
                break;

            case 'review':
                $this->render_review_fields($schema_data, $post);
                break;

            case 'recipe':
                $this->render_recipe_fields($schema_data, $post);
                break;

            case 'faq':
                $this->render_faq_fields($schema_data, $post);
                break;

            case 'how_to':
                $this->render_howto_fields($schema_data, $post);
                break;

            case 'event':
                $this->render_event_fields($schema_data, $post);
                break;

            case 'video':
                $this->render_video_fields($schema_data, $post);
                break;

            case 'breadcrumb':
                $this->render_breadcrumb_fields($schema_data, $post);
                break;

            case 'table':
                $this->render_table_fields($schema_data, $post);
                break;

            case 'service':
                $this->render_service_fields($schema_data, $post);
                break;

            case 'job_posting':
                $this->render_jobposting_fields($schema_data, $post);
                break;

            default:
                echo '<div class="seo-fury-section">';
                echo '<p class="description">' . esc_html__('Schema fields for this type will be available soon. You can manually add schema using the JSON-LD editor above.', 'seo-fury') . '</p>';
                echo '</div>';
                break;
        }

        echo '</div>';
    }

    private function render_article_fields($data, $post) {
        ?>
        <div class="seo-fury-section">
            <h4><?php esc_html_e('Article Schema Details', 'seo-fury'); ?></h4>
            <div class="seo-fury-fields-grid">
                <div class="seo-fury-field">
                    <label for="schema_headline"><?php esc_html_e('Headline', 'seo-fury'); ?></label>
                    <input type="text" id="schema_headline" name="schema_data[headline]"
                           value="<?php echo esc_attr($data['headline'] ?? $post->post_title); ?>"
                           placeholder="<?php echo esc_attr($post->post_title); ?>">
                    <p class="description"><?php esc_html_e('Article headline (defaults to post title)', 'seo-fury'); ?></p>
                </div>

                <div class="seo-fury-field">
                    <label for="schema_author"><?php esc_html_e('Author Name', 'seo-fury'); ?></label>
                    <input type="text" id="schema_author" name="schema_data[author]"
                           value="<?php echo esc_attr($data['author'] ?? get_the_author_meta('display_name', $post->post_author)); ?>">
                    <p class="description"><?php esc_html_e('Author of the article', 'seo-fury'); ?></p>
                </div>

                <div class="seo-fury-field">
                    <label for="schema_publisher"><?php esc_html_e('Publisher Name', 'seo-fury'); ?></label>
                    <input type="text" id="schema_publisher" name="schema_data[publisher]"
                           value="<?php echo esc_attr($data['publisher'] ?? get_bloginfo('name')); ?>">
                    <p class="description"><?php esc_html_e('Publisher organization name', 'seo-fury'); ?></p>
                </div>

                <div class="seo-fury-field">
                    <label for="schema_publisher_logo"><?php esc_html_e('Publisher Logo URL', 'seo-fury'); ?></label>
                    <input type="url" id="schema_publisher_logo" name="schema_data[publisher_logo]"
                           value="<?php echo esc_url($data['publisher_logo'] ?? ''); ?>"
                           placeholder="https://example.com/logo.png">
                    <p class="description"><?php esc_html_e('URL of publisher\'s logo (recommended: 600x60px)', 'seo-fury'); ?></p>
                </div>

                <div class="seo-fury-field">
                    <label for="schema_image"><?php esc_html_e('Featured Image URL', 'seo-fury'); ?></label>
                    <input type="url" id="schema_image" name="schema_data[image]"
                           value="<?php echo esc_url($data['image'] ?? get_the_post_thumbnail_url($post->ID, 'full')); ?>">
                    <p class="description"><?php esc_html_e('Main image for the article', 'seo-fury'); ?></p>
                </div>

                <div class="seo-fury-field">
                    <label for="schema_date_published"><?php esc_html_e('Date Published', 'seo-fury'); ?></label>
                    <input type="datetime-local" id="schema_date_published" name="schema_data[date_published]"
                           value="<?php echo esc_attr($data['date_published'] ?? date('Y-m-d\TH:i', strtotime($post->post_date))); ?>">
                    <p class="description"><?php esc_html_e('When was this article published?', 'seo-fury'); ?></p>
                </div>

                <div class="seo-fury-field">
                    <label for="schema_date_modified"><?php esc_html_e('Date Modified', 'seo-fury'); ?></label>
                    <input type="datetime-local" id="schema_date_modified" name="schema_data[date_modified]"
                           value="<?php echo esc_attr($data['date_modified'] ?? date('Y-m-d\TH:i', strtotime($post->post_modified))); ?>">
                    <p class="description"><?php esc_html_e('Last modification date', 'seo-fury'); ?></p>
                </div>
            </div>
        </div>
        <?php
    }

    private function render_product_fields($data, $post) {
        ?>
        <div class="seo-fury-section">
            <h4><?php esc_html_e('Product Schema Details', 'seo-fury'); ?></h4>
            <div class="seo-fury-fields-grid">
                <div class="seo-fury-field">
                    <label for="schema_product_name"><?php esc_html_e('Product Name', 'seo-fury'); ?></label>
                    <input type="text" id="schema_product_name" name="schema_data[name]"
                           value="<?php echo esc_attr($data['name'] ?? $post->post_title); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_product_price"><?php esc_html_e('Price', 'seo-fury'); ?></label>
                    <input type="number" step="0.01" id="schema_product_price" name="schema_data[price]"
                           value="<?php echo esc_attr($data['price'] ?? ''); ?>" placeholder="99.99">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_product_currency"><?php esc_html_e('Currency', 'seo-fury'); ?></label>
                    <input type="text" id="schema_product_currency" name="schema_data[currency]"
                           value="<?php echo esc_attr($data['currency'] ?? 'USD'); ?>" placeholder="USD">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_product_availability"><?php esc_html_e('Availability', 'seo-fury'); ?></label>
                    <select id="schema_product_availability" name="schema_data[availability]">
                        <option value="InStock" <?php selected($data['availability'] ?? '', 'InStock'); ?>><?php esc_html_e('In Stock', 'seo-fury'); ?></option>
                        <option value="OutOfStock" <?php selected($data['availability'] ?? '', 'OutOfStock'); ?>><?php esc_html_e('Out of Stock', 'seo-fury'); ?></option>
                        <option value="PreOrder" <?php selected($data['availability'] ?? '', 'PreOrder'); ?>><?php esc_html_e('Pre Order', 'seo-fury'); ?></option>
                    </select>
                </div>

                <div class="seo-fury-field">
                    <label for="schema_product_brand"><?php esc_html_e('Brand', 'seo-fury'); ?></label>
                    <input type="text" id="schema_product_brand" name="schema_data[brand]"
                           value="<?php echo esc_attr($data['brand'] ?? ''); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_product_sku"><?php esc_html_e('SKU', 'seo-fury'); ?></label>
                    <input type="text" id="schema_product_sku" name="schema_data[sku]"
                           value="<?php echo esc_attr($data['sku'] ?? ''); ?>">
                </div>
            </div>
        </div>
        <?php
    }

    private function render_faq_fields($data, $post) {
        $faqs = $data['faq_items'] ?? array(array('question' => '', 'answer' => ''));
        ?>
        <div class="seo-fury-section">
            <h4><?php esc_html_e('FAQ Schema Details', 'seo-fury'); ?></h4>
            <p class="description"><?php esc_html_e('Add frequently asked questions and their answers', 'seo-fury'); ?></p>

            <div id="faq-items-container">
                <?php foreach ($faqs as $index => $faq): ?>
                    <div class="faq-item" data-index="<?php echo $index; ?>">
                        <div class="seo-fury-field">
                            <?php /* translators: %d: question number */ ?>
                            <label><?php printf(esc_html__('Question %d', 'seo-fury'), $index + 1); ?></label>
                            <input type="text" name="schema_data[faq_items][<?php echo $index; ?>][question]"
                                   value="<?php echo esc_attr($faq['question'] ?? ''); ?>"
                                   placeholder="<?php echo esc_attr__('What is your question?', 'seo-fury'); ?>">
                        </div>
                        <div class="seo-fury-field">
                            <label><?php esc_html_e('Answer', 'seo-fury'); ?></label>
                            <textarea name="schema_data[faq_items][<?php echo $index; ?>][answer]"
                                      rows="3" placeholder="<?php echo esc_attr__('Type the answer here...', 'seo-fury'); ?>"><?php echo esc_textarea($faq['answer'] ?? ''); ?></textarea>
                        </div>
                        <button type="button" class="button remove-faq-item"><?php esc_html_e('Remove', 'seo-fury'); ?></button>
                        <hr style="margin: 15px 0;">
                    </div>
                <?php endforeach; ?>
            </div>

            <button type="button" class="button button-secondary" id="add-faq-item">
                <span class="dashicons dashicons-plus-alt" style="margin-top: 3px;"></span>
                <?php esc_html_e('Add FAQ Item', 'seo-fury'); ?>
            </button>
        </div>
        <?php
    }

    private function render_howto_fields($data, $post) {
        $steps = $data['steps'] ?? array(array('name' => '', 'text' => ''));
        ?>
        <div class="seo-fury-section">
            <h4><?php esc_html_e('How-To Schema Details', 'seo-fury'); ?></h4>
            <div class="seo-fury-field">
                <label for="schema_howto_name"><?php esc_html_e('Guide Name', 'seo-fury'); ?></label>
                <input type="text" id="schema_howto_name" name="schema_data[name]"
                       value="<?php echo esc_attr($data['name'] ?? $post->post_title); ?>">
            </div>

            <div class="seo-fury-field">
                <label for="schema_howto_description"><?php esc_html_e('Description', 'seo-fury'); ?></label>
                <textarea id="schema_howto_description" name="schema_data[description]"
                          rows="3"><?php echo esc_textarea($data['description'] ?? ''); ?></textarea>
            </div>

            <h5 style="margin: 20px 0 10px 0;"><?php esc_html_e('Steps', 'seo-fury'); ?></h5>
            <div id="howto-steps-container">
                <?php foreach ($steps as $index => $step): ?>
                    <div class="howto-step" data-index="<?php echo $index; ?>">
                        <div class="seo-fury-field">
                            <?php /* translators: %d: step number */ ?>
                            <label><?php printf(esc_html__('Step %d Name', 'seo-fury'), $index + 1); ?></label>
                            <input type="text" name="schema_data[steps][<?php echo $index; ?>][name]"
                                   value="<?php echo esc_attr($step['name'] ?? ''); ?>">
                        </div>
                        <div class="seo-fury-field">
                            <label><?php esc_html_e('Step Description', 'seo-fury'); ?></label>
                            <textarea name="schema_data[steps][<?php echo $index; ?>][text]"
                                      rows="2"><?php echo esc_textarea($step['text'] ?? ''); ?></textarea>
                        </div>
                        <button type="button" class="button remove-howto-step"><?php esc_html_e('Remove Step', 'seo-fury'); ?></button>
                        <hr style="margin: 15px 0;">
                    </div>
                <?php endforeach; ?>
            </div>

            <button type="button" class="button button-secondary" id="add-howto-step">
                <span class="dashicons dashicons-plus-alt" style="margin-top: 3px;"></span>
                <?php esc_html_e('Add Step', 'seo-fury'); ?>
            </button>
        </div>
        <?php
    }

    private function render_review_fields($data, $post) {
        ?>
        <div class="seo-fury-section">
            <h4><?php esc_html_e('Review Schema Details', 'seo-fury'); ?></h4>
            <div class="seo-fury-fields-grid">
                <div class="seo-fury-field">
                    <label for="schema_review_item"><?php esc_html_e('Item Reviewed', 'seo-fury'); ?></label>
                    <input type="text" id="schema_review_item" name="schema_data[item_name]"
                           value="<?php echo esc_attr($data['item_name'] ?? ''); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_review_rating"><?php esc_html_e('Rating (1-5)', 'seo-fury'); ?></label>
                    <input type="number" min="1" max="5" step="0.1" id="schema_review_rating"
                           name="schema_data[rating]" value="<?php echo esc_attr($data['rating'] ?? '5'); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_review_author"><?php esc_html_e('Reviewer Name', 'seo-fury'); ?></label>
                    <input type="text" id="schema_review_author" name="schema_data[author]"
                           value="<?php echo esc_attr($data['author'] ?? get_the_author_meta('display_name', $post->post_author)); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_review_date"><?php esc_html_e('Review Date', 'seo-fury'); ?></label>
                    <input type="date" id="schema_review_date" name="schema_data[review_date]"
                           value="<?php echo esc_attr($data['review_date'] ?? date('Y-m-d')); ?>">
                </div>
            </div>
        </div>
        <?php
    }

    private function render_recipe_fields($data, $post) {
        ?>
        <div class="seo-fury-section">
            <h4><?php esc_html_e('Recipe Schema Details', 'seo-fury'); ?></h4>
            <div class="seo-fury-fields-grid">
                <div class="seo-fury-field">
                    <label for="schema_recipe_name"><?php esc_html_e('Recipe Name', 'seo-fury'); ?></label>
                    <input type="text" id="schema_recipe_name" name="schema_data[name]"
                           value="<?php echo esc_attr($data['name'] ?? $post->post_title); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_recipe_prep_time"><?php esc_html_e('Prep Time (minutes)', 'seo-fury'); ?></label>
                    <input type="number" id="schema_recipe_prep_time" name="schema_data[prep_time]"
                           value="<?php echo esc_attr($data['prep_time'] ?? ''); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_recipe_cook_time"><?php esc_html_e('Cook Time (minutes)', 'seo-fury'); ?></label>
                    <input type="number" id="schema_recipe_cook_time" name="schema_data[cook_time]"
                           value="<?php echo esc_attr($data['cook_time'] ?? ''); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_recipe_yield"><?php esc_html_e('Servings', 'seo-fury'); ?></label>
                    <input type="text" id="schema_recipe_yield" name="schema_data[recipe_yield]"
                           value="<?php echo esc_attr($data['recipe_yield'] ?? ''); ?>" placeholder="<?php echo esc_attr__('4 servings', 'seo-fury'); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_recipe_calories"><?php esc_html_e('Calories', 'seo-fury'); ?></label>
                    <input type="number" id="schema_recipe_calories" name="schema_data[calories]"
                           value="<?php echo esc_attr($data['calories'] ?? ''); ?>">
                </div>
            </div>

            <div class="seo-fury-field">
                <label for="schema_recipe_ingredients"><?php esc_html_e('Ingredients (one per line)', 'seo-fury'); ?></label>
                <textarea id="schema_recipe_ingredients" name="schema_data[ingredients]"
                          rows="5"><?php echo esc_textarea($data['ingredients'] ?? ''); ?></textarea>
            </div>

            <div class="seo-fury-field">
                <label for="schema_recipe_instructions"><?php esc_html_e('Instructions', 'seo-fury'); ?></label>
                <textarea id="schema_recipe_instructions" name="schema_data[instructions]"
                          rows="8"><?php echo esc_textarea($data['instructions'] ?? ''); ?></textarea>
            </div>
        </div>
        <?php
    }

    private function render_event_fields($data, $post) {
        ?>
        <div class="seo-fury-section">
            <h4><?php esc_html_e('Event Schema Details', 'seo-fury'); ?></h4>
            <div class="seo-fury-fields-grid">
                <div class="seo-fury-field">
                    <label for="schema_event_name"><?php esc_html_e('Event Name', 'seo-fury'); ?></label>
                    <input type="text" id="schema_event_name" name="schema_data[name]"
                           value="<?php echo esc_attr($data['name'] ?? $post->post_title); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_event_start"><?php esc_html_e('Start Date & Time', 'seo-fury'); ?></label>
                    <input type="datetime-local" id="schema_event_start" name="schema_data[start_date]"
                           value="<?php echo esc_attr($data['start_date'] ?? ''); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_event_end"><?php esc_html_e('End Date & Time', 'seo-fury'); ?></label>
                    <input type="datetime-local" id="schema_event_end" name="schema_data[end_date]"
                           value="<?php echo esc_attr($data['end_date'] ?? ''); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_event_location"><?php esc_html_e('Location Name', 'seo-fury'); ?></label>
                    <input type="text" id="schema_event_location" name="schema_data[location_name]"
                           value="<?php echo esc_attr($data['location_name'] ?? ''); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_event_address"><?php esc_html_e('Address', 'seo-fury'); ?></label>
                    <input type="text" id="schema_event_address" name="schema_data[address]"
                           value="<?php echo esc_attr($data['address'] ?? ''); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_event_price"><?php esc_html_e('Price', 'seo-fury'); ?></label>
                    <input type="number" step="0.01" id="schema_event_price" name="schema_data[price]"
                           value="<?php echo esc_attr($data['price'] ?? ''); ?>">
                </div>
            </div>
        </div>
        <?php
    }

    private function render_video_fields($data, $post) {
        ?>
        <div class="seo-fury-section">
            <h4><?php esc_html_e('Video Schema Details', 'seo-fury'); ?></h4>
            <div class="seo-fury-fields-grid">
                <div class="seo-fury-field">
                    <label for="schema_video_name"><?php esc_html_e('Video Title', 'seo-fury'); ?></label>
                    <input type="text" id="schema_video_name" name="schema_data[name]"
                           value="<?php echo esc_attr($data['name'] ?? $post->post_title); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_video_description"><?php esc_html_e('Description', 'seo-fury'); ?></label>
                    <textarea id="schema_video_description" name="schema_data[description]"
                              rows="3"><?php echo esc_textarea($data['description'] ?? ''); ?></textarea>
                </div>

                <div class="seo-fury-field">
                    <label for="schema_video_url"><?php esc_html_e('Video URL', 'seo-fury'); ?></label>
                    <input type="url" id="schema_video_url" name="schema_data[content_url]"
                           value="<?php echo esc_url($data['content_url'] ?? ''); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_video_thumbnail"><?php esc_html_e('Thumbnail URL', 'seo-fury'); ?></label>
                    <input type="url" id="schema_video_thumbnail" name="schema_data[thumbnail_url]"
                           value="<?php echo esc_url($data['thumbnail_url'] ?? ''); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_video_upload_date"><?php esc_html_e('Upload Date', 'seo-fury'); ?></label>
                    <input type="date" id="schema_video_upload_date" name="schema_data[upload_date]"
                           value="<?php echo esc_attr($data['upload_date'] ?? date('Y-m-d')); ?>">
                </div>

                <div class="seo-fury-field">
                    <label for="schema_video_duration"><?php esc_html_e('Duration (ISO 8601)', 'seo-fury'); ?></label>
                    <input type="text" id="schema_video_duration" name="schema_data[duration]"
                           value="<?php echo esc_attr($data['duration'] ?? ''); ?>" placeholder="PT1M30S">
                    <p class="description"><?php esc_html_e('Format: PT1H30M (1 hour 30 minutes)', 'seo-fury'); ?></p>
                </div>
            </div>
        </div>
        <?php
    }

    private function render_breadcrumb_fields($data, $post) {
        $items = $data['items'] ?? '';
        ?>
        <div class="seo-fury-section">
            <h4><?php esc_html_e('BreadcrumbList Details', 'seo-fury'); ?></h4>
            <p class="description"><?php esc_html_e('Enter one breadcrumb per line: Label | URL', 'seo-fury'); ?></p>
            <div class="seo-fury-field">
                <textarea id="schema_breadcrumb_items" name="schema_data[items]" rows="4" placeholder="Home | https://example.com&#10;Blog | https://example.com/blog"><?php echo esc_textarea($items); ?></textarea>
            </div>
        </div>
        <?php
    }

    private function render_table_fields($data, $post) {
        ?>
        <div class="seo-fury-section">
            <h4><?php esc_html_e('Table Schema Details', 'seo-fury'); ?></h4>
            <div class="seo-fury-fields-grid">
                <div class="seo-fury-field">
                    <label for="schema_table_name"><?php esc_html_e('Table Name', 'seo-fury'); ?></label>
                    <input type="text" id="schema_table_name" name="schema_data[name]" value="<?php echo esc_attr($data['name'] ?? $post->post_title); ?>">
                </div>
                <div class="seo-fury-field">
                    <label for="schema_table_description"><?php esc_html_e('Description', 'seo-fury'); ?></label>
                    <textarea id="schema_table_description" name="schema_data[description]" rows="3"><?php echo esc_textarea($data['description'] ?? ''); ?></textarea>
                </div>
                <div class="seo-fury-field">
                    <label for="schema_table_rows"><?php esc_html_e('Rows', 'seo-fury'); ?></label>
                    <textarea id="schema_table_rows" name="schema_data[rows]" rows="4" placeholder="Header A | Header B&#10;Row 1 A | Row 1 B"><?php echo esc_textarea($data['rows'] ?? ''); ?></textarea>
                    <p class="description"><?php esc_html_e('Use one row per line, columns separated by |', 'seo-fury'); ?></p>
                </div>
            </div>
        </div>
        <?php
    }

    private function render_service_fields($data, $post) {
        ?>
        <div class="seo-fury-section">
            <h4><?php esc_html_e('Service Schema Details', 'seo-fury'); ?></h4>
            <div class="seo-fury-fields-grid">
                <div class="seo-fury-field">
                    <label for="schema_service_name"><?php esc_html_e('Service Name', 'seo-fury'); ?></label>
                    <input type="text" id="schema_service_name" name="schema_data[name]" value="<?php echo esc_attr($data['name'] ?? $post->post_title); ?>">
                </div>
                <div class="seo-fury-field">
                    <label for="schema_service_description"><?php esc_html_e('Description', 'seo-fury'); ?></label>
                    <textarea id="schema_service_description" name="schema_data[description]" rows="3"><?php echo esc_textarea($data['description'] ?? ''); ?></textarea>
                </div>
                <div class="seo-fury-field">
                    <label for="schema_service_provider"><?php esc_html_e('Provider Name', 'seo-fury'); ?></label>
                    <input type="text" id="schema_service_provider" name="schema_data[provider]" value="<?php echo esc_attr($data['provider'] ?? get_bloginfo('name')); ?>">
                </div>
                <div class="seo-fury-field">
                    <label for="schema_service_area"><?php esc_html_e('Area Served', 'seo-fury'); ?></label>
                    <input type="text" id="schema_service_area" name="schema_data[area_served]" value="<?php echo esc_attr($data['area_served'] ?? ''); ?>">
                </div>
                <div class="seo-fury-field">
                    <label for="schema_service_type"><?php esc_html_e('Service Type', 'seo-fury'); ?></label>
                    <input type="text" id="schema_service_type" name="schema_data[service_type]" value="<?php echo esc_attr($data['service_type'] ?? ''); ?>">
                </div>
            </div>
        </div>
        <?php
    }

    private function render_jobposting_fields($data, $post) {
        ?>
        <div class="seo-fury-section">
            <h4><?php esc_html_e('Job Posting Details', 'seo-fury'); ?></h4>
            <div class="seo-fury-fields-grid">
                <div class="seo-fury-field">
                    <label for="schema_job_title"><?php esc_html_e('Job Title', 'seo-fury'); ?></label>
                    <input type="text" id="schema_job_title" name="schema_data[title]" value="<?php echo esc_attr($data['title'] ?? $post->post_title); ?>">
                </div>
                <div class="seo-fury-field">
                    <label for="schema_job_description"><?php esc_html_e('Description', 'seo-fury'); ?></label>
                    <textarea id="schema_job_description" name="schema_data[description]" rows="3"><?php echo esc_textarea($data['description'] ?? ''); ?></textarea>
                </div>
                <div class="seo-fury-field">
                    <label for="schema_job_date_posted"><?php esc_html_e('Date Posted', 'seo-fury'); ?></label>
                    <input type="date" id="schema_job_date_posted" name="schema_data[date_posted]" value="<?php echo esc_attr($data['date_posted'] ?? date('Y-m-d', strtotime($post->post_date))); ?>">
                </div>
                <div class="seo-fury-field">
                    <label for="schema_job_valid_through"><?php esc_html_e('Valid Through', 'seo-fury'); ?></label>
                    <input type="date" id="schema_job_valid_through" name="schema_data[valid_through]" value="<?php echo esc_attr($data['valid_through'] ?? ''); ?>">
                </div>
                <div class="seo-fury-field">
                    <label for="schema_job_employment_type"><?php esc_html_e('Employment Type', 'seo-fury'); ?></label>
                    <input type="text" id="schema_job_employment_type" name="schema_data[employment_type]" value="<?php echo esc_attr($data['employment_type'] ?? ''); ?>" placeholder="<?php echo esc_attr__('Full-time', 'seo-fury'); ?>">
                </div>
                <div class="seo-fury-field">
                    <label for="schema_job_org"><?php esc_html_e('Hiring Organization', 'seo-fury'); ?></label>
                    <input type="text" id="schema_job_org" name="schema_data[hiring_org]" value="<?php echo esc_attr($data['hiring_org'] ?? get_bloginfo('name')); ?>">
                </div>
                <div class="seo-fury-field">
                    <label for="schema_job_location"><?php esc_html_e('Job Location', 'seo-fury'); ?></label>
                    <input type="text" id="schema_job_location" name="schema_data[job_location]" value="<?php echo esc_attr($data['job_location'] ?? ''); ?>" placeholder="<?php echo esc_attr__('City, Country', 'seo-fury'); ?>">
                </div>
            </div>
        </div>
        <?php
    }

    private function generate_schema_preview($schema_type, $data, $post) {
        if ($schema_type === 'none') {
            return '';
        }

        $schema = $this->build_schema_json($schema_type, $data, $post);
        return json_encode($schema, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    }

    private function get_custom_schema_preview($custom_schema) {
        $custom_schema = trim((string) $custom_schema);
        if ($custom_schema === '') {
            return '';
        }
        return $custom_schema;
    }

    public function build_schema_json($schema_type, $data, $post) {
        $schema = array(
            '@context' => 'https://schema.org',
            '@type' => ucfirst(str_replace('_', '', $schema_type))
        );

        switch ($schema_type) {
            case 'article':
            case 'blog_posting':
            case 'news_article':
                $type_map = array(
                    'article' => 'Article',
                    'blog_posting' => 'BlogPosting',
                    'news_article' => 'NewsArticle'
                );
                $schema['@type'] = $type_map[$schema_type];
                $schema['headline'] = $data['headline'] ?? $post->post_title;
                $schema['author'] = array(
                    '@type' => 'Person',
                    'name' => $data['author'] ?? get_the_author_meta('display_name', $post->post_author)
                );
                $schema['publisher'] = array(
                    '@type' => 'Organization',
                    'name' => $data['publisher'] ?? get_bloginfo('name'),
                    'logo' => array(
                        '@type' => 'ImageObject',
                        'url' => $data['publisher_logo'] ?? ''
                    )
                );
                $schema['datePublished'] = isset($data['date_published']) ? date('c', strtotime($data['date_published'])) : get_the_date('c', $post->ID);
                $schema['dateModified'] = isset($data['date_modified']) ? date('c', strtotime($data['date_modified'])) : get_the_modified_date('c', $post->ID);
                if (!empty($data['image'])) {
                    $schema['image'] = $data['image'];
                }
                break;

            case 'product':
                $schema['@type'] = 'Product';
                $schema['name'] = $data['name'] ?? $post->post_title;
                if (!empty($data['price'])) {
                    $schema['offers'] = array(
                        '@type' => 'Offer',
                        'price' => $data['price'],
                        'priceCurrency' => $data['currency'] ?? 'USD',
                        'availability' => 'https://schema.org/' . ($data['availability'] ?? 'InStock')
                    );
                }
                if (!empty($data['brand'])) {
                    $schema['brand'] = array('@type' => 'Brand', 'name' => $data['brand']);
                }
                if (!empty($data['sku'])) {
                    $schema['sku'] = $data['sku'];
                }
                break;

            case 'faq':
                $schema['@type'] = 'FAQPage';
                $schema['mainEntity'] = array();
                if (!empty($data['faq_items'])) {
                    foreach ($data['faq_items'] as $faq) {
                        if (!empty($faq['question']) && !empty($faq['answer'])) {
                            $schema['mainEntity'][] = array(
                                '@type' => 'Question',
                                'name' => $faq['question'],
                                'acceptedAnswer' => array(
                                    '@type' => 'Answer',
                                    'text' => $faq['answer']
                                )
                            );
                        }
                    }
                }
                break;

            case 'breadcrumb':
                $schema['@type'] = 'BreadcrumbList';
                $schema['itemListElement'] = array();
                if (!empty($data['items'])) {
                    $lines = preg_split("/\r\n|\n|\r/", $data['items']);
                    $position = 1;
                    foreach ($lines as $line) {
                        $parts = array_map('trim', explode('|', $line));
                        if (count($parts) < 2) {
                            continue;
                        }
                        $schema['itemListElement'][] = array(
                            '@type' => 'ListItem',
                            'position' => $position,
                            'name' => $parts[0],
                            'item' => $parts[1],
                        );
                        $position++;
                    }
                } else {
                    $auto_items = $this->build_auto_breadcrumb_items($post);
                    if (!empty($auto_items)) {
                        $schema['itemListElement'] = $this->format_breadcrumb_items($auto_items);
                    }
                }
                break;

            case 'table':
                $schema['@type'] = 'Table';
                $schema['name'] = $data['name'] ?? $post->post_title;
                if (!empty($data['description'])) {
                    $schema['description'] = $data['description'];
                }
                if (!empty($data['rows'])) {
                    $schema['tableRows'] = preg_split("/\r\n|\n|\r/", $data['rows']);
                }
                break;

            case 'service':
                $schema['@type'] = 'Service';
                $schema['name'] = $data['name'] ?? $post->post_title;
                if (!empty($data['description'])) {
                    $schema['description'] = $data['description'];
                }
                if (!empty($data['provider'])) {
                    $schema['provider'] = array(
                        '@type' => 'Organization',
                        'name' => $data['provider']
                    );
                }
                if (!empty($data['area_served'])) {
                    $schema['areaServed'] = $data['area_served'];
                }
                if (!empty($data['service_type'])) {
                    $schema['serviceType'] = $data['service_type'];
                }
                break;

            case 'job_posting':
                $schema['@type'] = 'JobPosting';
                $schema['title'] = $data['title'] ?? $post->post_title;
                if (!empty($data['description'])) {
                    $schema['description'] = $data['description'];
                }
                if (!empty($data['date_posted'])) {
                    $schema['datePosted'] = $data['date_posted'];
                }
                if (!empty($data['valid_through'])) {
                    $schema['validThrough'] = $data['valid_through'];
                }
                if (!empty($data['employment_type'])) {
                    $schema['employmentType'] = $data['employment_type'];
                }
                if (!empty($data['hiring_org'])) {
                    $schema['hiringOrganization'] = array(
                        '@type' => 'Organization',
                        'name' => $data['hiring_org']
                    );
                }
                if (!empty($data['job_location'])) {
                    $schema['jobLocation'] = array(
                        '@type' => 'Place',
                        'address' => $data['job_location']
                    );
                }
                break;
        }

        return apply_filters('seo_fury_schema_json', $schema, $schema_type, $data, $post);
    }

    private function format_breadcrumb_items($items) {
        $list = array();
        $position = 1;
        foreach ($items as $item) {
            if (empty($item['name']) || empty($item['url'])) {
                continue;
            }
            $list[] = array(
                '@type' => 'ListItem',
                'position' => $position,
                'name' => $item['name'],
                'item' => $item['url'],
            );
            $position++;
        }
        return $list;
    }

    private function normalize_breadcrumb_url($url) {
        return untrailingslashit((string) $url);
    }

    private function build_auto_breadcrumb_items($post) {
        $items = $this->build_breadcrumb_from_menu($post);
        if (!empty($items)) {
            return $items;
        }

        if ($post->post_type === 'page') {
            return $this->build_breadcrumb_from_pages($post);
        }

        if ($post->post_type === 'post') {
            return $this->build_breadcrumb_from_categories($post);
        }

        return array(
            array('name' => __('Home', 'seo-fury'), 'url' => home_url('/')),
            array('name' => get_the_title($post), 'url' => get_permalink($post)),
        );
    }

    private function build_breadcrumb_from_menu($post) {
        $locations = get_nav_menu_locations();
        if (empty($locations)) {
            return array();
        }

        $current_url = $this->normalize_breadcrumb_url(get_permalink($post));
        foreach ($locations as $menu_id) {
            $menu_items = wp_get_nav_menu_items($menu_id);
            if (empty($menu_items)) {
                continue;
            }
            $items_by_id = array();
            $match = null;
            foreach ($menu_items as $menu_item) {
                $items_by_id[$menu_item->ID] = $menu_item;
                if ((int) $menu_item->object_id === (int) $post->ID) {
                    $match = $menu_item;
                } elseif ($this->normalize_breadcrumb_url($menu_item->url) === $current_url) {
                    $match = $menu_item;
                }
            }

            if (!$match) {
                continue;
            }

            $trail = array();
            $node = $match;
            while ($node) {
                array_unshift($trail, $node);
                $parent_id = (int) $node->menu_item_parent;
                if (!$parent_id || !isset($items_by_id[$parent_id])) {
                    break;
                }
                $node = $items_by_id[$parent_id];
            }

            $items = array();
            foreach ($trail as $trail_item) {
                if (empty($trail_item->title) || empty($trail_item->url)) {
                    continue;
                }
                $items[] = array(
                    'name' => wp_strip_all_tags($trail_item->title),
                    'url' => $trail_item->url,
                );
            }
            if (!empty($items)) {
                return $items;
            }
        }

        return array();
    }

    private function build_breadcrumb_from_pages($post) {
        $items = array();
        $items[] = array('name' => __('Home', 'seo-fury'), 'url' => home_url('/'));

        $ancestors = array_reverse(get_post_ancestors($post));
        foreach ($ancestors as $ancestor_id) {
            $items[] = array(
                'name' => get_the_title($ancestor_id),
                'url' => get_permalink($ancestor_id),
            );
        }

        $items[] = array('name' => get_the_title($post), 'url' => get_permalink($post));
        return $items;
    }

    private function build_breadcrumb_from_categories($post) {
        $items = array();
        $items[] = array('name' => __('Home', 'seo-fury'), 'url' => home_url('/'));

        $categories = get_the_category($post->ID);
        if (!empty($categories)) {
            $category = $categories[0];
            $ancestors = array_reverse(get_ancestors($category->term_id, 'category'));
            foreach ($ancestors as $ancestor_id) {
                $ancestor = get_term($ancestor_id, 'category');
                if ($ancestor && !is_wp_error($ancestor)) {
                    $items[] = array(
                        'name' => $ancestor->name,
                        'url' => get_term_link($ancestor),
                    );
                }
            }
            $items[] = array(
                'name' => $category->name,
                'url' => get_term_link($category),
            );
        }

        $items[] = array('name' => get_the_title($post), 'url' => get_permalink($post));
        return $items;
    }

    public function save_meta($post_id) {
        $items = array();
        if (isset($_POST['my_seo_schema_items'])) {
            $items_json = wp_unslash($_POST['my_seo_schema_items']);
            $decoded = json_decode($items_json, true);
            if (is_array($decoded)) {
                $items = $decoded;
            }
        }

        $active_id = isset($_POST['my_seo_schema_active']) ? sanitize_text_field($_POST['my_seo_schema_active']) : '';
        $mode = isset($_POST['my_seo_schema_mode']) ? sanitize_text_field($_POST['my_seo_schema_mode']) : 'generator';
        $type = isset($_POST['my_seo_schema_type']) ? sanitize_text_field($_POST['my_seo_schema_type']) : 'none';
        $custom = isset($_POST['my_seo_schema_custom']) ? wp_kses_post($_POST['my_seo_schema_custom']) : '';
        $schema_data = isset($_POST['schema_data']) ? $this->sanitize_schema_data($_POST['schema_data']) : array();

        $active_updated = false;
        foreach ($items as &$item) {
            if (!isset($item['id']) || $item['id'] !== $active_id) {
                continue;
            }
            $item['mode'] = $mode;
            $item['type'] = $mode === 'custom' ? 'custom' : $type;
            $item['data'] = $mode === 'custom' ? array() : $schema_data;
            $item['custom'] = $mode === 'custom' ? $custom : '';
            $active_updated = true;
            break;
        }
        unset($item);

        if (!$active_updated && $active_id) {
            if (($mode === 'custom' && !empty($custom)) || ($mode !== 'custom' && !empty($type) && $type !== 'none')) {
                $items[] = array(
                    'id' => $active_id,
                    'mode' => $mode,
                    'type' => $mode === 'custom' ? 'custom' : $type,
                    'data' => $mode === 'custom' ? array() : $schema_data,
                    'custom' => $mode === 'custom' ? $custom : '',
                );
            }
        }

        update_post_meta($post_id, '_my_seo_schema_items', $items);
        update_post_meta($post_id, '_my_seo_schema_active', $active_id);
        update_post_meta($post_id, '_my_seo_schema_mode', $mode);
        update_post_meta($post_id, '_my_seo_schema_type', $type);
        update_post_meta($post_id, '_my_seo_schema_custom', $custom);
        update_post_meta($post_id, '_my_seo_schema_data', $schema_data);
    }

    private function sanitize_schema_data($data) {
        $sanitized = array();
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $sanitized[$key] = $this->sanitize_schema_data($value);
            } else {
                $sanitized[$key] = sanitize_text_field($value);
            }
        }
        return $sanitized;
    }

    public function output_schema($post) {
        $items = get_post_meta($post->ID, '_my_seo_schema_items', true);
        if (is_array($items) && !empty($items)) {
            foreach ($items as $item) {
                $mode = $item['mode'] ?? 'generator';
                if ($mode === 'custom') {
                    $custom_schema = trim((string) ($item['custom'] ?? ''));
                    if ($custom_schema === '') {
                        continue;
                    }
                    $decoded = json_decode($custom_schema, true);
                    if (json_last_error() !== JSON_ERROR_NONE || empty($decoded)) {
                        continue;
                    }
                    echo '<script type="application/ld+json">' . "\n";
                    echo json_encode($decoded, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . "\n";
                    echo '</script>' . "\n";
                    continue;
                }

                $schema_type = $item['type'] ?? '';
                if (empty($schema_type) || $schema_type === 'none') {
                    continue;
                }
                $schema_data = $item['data'] ?? array();
                $schema_json = $this->build_schema_json($schema_type, $schema_data, $post);
                if (!empty($schema_json)) {
                    echo '<script type="application/ld+json">' . "\n";
                    echo json_encode($schema_json, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . "\n";
                    echo '</script>' . "\n";
                }
            }
            return;
        }

        $schema_mode = get_post_meta($post->ID, '_my_seo_schema_mode', true) ?: 'generator';
        if ($schema_mode === 'custom') {
            $custom_schema = get_post_meta($post->ID, '_my_seo_schema_custom', true);
            $custom_schema = trim((string) $custom_schema);
            if ($custom_schema === '') {
                return;
            }
            $decoded = json_decode($custom_schema, true);
            if (json_last_error() !== JSON_ERROR_NONE || empty($decoded)) {
                return;
            }
            echo '<script type="application/ld+json">' . "\n";
            echo json_encode($decoded, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . "\n";
            echo '</script>' . "\n";
            return;
        }

        $schema_type = get_post_meta($post->ID, '_my_seo_schema_type', true);
        if (empty($schema_type) || $schema_type === 'none') {
            return;
        }

        $schema_data = get_post_meta($post->ID, '_my_seo_schema_data', true) ?: array();
        $schema_json = $this->build_schema_json($schema_type, $schema_data, $post);
        if (!empty($schema_json)) {
            echo '<script type="application/ld+json">' . "\n";
            echo json_encode($schema_json, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . "\n";
            echo '</script>' . "\n";
        }
    }
}
