<?php

if (!defined('ABSPATH')) {
    exit;
}

require_once SEO_FURY_PATH . 'includes/class-post-meta-basic.php';
require_once SEO_FURY_PATH . 'includes/class-post-meta-social.php';
require_once SEO_FURY_PATH . 'includes/class-post-meta-schema.php';

class My_SEO_Plugin_Post_Meta
{
    private $basic_meta;
    private $social_meta;
    private $schema_meta;
    private $internal_links;

    public function __construct()
    {
        $this->init_hooks();
        $this->basic_meta = new MySEO_Plugin_Post_Meta_Basic();
        $this->social_meta = new MySEO_Plugin_Post_Meta_Social();
        $this->schema_meta = new MySEO_Plugin_Post_Meta_Schema();
        $this->internal_links = new SEO_Fury_Post_Meta_Internal_Links();
    }

    private function init_hooks()
    {
        add_action('add_meta_boxes', array($this, 'add_seo_meta_box'));

        add_action('save_post', array($this, 'save_seo_meta'), 10, 2);

        add_action('admin_enqueue_scripts', array($this, 'enqueue_post_meta_assets'));
        add_action('enqueue_block_editor_assets', array($this, 'enqueue_block_editor_assets'));

        add_action('wp_head', array($this, 'output_seo_meta_tags'), -1);
    }


    public function remove_default_meta_tags()
    {
        if (!is_singular()) {
            return;
        }

        global $post;

        $has_page_robots = get_post_meta($post->ID, '_my_seo_page_robots_noindex', true) ||
            get_post_meta($post->ID, '_my_seo_page_robots_nofollow', true);

        if ($has_page_robots) {
            remove_action('wp_head', 'wp_robots');
            add_filter('wp_robots', '__return_empty_array', 999);
        }

        $custom_canonical = get_post_meta($post->ID, '_my_seo_canonical', true);

        if ($custom_canonical) {
            remove_action('wp_head', 'rel_canonical');
            add_filter('wpseo_canonical', '__return_false');
            add_filter('aioseop_canonical_url', '__return_false');
            add_filter('rank_math/frontend/canonical', '__return_false');
        }
    }

    public function add_seo_meta_box()
    {
        $post_types = get_post_types(array('public' => true), 'names');
        $brand = function_exists('seo_fury_get_brand_name') ? seo_fury_get_brand_name() : 'SEO Fury';

        foreach ($post_types as $post_type) {
            add_meta_box(
                'my_seo_post_meta',
                $brand,
                array($this, 'render_meta_box'),
                $post_type,
                'normal',
                'high'
            );
        }
    }

    public function enqueue_post_meta_assets($hook)
    {
        if (!in_array($hook, array('post.php', 'post-new.php'))) {
            return;
        }

        wp_enqueue_media();

        wp_enqueue_style(
            'my-seo-post-meta',
            MY_SEO_PLUGIN_URL . 'assets/post-meta.css',
            array(),
            MY_SEO_PLUGIN_VERSION
        );

        wp_enqueue_script(
            'my-seo-post-meta',
            MY_SEO_PLUGIN_URL . 'assets/post-meta.js',
            array('jquery'),
            MY_SEO_PLUGIN_VERSION,
            true
        );

        global $post;
        if ($post) {
            $content = !empty($post->post_content) ? $post->post_content : '';
            $content = strip_shortcodes($content);
            $content = wp_strip_all_tags($content);
            $excerpt = wp_trim_words($content, 20, '...');

            wp_localize_script('my-seo-post-meta', 'seoFuryData', array(
                'siteName' => get_bloginfo('name'),
                'separator' => get_option('my_seo_title_separator', '—'),
                'postTitle' => $post->post_title ?: '',
                'postExcerpt' => $excerpt ?: '',
                'postId' => $post->ID,
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'internalLinksNonce' => wp_create_nonce('seo_fury_internal_links_nonce'),
                'i18n' => array(
                    'focusKeywordSet' => __('Focus keyword is set', 'seo-fury'),
                    'addFocusKeyword' => __('Add a focus keyword', 'seo-fury'),
                    'keywordInTitle' => __('Focus keyword in SEO title', 'seo-fury'),
                    'addKeywordToTitle' => __('Add focus keyword to SEO title', 'seo-fury'),
                    'keywordInDescription' => __('Focus keyword in meta description', 'seo-fury'),
                    'addKeywordToDescription' => __('Add focus keyword to meta description', 'seo-fury'),
                    'keywordInContent' => __('Focus keyword in content', 'seo-fury'),
                    'addKeywordToContent' => __('Add focus keyword to content', 'seo-fury'),
                    'keywordInUrl' => __('Focus keyword in URL', 'seo-fury'),
                    'addKeywordToUrl' => __('Add focus keyword to URL', 'seo-fury'),
                    'keywordDensityOk' => __('Keyword density is OK (0.5–2.5%)', 'seo-fury'),
                    'keywordDensityBad' => __('Keyword density out of range (0.5–2.5%)', 'seo-fury'),
                    'titleLengthOk' => __('SEO title length is OK (50–60)', 'seo-fury'),
                    'titleLengthBad' => __('Optimize SEO title length (50–60)', 'seo-fury'),
                    'descLengthOk' => __('Meta description length is OK (120–160)', 'seo-fury'),
                    'descLengthBad' => __('Optimize meta description length (120–160)', 'seo-fury'),
                    'contentLengthOk' => __('Content length is sufficient (300+ words)', 'seo-fury'),
                    'contentLengthBad' => __('Add more content (minimum 300 words)', 'seo-fury'),
                    'h1Present' => __('H1 heading is present', 'seo-fury'),
                    'addH1' => __('Add an H1 heading', 'seo-fury'),
                    'h2Present' => __('H2 subheadings are present', 'seo-fury'),
                    'addH2' => __('Add H2 subheadings', 'seo-fury'),
                    'imagesAltOk' => __('Images have alt attributes', 'seo-fury'),
                    'addImageAlt' => __('Add alt attributes to images', 'seo-fury'),
                    'internalLinkPresent' => __('Internal link is present', 'seo-fury'),
                    'addInternalLink' => __('Add an internal link', 'seo-fury'),
                    'enoughSentences' => __('Enough sentences for readability', 'seo-fury'),
                    'addMoreSentences' => __('Add more sentences', 'seo-fury'),
                    'avgSentenceLengthOk' => __('Average sentence length is OK', 'seo-fury'),
                    'makeSentencesShorter' => __('Make sentences shorter', 'seo-fury'),
                    'noLongSentences' => __('No overly long sentences', 'seo-fury'),
                    'longSentences' => __('There are overly long sentences', 'seo-fury'),
                    'multipleParagraphs' => __('Multiple paragraphs detected', 'seo-fury'),
                    'splitParagraphs' => __('Split text into paragraphs', 'seo-fury'),
                    'paragraphLengthOk' => __('Paragraph length is OK', 'seo-fury'),
                    'makeParagraphsShorter' => __('Make paragraphs shorter', 'seo-fury'),
                    'listsPresent' => __('Lists are present', 'seo-fury'),
                    'addLists' => __('Add lists (ul/ol)', 'seo-fury'),
                    'excellent' => __('Excellent', 'seo-fury'),
                    'okay' => __('Okay', 'seo-fury'),
                    'needsWork' => __('Needs work', 'seo-fury'),
                    'selectImage' => __('Select an image', 'seo-fury'),
                    'useThisImage' => __('Use this image', 'seo-fury'),
                    'keywordInFirstParagraph' => __('Keyword in first paragraph', 'seo-fury'),
                    'addKeywordToFirstParagraph' => __('Add keyword to first paragraph', 'seo-fury'),
                    'urlLengthOk' => __('URL length is OK', 'seo-fury'),
                    'shortenUrl' => __('Shorten URL (up to 75 chars)', 'seo-fury'),
                    'externalLinkPresent' => __('External link is present', 'seo-fury'),
                    'addExternalLink' => __('Add an external link', 'seo-fury'),
                    'imagesInContent' => __('Images are present in content', 'seo-fury'),
                    'addImage' => __('Add an image', 'seo-fury'),
                    'scanContent' => __('Scan Content', 'seo-fury'),
                    'scanning' => __('Scanning...', 'seo-fury'),
                    'linked' => __('Linked!', 'seo-fury'),
                    'notFound' => __('Not found', 'seo-fury'),
                    'titlePlaceholder' => __('Title', 'seo-fury'),
                    'descriptionPlaceholder' => __('Description', 'seo-fury'),
                    'pasteJsonLd' => __('// Paste your JSON-LD here', 'seo-fury'),
                    'noSchemaSelected' => __('No schema selected. Choose a template or import JSON-LD.', 'seo-fury'),
                    'customJsonLd' => __('Custom JSON-LD', 'seo-fury'),
                    'preview' => __('Preview', 'seo-fury'),
                    'edit' => __('Edit', 'seo-fury'),
                    'remove' => __('Remove', 'seo-fury'),
                    'jsonLdEmpty' => __('JSON-LD is empty.', 'seo-fury'),
                    'insertLink' => __('+ Link', 'seo-fury'),
                ),
            ));
        }
    }

    public function enqueue_block_editor_assets()
    {
        wp_enqueue_script(
            'my-seo-post-meta-gutenberg',
            MY_SEO_PLUGIN_URL . 'assets/post-meta-gutenberg.js',
            array('wp-plugins', 'wp-edit-post', 'wp-element', 'wp-components', 'wp-dom-ready'),
            MY_SEO_PLUGIN_VERSION,
            true
        );

        wp_localize_script('my-seo-post-meta-gutenberg', 'seoFuryGutenberg', array(
            'brandName' => function_exists('seo_fury_get_brand_name') ? seo_fury_get_brand_name() : 'SEO Fury',
        ));
    }

    public function parse_shortcodes($text, $post)
    {

        $title = $post->post_title ?? '';

        if (empty($title)) {
            $title = get_the_title($post->ID);
        }

        if (empty($title) || $title === 'Auto Draft' || $title === '(no title)') {
            $title = __('Untitled', 'seo-fury');
        }


        $content = $post->post_content ?? '';

        if (empty($content) && !empty($post->post_excerpt)) {
            $content = $post->post_excerpt;
        }

        if (!empty($content)) {
            $content = strip_shortcodes($content);
            $content = wp_strip_all_tags($content);
            $excerpt = wp_trim_words($content, 20, '...');
        } else {
            $excerpt = __('No description available', 'seo-fury');
        }


        $replacements = array(
            '%title%'     => $title,
            '%sitename%'  => get_bloginfo('name'),
            '%sep%'       => get_option('my_seo_title_separator', '—'),
            '%separator%' => get_option('my_seo_title_separator', '—'),
            '%page%'      => '',
            '%excerpt%'   => $excerpt,
            '%date%'      => get_the_date('', $post->ID),
            '%year%'      => get_the_date('Y', $post->ID),
            '%month%'     => get_the_date('m', $post->ID),
            '%day%'       => get_the_date('d', $post->ID),
        );

        return str_replace(array_keys($replacements), array_values($replacements), $text);
    }




    public function render_meta_box($post)
    {
        wp_nonce_field('my_seo_post_meta_nonce', 'my_seo_post_meta_nonce_field');

        $meta_data = array(
            'meta_title' => get_post_meta($post->ID, '_my_seo_title', true),
            'meta_description' => get_post_meta($post->ID, '_my_seo_description', true),
            'meta_keywords' => get_post_meta($post->ID, '_my_seo_keywords', true),
            'focus_keyword' => get_post_meta($post->ID, '_my_seo_focus_keyword', true),
            'page_robots_noindex' => get_post_meta($post->ID, '_my_seo_page_robots_noindex', true),
            'page_robots_nofollow' => get_post_meta($post->ID, '_my_seo_page_robots_nofollow', true),
            'page_robots_noarchive' => get_post_meta($post->ID, '_my_seo_page_robots_noarchive', true),
            'page_robots_nosnippet' => get_post_meta($post->ID, '_my_seo_page_robots_nosnippet', true),
            'page_robots_noimageindex' => get_post_meta($post->ID, '_my_seo_page_robots_noimageindex', true),
            'page_robots_max_snippet' => get_post_meta($post->ID, '_my_seo_page_robots_max_snippet', true),
            'page_robots_max_video' => get_post_meta($post->ID, '_my_seo_page_robots_max_video', true),
            'page_robots_max_image' => get_post_meta($post->ID, '_my_seo_page_robots_max_image', true),
            'canonical_url' => get_post_meta($post->ID, '_my_seo_canonical', true),
            'og_title' => get_post_meta($post->ID, '_my_seo_og_title', true),
            'og_description' => get_post_meta($post->ID, '_my_seo_og_description', true),
            'og_image' => get_post_meta($post->ID, '_my_seo_og_image', true),
            'og_type' => get_post_meta($post->ID, '_my_seo_og_type', true),
            'twitter_title' => get_post_meta($post->ID, '_my_seo_twitter_title', true),
            'twitter_description' => get_post_meta($post->ID, '_my_seo_twitter_description', true),
            'twitter_image' => get_post_meta($post->ID, '_my_seo_twitter_image', true),
            'twitter_card_type' => get_post_meta($post->ID, '_my_seo_twitter_card_type', true),
            'fb_title' => get_post_meta($post->ID, '_my_seo_fb_title', true),
            'fb_description' => get_post_meta($post->ID, '_my_seo_fb_description', true),
            'fb_image' => get_post_meta($post->ID, '_my_seo_fb_image', true),
        );

        if (empty($meta_data['canonical_url'])) {
            $meta_data['canonical_url'] = get_permalink($post->ID);
            if (!$meta_data['canonical_url']) $meta_data['canonical_url'] = '';
        }

        if (empty($meta_data['og_type'])) $meta_data['og_type'] = 'article';
        if (empty($meta_data['twitter_card_type'])) $meta_data['twitter_card_type'] = 'summary_large_image';

        $preview_title = $this->parse_shortcodes($meta_data['meta_title'] ?: '%title%', $post);
        $preview_description = $this->parse_shortcodes($meta_data['meta_description'] ?: '%excerpt%', $post);

        $current_url = get_permalink($post->ID);
        if (!$current_url) $current_url = home_url();

        $og_preview_title = !empty($meta_data['og_title']) ? $meta_data['og_title'] : $preview_title;
        $og_preview_desc = !empty($meta_data['og_description']) ? $meta_data['og_description'] : $preview_description;
        $og_preview_image = !empty($meta_data['og_image']) ? $meta_data['og_image'] : '';

        $tw_preview_title = !empty($meta_data['twitter_title']) ? $meta_data['twitter_title'] : $og_preview_title;
        $tw_preview_desc = !empty($meta_data['twitter_description']) ? $meta_data['twitter_description'] : $og_preview_desc;
        $tw_preview_image = !empty($meta_data['twitter_image']) ? $meta_data['twitter_image'] : $og_preview_image;

        $fb_preview_title = !empty($meta_data['fb_title']) ? $meta_data['fb_title'] : $og_preview_title;
        $fb_preview_desc = !empty($meta_data['fb_description']) ? $meta_data['fb_description'] : $og_preview_desc;
        $fb_preview_image = !empty($meta_data['fb_image']) ? $meta_data['fb_image'] : $og_preview_image;

        $preview_data = array(
            'preview_title' => $preview_title,
            'preview_description' => $preview_description,
            'current_url' => $current_url,
            'og_preview_title' => $og_preview_title,
            'og_preview_desc' => $og_preview_desc,
            'og_preview_image' => $og_preview_image,
            'tw_preview_title' => $tw_preview_title,
            'tw_preview_desc' => $tw_preview_desc,
            'tw_preview_image' => $tw_preview_image,
            'fb_preview_title' => $fb_preview_title,
            'fb_preview_desc' => $fb_preview_desc,
            'fb_preview_image' => $fb_preview_image,
        );

?>
        <div class="seo-fury-wrapper">
            <div class="seo-fury-content">
                <!-- Navigation tabs -->
                <div class="seo-fury-tabs">
                    <button type="button" class="seo-fury-tab active" data-tab="basic">
                        <span class="dashicons dashicons-search"></span>
                        <span class="seo-fury-tab-label"><?php esc_html_e('SEO Basics', 'seo-fury'); ?></span>
                    </button>
                    <button type="button" class="seo-fury-tab" data-tab="advanced">
                        <span class="dashicons dashicons-admin-generic"></span>
                        <span class="seo-fury-tab-label"><?php esc_html_e('Advanced', 'seo-fury'); ?></span>
                    </button>
                    <button type="button" class="seo-fury-tab" data-tab="schema">
                        <span class="dashicons dashicons-editor-code"></span>
                        <span class="seo-fury-tab-label"><?php esc_html_e('Schema', 'seo-fury'); ?></span>
                    </button>
                    <button type="button" class="seo-fury-tab" data-tab="links">
                        <span class="dashicons dashicons-admin-links"></span>
                        <span class="seo-fury-tab-label"><?php esc_html_e('Links', 'seo-fury'); ?></span>
                    </button>
                </div>

                <!-- Content tabs -->
                <div class="seo-fury-tab-panels">
                    <?php
                    $this->basic_meta->render_basic_tab($post, $meta_data, $preview_data);
                    $this->basic_meta->render_advanced_tab($post, $meta_data);
                    $this->schema_meta->render_schema_tab($post, $meta_data);
                    ?>
                    <div id="tab-links" class="seo-fury-tab-content">
                        <?php $this->internal_links->render_tab($post); ?>
                    </div>
                </div>
            </div>
        </div>

        <!-- Snippet editing modal -->
        <div id="seo-fury-snippet-modal" class="seo-fury-modal" style="display: none;">
            <div class="seo-fury-modal-overlay"></div>
            <div class="seo-fury-modal-content">
                <div class="seo-fury-modal-header">
                    <h2><?php esc_html_e('Edit SEO Snippet', 'seo-fury'); ?></h2>
                    <button type="button" class="seo-fury-modal-close">
                        <span class="dashicons dashicons-no"></span>
                    </button>
                </div>

                <div class="seo-fury-modal-body">
                    <div class="seo-fury-modal-tabs">
                        <button type="button" class="seo-fury-modal-tab active" data-tab="basic">
                            <span class="dashicons dashicons-admin-generic"></span>
                            <?php esc_html_e('General', 'seo-fury'); ?>
                        </button>
                        <button type="button" class="seo-fury-modal-tab" data-tab="social">
                            <span class="dashicons dashicons-share"></span>
                            <?php esc_html_e('Social', 'seo-fury'); ?>
                        </button>
                    </div>

                    <div class="seo-fury-modal-tab-panels">
                        <div class="seo-fury-modal-panel active" data-panel="basic">
                            <div class="seo-fury-modal-grid">
                                <div class="seo-fury-modal-grid-left">
                                    <div class="seo-fury-modal-preview">
                                        <h4><?php esc_html_e('Google Preview', 'seo-fury'); ?></h4>
                                        <div class="google-preview google-preview-modal">
                                            <div class="preview-url"><?php echo esc_html(parse_url($current_url, PHP_URL_HOST)); ?></div>
                                            <div class="preview-title"><?php echo esc_html($preview_title); ?></div>
                                            <div class="preview-description"><?php echo esc_html($preview_description); ?></div>
                                        </div>
                                    </div>
                                </div>

                                <div class="seo-fury-modal-grid-right">
                                    <div class="seo-fury-field">
                                        <div class="field-header">
                                            <label for="my_seo_title"><?php esc_html_e('SEO Title', 'seo-fury'); ?></label>
                                            <div class="char-counter">
                                                <span><span id="title-length">0</span> / 60</span>
                                                <span class="seo-fury-pixel-counter"><span id="title-pixels">0</span> / 580px</span>
                                            </div>
                                        </div>
                                        <div class="input-with-dropdown">
                                            <input type="text" id="my_seo_title" name="my_seo_title" value="<?php echo esc_attr($meta_data['meta_title']); ?>">
                                            <button type="button" class="shortcode-dropdown-btn" data-target="my_seo_title">
                                                <span class="dashicons dashicons-arrow-down-alt2"></span>
                                            </button>
                                            <div class="shortcode-dropdown-menu">
                                                <div class="shortcode-item" data-var="%sep%"><strong>%sep%</strong> <span><?php esc_html_e('Separator -, —', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%sitename%"><strong>%sitename%</strong> <span><?php esc_html_e('Site name', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%sitedesc%"><strong>%sitedesc%</strong> <span><?php esc_html_e('Site description', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%title%"><strong>%title%</strong> <span><?php esc_html_e('Post title', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%parenttitle%"><strong>%parenttitle%</strong> <span><?php esc_html_e('Parent page', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%seotitle%"><strong>%seotitle%</strong> <span><?php esc_html_e('SEO title', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%category%"><strong>%category%</strong> <span><?php esc_html_e('Category', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%tag%"><strong>%tag%</strong> <span><?php esc_html_e('Tag', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%term%"><strong>%term%</strong> <span><?php esc_html_e('Taxonomy term', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%date%"><strong>%date%</strong> <span><?php esc_html_e('Publish date', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%currentdate%"><strong>%currentdate%</strong> <span><?php esc_html_e('Current date', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%currentyear%"><strong>%currentyear%</strong> <span><?php esc_html_e('Current year', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%name%"><strong>%name%</strong> <span><?php esc_html_e('Author name', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%page%"><strong>%page%</strong> <span><?php esc_html_e('Page number (2 of 4)', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%ptsingle%"><strong>%ptsingle%</strong> <span><?php esc_html_e('Post type (singular)', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%searchquery%"><strong>%searchquery%</strong> <span><?php esc_html_e('Search query', 'seo-fury'); ?></span></div>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="seo-fury-field">
                                        <div class="field-header">
                                            <label for="my_seo_description"><?php esc_html_e('Meta Description', 'seo-fury'); ?></label>
                                            <div class="char-counter">
                                                <span><span id="description-length">0</span> / 160</span>
                                                <span class="seo-fury-pixel-counter"><span id="description-pixels">0</span> / 920px</span>
                                            </div>
                                        </div>
                                        <div class="input-with-dropdown">
                                            <textarea id="my_seo_description" name="my_seo_description" rows="2"><?php echo esc_textarea($meta_data['meta_description']); ?></textarea>
                                            <button type="button" class="shortcode-dropdown-btn" data-target="my_seo_description">
                                                <span class="dashicons dashicons-arrow-down-alt2"></span>
                                            </button>
                                            <div class="shortcode-dropdown-menu">
                                                <div class="shortcode-item" data-var="%excerpt%"><strong>%excerpt%</strong> <span><?php esc_html_e('Post excerpt', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%excerptonly%"><strong>%excerptonly%</strong> <span><?php esc_html_e('Excerpt field only', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%seodescription%"><strong>%seodescription%</strong> <span><?php esc_html_e('SEO description', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%title%"><strong>%title%</strong> <span><?php esc_html_e('Post title', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%sitename%"><strong>%sitename%</strong> <span><?php esc_html_e('Site name', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%sitedesc%"><strong>%sitedesc%</strong> <span><?php esc_html_e('Site description', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%category%"><strong>%category%</strong> <span><?php esc_html_e('Category', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%termdescription%"><strong>%termdescription%</strong> <span><?php esc_html_e('Term description', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%date%"><strong>%date%</strong> <span><?php esc_html_e('Publish date', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%name%"><strong>%name%</strong> <span><?php esc_html_e('Author name', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%userdescription%"><strong>%userdescription%</strong> <span><?php esc_html_e('Author bio', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%wcshortdesc%"><strong>%wcshortdesc%</strong> <span><?php esc_html_e('Product short description', 'seo-fury'); ?></span></div>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="seo-fury-field">
                                        <label for="my_seo_keywords"><?php esc_html_e('Meta Keywords', 'seo-fury'); ?></label>
                                        <div class="input-with-dropdown">
                                            <input type="text" id="my_seo_keywords" name="my_seo_keywords" value="<?php echo esc_attr($meta_data['meta_keywords']); ?>">
                                            <button type="button" class="shortcode-dropdown-btn" data-target="my_seo_keywords">
                                                <span class="dashicons dashicons-arrow-down-alt2"></span>
                                            </button>
                                            <div class="shortcode-dropdown-menu">
                                                <div class="shortcode-item" data-var="%focuskw%"><strong>%focuskw%</strong> <span><?php esc_html_e('Focus keyword', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%keywords%"><strong>%keywords%</strong> <span><?php esc_html_e('Keywords', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%tags%"><strong>%tags%</strong> <span><?php esc_html_e('Tags', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%category%"><strong>%category%</strong> <span><?php esc_html_e('Category', 'seo-fury'); ?></span></div>
                                                <div class="shortcode-item" data-var="%categories%"><strong>%categories%</strong> <span><?php esc_html_e('All categories', 'seo-fury'); ?></span></div>
                                            </div>
                                        </div>
                                        <p class="description"><?php esc_html_e('Separate keywords with commas', 'seo-fury'); ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="seo-fury-modal-panel" data-panel="social">
                            <div class="seo-fury-modal-social-grid">
                                <div class="seo-fury-modal-social-left">
                                    <div class="seo-fury-section seo-fury-social-settings">
                                        <h4><?php esc_html_e('Social Settings', 'seo-fury'); ?></h4>
                                        <div class="seo-fury-social-settings-tabs">
                                            <button type="button" class="seo-fury-social-settings-tab active" data-settings-tab="all"><?php esc_html_e('All Networks', 'seo-fury'); ?></button>
                                            <button type="button" class="seo-fury-social-settings-tab" data-settings-tab="facebook"><?php esc_html_e('Facebook', 'seo-fury'); ?></button>
                                            <button type="button" class="seo-fury-social-settings-tab" data-settings-tab="twitter"><?php esc_html_e('Twitter', 'seo-fury'); ?></button>
                                        </div>

                                        <div class="seo-fury-social-settings-panels">
                                            <div class="seo-fury-social-settings-panel active" data-settings-panel="all">
                                                <div class="seo-fury-fields-grid">
                                                    <div class="seo-fury-field">
                                                        <label for="my_seo_social_title_modal"><?php esc_html_e('Title', 'seo-fury'); ?></label>
                                                        <input type="text" id="my_seo_social_title_modal" class="seo-fury-social-all-title" placeholder="<?php echo esc_attr__('Title for all networks', 'seo-fury'); ?>">
                                                    </div>
                                                    <div class="seo-fury-field">
                                                        <label for="my_seo_social_description_modal"><?php esc_html_e('Description', 'seo-fury'); ?></label>
                                                        <textarea id="my_seo_social_description_modal" class="seo-fury-social-all-description" rows="3" placeholder="<?php echo esc_attr__('Description for all networks', 'seo-fury'); ?>"></textarea>
                                                    </div>
                                                    <div class="seo-fury-field">
                                                        <label for="my_seo_social_image_modal"><?php esc_html_e('Image', 'seo-fury'); ?></label>
                                                        <input type="url" id="my_seo_social_image_modal" class="seo-fury-social-all-image" placeholder="https://">
                                                        <button type="button" class="button my-seo-upload-image" data-target="my_seo_social_image_modal">
                                                            <span class="dashicons dashicons-format-image" style="vertical-align: middle;"></span>
                                                            <?php esc_html_e('Choose Image', 'seo-fury'); ?>
                                                        </button>
                                                    </div>
                                                </div>
                                                <p class="description"><?php esc_html_e('Values apply to Facebook and Twitter', 'seo-fury'); ?></p>
                                            </div>

                                            <div class="seo-fury-social-settings-panel" data-settings-panel="facebook">
                                                <div class="seo-fury-fields-grid">
                                                    <div class="seo-fury-field">
                                                        <label for="my_seo_fb_title"><?php esc_html_e('Facebook Title', 'seo-fury'); ?></label>
                                                        <div class="input-with-dropdown">
                                                            <input type="text" id="my_seo_fb_title" name="my_seo_fb_title" value="<?php echo esc_attr($meta_data['fb_title']); ?>">
                                                            <button type="button" class="shortcode-dropdown-btn" data-target="my_seo_fb_title">
                                                                <span class="dashicons dashicons-arrow-down-alt2"></span>
                                                            </button>
                                                            <div class="shortcode-dropdown-menu">
                                                                <div class="shortcode-item" data-var="%title%"><strong>%title%</strong> <span><?php esc_html_e('Post title', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%seotitle%"><strong>%seotitle%</strong> <span><?php esc_html_e('SEO title', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%sitename%"><strong>%sitename%</strong> <span><?php esc_html_e('Site name', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%sep%"><strong>%sep%</strong> <span><?php esc_html_e('Separator —', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%category%"><strong>%category%</strong> <span><?php esc_html_e('Category', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%date%"><strong>%date%</strong> <span><?php esc_html_e('Publish date', 'seo-fury'); ?></span></div>
                                                            </div>
                                                        </div>
                                                    </div>

                                                    <div class="seo-fury-field">
                                                        <label for="my_seo_fb_image"><?php esc_html_e('Facebook Image', 'seo-fury'); ?></label>
                                                        <input type="url" id="my_seo_fb_image" name="my_seo_fb_image" value="<?php echo esc_attr($meta_data['fb_image']); ?>">
                                                        <button type="button" class="button my-seo-upload-image" data-target="my_seo_fb_image">
                                                            <span class="dashicons dashicons-format-image" style="vertical-align: middle;"></span>
                                                            <?php esc_html_e('Choose Image', 'seo-fury'); ?>
                                                        </button>
                                                        <button type="button" class="button my-seo-remove-image-button" data-target="my_seo_fb_image"
                                                            style="<?php echo $meta_data['fb_image'] ? '' : 'display:none;'; ?>margin-left: 5px;">
                                                            <span class="dashicons dashicons-no" style="vertical-align: middle;"></span>
                                                            <?php esc_html_e('Remove', 'seo-fury'); ?>
                                                        </button>
                                                    </div>

                                                    <div class="seo-fury-field">
                                                        <label for="my_seo_fb_description"><?php esc_html_e('Facebook Description', 'seo-fury'); ?></label>
                                                        <div class="input-with-dropdown">
                                                            <textarea id="my_seo_fb_description" name="my_seo_fb_description" rows="3"><?php echo esc_textarea($meta_data['fb_description']); ?></textarea>
                                                            <button type="button" class="shortcode-dropdown-btn" data-target="my_seo_fb_description">
                                                                <span class="dashicons dashicons-arrow-down-alt2"></span>
                                                            </button>
                                                            <div class="shortcode-dropdown-menu">
                                                                <div class="shortcode-item" data-var="%excerpt%"><strong>%excerpt%</strong> <span><?php esc_html_e('Post excerpt', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%seodescription%"><strong>%seodescription%</strong> <span><?php esc_html_e('SEO description', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%title%"><strong>%title%</strong> <span><?php esc_html_e('Post title', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%sitedesc%"><strong>%sitedesc%</strong> <span><?php esc_html_e('Site description', 'seo-fury'); ?></span></div>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="seo-fury-social-settings-panel" data-settings-panel="twitter">
                                                <div class="seo-fury-fields-grid">
                                                    <div class="seo-fury-field">
                                                        <label for="my_seo_twitter_title"><?php esc_html_e('Twitter Title', 'seo-fury'); ?></label>
                                                        <div class="input-with-dropdown">
                                                            <input type="text" id="my_seo_twitter_title" name="my_seo_twitter_title" value="<?php echo esc_attr($meta_data['twitter_title']); ?>">
                                                            <button type="button" class="shortcode-dropdown-btn" data-target="my_seo_twitter_title">
                                                                <span class="dashicons dashicons-arrow-down-alt2"></span>
                                                            </button>
                                                            <div class="shortcode-dropdown-menu">
                                                                <div class="shortcode-item" data-var="%title%"><strong>%title%</strong> <span><?php esc_html_e('Post title', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%seotitle%"><strong>%seotitle%</strong> <span><?php esc_html_e('SEO title', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%sitename%"><strong>%sitename%</strong> <span><?php esc_html_e('Site name', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%sep%"><strong>%sep%</strong> <span><?php esc_html_e('Separator —', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%category%"><strong>%category%</strong> <span><?php esc_html_e('Category', 'seo-fury'); ?></span></div>
                                                            </div>
                                                        </div>
                                                    </div>

                                                    <div class="seo-fury-field">
                                                        <label for="my_seo_twitter_card_type"><?php esc_html_e('Twitter Card Type', 'seo-fury'); ?></label>
                                                        <select id="my_seo_twitter_card_type" name="my_seo_twitter_card_type">
                                                            <option value="summary" <?php selected($meta_data['twitter_card_type'], 'summary'); ?>><?php esc_html_e('Summary', 'seo-fury'); ?></option>
                                                            <option value="summary_large_image" <?php selected($meta_data['twitter_card_type'], 'summary_large_image'); ?>><?php esc_html_e('Summary Large Image', 'seo-fury'); ?></option>
                                                        </select>
                                                    </div>

                                                    <div class="seo-fury-field">
                                                        <label for="my_seo_twitter_description"><?php esc_html_e('Twitter Description', 'seo-fury'); ?></label>
                                                        <div class="input-with-dropdown">
                                                            <textarea id="my_seo_twitter_description" name="my_seo_twitter_description" rows="3"><?php echo esc_textarea($meta_data['twitter_description']); ?></textarea>
                                                            <button type="button" class="shortcode-dropdown-btn" data-target="my_seo_twitter_description">
                                                                <span class="dashicons dashicons-arrow-down-alt2"></span>
                                                            </button>
                                                            <div class="shortcode-dropdown-menu">
                                                                <div class="shortcode-item" data-var="%excerpt%"><strong>%excerpt%</strong> <span><?php esc_html_e('Post excerpt', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%seodescription%"><strong>%seodescription%</strong> <span><?php esc_html_e('SEO description', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%title%"><strong>%title%</strong> <span><?php esc_html_e('Post title', 'seo-fury'); ?></span></div>
                                                                <div class="shortcode-item" data-var="%sitedesc%"><strong>%sitedesc%</strong> <span><?php esc_html_e('Site description', 'seo-fury'); ?></span></div>
                                                            </div>
                                                        </div>
                                                    </div>

                                                    <div class="seo-fury-field">
                                                        <label for="my_seo_twitter_image"><?php esc_html_e('Twitter Image', 'seo-fury'); ?></label>
                                                        <input type="url" id="my_seo_twitter_image" name="my_seo_twitter_image" value="<?php echo esc_attr($meta_data['twitter_image']); ?>">
                                                        <button type="button" class="button my-seo-upload-image" data-target="my_seo_twitter_image">
                                                            <span class="dashicons dashicons-format-image" style="vertical-align: middle;"></span>
                                                            <?php esc_html_e('Choose Image', 'seo-fury'); ?>
                                                        </button>
                                                        <button type="button" class="button my-seo-remove-image-button" data-target="my_seo_twitter_image"
                                                            style="<?php echo $meta_data['twitter_image'] ? '' : 'display:none;'; ?>margin-left: 5px;">
                                                            <span class="dashicons dashicons-no" style="vertical-align: middle;"></span>
                                                            <?php esc_html_e('Remove', 'seo-fury'); ?>
                                                        </button>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div class="seo-fury-modal-social-right">
                                    <div class="seo-fury-section">
                                        <h4><?php esc_html_e('Social Preview', 'seo-fury'); ?></h4>
                                        <div class="seo-fury-social-preview-tabs">
                                            <button type="button" class="seo-fury-social-tab active" data-social-tab="facebook">
                                                <span class="dashicons dashicons-facebook"></span>
                                                <?php esc_html_e('Facebook', 'seo-fury'); ?>
                                            </button>
                                            <button type="button" class="seo-fury-social-tab" data-social-tab="twitter">
                                                <span class="dashicons dashicons-twitter"></span>
                                                <?php esc_html_e('Twitter', 'seo-fury'); ?>
                                            </button>
                                        </div>
                                        <div class="seo-fury-social-preview-panels">
                                            <div class="seo-fury-social-panel active" data-social-panel="facebook">
                                                <div class="social-post-card facebook-card">
                                                    <div class="social-post-header">
                                                        <div class="social-avatar"></div>
                                                        <div class="social-post-meta">
                                                            <strong class="social-username"><?php esc_html_e('Facebook user', 'seo-fury'); ?></strong>
                                                            <div class="social-time"><?php esc_html_e('2 hours ago', 'seo-fury'); ?></div>
                                                        </div>
                                                    </div>
                                                    <?php if ($preview_data['fb_preview_image']): ?>
                                                        <div class="social-post-image">
                                                            <img src="<?php echo esc_url($preview_data['fb_preview_image']); ?>" alt="<?php echo esc_attr__('FB Preview', 'seo-fury'); ?>">
                                                        </div>
                                                    <?php endif; ?>
                                                    <div class="social-post-footer facebook-footer">
                                                        <div class="social-link-domain"><?php echo esc_html(strtoupper(parse_url($preview_data['current_url'], PHP_URL_HOST))); ?></div>
                                                        <div class="social-link-title fb-title"><?php echo esc_html($preview_data['fb_preview_title']); ?></div>
                                                        <div class="social-link-description fb-description"><?php echo esc_html($preview_data['fb_preview_desc']); ?></div>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="seo-fury-social-panel" data-social-panel="twitter">
                                                <div class="social-post-card twitter-card">
                                                    <div class="social-post-header">
                                                        <div class="social-avatar"></div>
                                                        <div class="social-post-meta">
                                                            <strong class="social-username"><?php esc_html_e('Twitter user', 'seo-fury'); ?></strong>
                                                            <div class="social-handle">@username · <?php esc_html_e('2 hours', 'seo-fury'); ?></div>
                                                        </div>
                                                    </div>
                                                    <div class="social-post-text"><?php esc_html_e('Post with a link!', 'seo-fury'); ?></div>
                                                    <?php if ($preview_data['tw_preview_image']): ?>
                                                        <div class="social-post-image">
                                                            <img src="<?php echo esc_url($preview_data['tw_preview_image']); ?>" alt="<?php echo esc_attr__('Twitter Preview', 'seo-fury'); ?>">
                                                        </div>
                                                    <?php endif; ?>
                                                    <div class="social-post-footer twitter-footer">
                                                        <div class="social-link-title tw-title"><?php echo esc_html($preview_data['tw_preview_title']); ?></div>
                                                        <div class="social-link-description tw-description"><?php echo esc_html($preview_data['tw_preview_desc']); ?></div>
                                                        <div class="social-link-domain"><?php echo esc_html(parse_url($preview_data['current_url'], PHP_URL_HOST)); ?></div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="seo-fury-modal-footer">
                    <button type="button" class="button button-primary seo-fury-modal-save"><?php esc_html_e('Save Changes', 'seo-fury'); ?></button>
                    <button type="button" class="button seo-fury-modal-close"><?php esc_html_e('Cancel', 'seo-fury'); ?></button>
                </div>
            </div>
        </div>
<?php
    }



    public function save_seo_meta($post_id, $post)
    {
        if (!isset($_POST['my_seo_post_meta_nonce_field']) || !wp_verify_nonce($_POST['my_seo_post_meta_nonce_field'], 'my_seo_post_meta_nonce')) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        if (isset($_POST['my_seo_title'])) {
            update_post_meta($post_id, '_my_seo_title', sanitize_text_field($_POST['my_seo_title']));
        }

        if (isset($_POST['my_seo_description'])) {
            update_post_meta($post_id, '_my_seo_description', sanitize_textarea_field($_POST['my_seo_description']));
        }

        if (isset($_POST['my_seo_keywords'])) {
            update_post_meta($post_id, '_my_seo_keywords', sanitize_text_field($_POST['my_seo_keywords']));
        }

        if (isset($_POST['my_seo_focus_keyword'])) {
            update_post_meta($post_id, '_my_seo_focus_keyword', sanitize_text_field($_POST['my_seo_focus_keyword']));
        }


        $this->basic_meta->save_meta($post_id);
        $this->social_meta->save_meta($post_id);
        $this->schema_meta->save_meta($post_id);
    }


    public function output_seo_meta_tags()
    {
        if (!is_singular()) {
            return;
        }

        global $post;

        error_log('POST-META DEBUG START');
        error_log('Post ID: ' . $post->ID);

        $meta_title = get_post_meta($post->ID, '_my_seo_title', true);
        if (!$meta_title) {
            $meta_title = get_option('my_seo_title_template_' . $post->post_type, '');
        }
        if ($meta_title) {
            error_log('Meta Title: ' . $meta_title);
            echo '<meta name="title" content="' . esc_attr($this->parse_shortcodes($meta_title, $post)) . '">' . "\n";
        }

        $meta_description = get_post_meta($post->ID, '_my_seo_description', true);
        if (!$meta_description) {
            $meta_description = get_option('my_seo_description_template_' . $post->post_type, '');
        }
        if ($meta_description) {
            error_log('Meta Description: ' . substr($meta_description, 0, 50));
            echo '<meta name="description" content="' . esc_attr($this->parse_shortcodes($meta_description, $post)) . '">' . "\n";
        }

        $meta_keywords = get_post_meta($post->ID, '_my_seo_keywords', true);
        if ($meta_keywords) {
            error_log('Meta Keywords: ' . $meta_keywords);
            echo '<meta name="keywords" content="' . esc_attr($this->parse_shortcodes($meta_keywords, $post)) . '">' . "\n";
        }

        $this->basic_meta->output_meta_tags($post);
        $this->social_meta->output_meta_tags($post, array($this, 'parse_shortcodes'));

        $this->schema_meta->output_schema($post);

        error_log('POST-META DEBUG END');
    }
}
