<?php

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Fury_Pro_Features {

    private $license;
    private $grace_period = 259200;
    private $verify_interval = 21600;
    private $api_url = 'https://api.seo-fury.com/api/v1';

    private $free_limits = array(
        'max_redirects' => 50,
        'max_404_logs' => 100,
        'block_ai_bots' => false,
        'advanced_seo' => false,
        'white_label' => false,
        'priority_support' => false,
        'schema_markup' => false,
        'social_cards_advanced' => false,
        'breadcrumbs' => false,
        'bulk_edit' => false,
        'csv_export' => false,
        'auto_redirects' => false,
        'internal_linking' => false
    );

    public function __construct($license) {
        $this->license = $license;
    }

    private function should_hide_branding() {
        return function_exists('seo_fury_hide_branding') ? seo_fury_hide_branding() : false;
    }

    private function get_upgrade_url() {
        return $this->should_hide_branding() ? '' : 'https://seo-fury.com/#pricing';
    }

    public function is_pro() {
        if (get_option('seo_fury_integrity_failed', 0)) {
            return false;
        }

        $license_data = $this->license->get_cached_license_data();

        if (!$license_data || !isset($license_data['status'])) {
            return false;
        }

        $plan = $license_data['plan'] ?? '';
        $is_local_pro = $license_data['status'] === 'active' &&
               (strpos($plan, 'pro_') === 0 || $plan === 'pro');

        if (!$is_local_pro) {
            return false;
        }

        return $this->verify_server_status();
    }

    private function verify_server_status() {
        $cache_key = '_sf_srv_verify';
        $cached = get_option($cache_key, array());

        if (!empty($cached) && isset($cached['ts']) && isset($cached['ok'])) {
            $age = time() - $cached['ts'];

            if ($age < $this->verify_interval && $cached['ok']) {
                return true;
            }

            if ($age < $this->grace_period && $cached['ok']) {
                if ($age >= $this->verify_interval) {
                    $this->schedule_background_verify();
                }
                return true;
            }

            if ($age >= $this->grace_period && !$cached['ok']) {
                return false;
            }

            if ($age >= $this->grace_period) {
                $result = $this->do_server_verify();
                if ($result === null) {
                    return false;
                }
                return $result;
            }
        }

        $result = $this->do_server_verify();
        if ($result === null) {
            update_option($cache_key, array('ts' => time(), 'ok' => true), false);
            return true;
        }
        return $result;
    }

    private function do_server_verify() {
        $license_key = $this->license->get_cached_license_data();
        $key = get_option($this->get_key_option());

        if (!$key) {
            return false;
        }

        $domain = $this->get_domain();
        $fingerprint = new My_SEO_Plugin_Fingerprint();
        $site_info = $fingerprint->get_site_info();

        $response = wp_remote_post($this->api_url . '/plugin/validate', array(
            'timeout' => 10,
            'headers' => array('Content-Type' => 'application/json'),
            'body' => json_encode(array(
                'licenseKey' => $key,
                'domain' => $domain,
                'siteHash' => $site_info['siteHash'],
                'pluginVersion' => defined('SEO_FURY_VERSION') ? SEO_FURY_VERSION : '0.0.0',
            )),
        ));

        if (is_wp_error($response)) {
            return null;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code < 200 || $code >= 300) {
            return null;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (!is_array($body)) {
            return null;
        }

        $is_valid = false;
        if (!empty($body['success'])) {
            if (!empty($body['valid']) || !empty($body['data']['valid']) ||
                (isset($body['data']['status']) && $body['data']['status'] === 'active')) {
                $is_valid = true;
            }
        }

        $cache_key = '_sf_srv_verify';
        update_option($cache_key, array('ts' => time(), 'ok' => $is_valid), false);

        if (!$is_valid) {
            $this->license->validate_license();
        }

        return $is_valid;
    }

    private function schedule_background_verify() {
        if (!wp_next_scheduled('sf_background_verify')) {
            wp_schedule_single_event(time(), 'sf_background_verify');
            add_action('sf_background_verify', array($this, 'background_verify_callback'));
        }
    }

    public function background_verify_callback() {
        $this->do_server_verify();
    }

    private function get_key_option() {
        $salt = 'seo_fury_2025';
        return '_sf_' . substr(md5($salt . 'key'), 0, 8);
    }

    private function get_domain() {
        $url = home_url();
        $parsed = parse_url($url);
        $domain = $parsed['host'] ?? 'localhost';
        return strtolower(preg_replace('/^www\./', '', $domain));
    }

    public function has_feature($feature_name) {
        if ($this->is_pro()) {
            return true;
        }

        if (isset($this->free_limits[$feature_name])) {
            return $this->free_limits[$feature_name];
        }

        return false;
    }

    public function get_limit($feature_name) {
        if ($this->is_pro()) {
            return PHP_INT_MAX;
        }

        if (isset($this->free_limits[$feature_name])) {
            return $this->free_limits[$feature_name];
        }

        return 0;
    }

    public function is_limit_reached($feature_name, $current_count) {
        $limit = $this->get_limit($feature_name);

        if (is_bool($limit)) {
            return !$limit;
        }

        return $current_count >= $limit;
    }

    public function get_plan_info() {
        if ($this->is_pro()) {
            return array(
                'name' => __('PRO', 'seo-fury'),
                'features' => array(
                    __('Unlimited Redirects', 'seo-fury'),
                    __('Unlimited 404 Monitoring', 'seo-fury'),
                    __('AI Bot Blocking', 'seo-fury'),
                    __('Advanced SEO', 'seo-fury'),
                    __('Schema Markup', 'seo-fury'),
                    __('Social Cards', 'seo-fury'),
                    __('Breadcrumbs', 'seo-fury'),
                    __('White Label', 'seo-fury'),
                    __('Priority Support', 'seo-fury')
                )
            );
        }

        $plan_info = array(
            'name' => __('FREE', 'seo-fury'),
            'features' => array(
                __('50 Redirects', 'seo-fury'),
                __('100 404 Logs', 'seo-fury'),
                __('Basic SEO', 'seo-fury'),
                __('Robots.txt', 'seo-fury'),
                __('Sitemap', 'seo-fury'),
                __('Meta Tags', 'seo-fury'),
                __('IndexNow', 'seo-fury')
            ),
        );
        $upgrade_url = $this->get_upgrade_url();
        if (!empty($upgrade_url)) {
            $plan_info['upgrade_url'] = $upgrade_url;
        }
        return $plan_info;
    }

    public function get_pro_badge($feature_name = '') {
        if ($this->is_pro()) {
            return '';
        }

        $tooltip = '';
        if (!empty($feature_name) && !$this->should_hide_branding()) {
            $tooltip = ' title="' . esc_attr(sprintf(
                /* translators: %s: feature name */
                __('Upgrade to PRO to unlock %s', 'seo-fury'),
                $feature_name
            )) . '"';
        }

        return '<span class="seo-fury-pro-badge"' . $tooltip . '>' . esc_html__('PRO', 'seo-fury') . '</span>';
    }

    public function get_limit_message($feature_name) {
        $limit = $this->get_limit($feature_name);

        if (is_bool($limit) && !$limit) {
            if ($this->should_hide_branding()) {
                return __('This feature requires a PRO license.', 'seo-fury');
            }
            return sprintf(
                /* translators: %1$s: opening link tag, %2$s: closing link tag */
                __('This feature requires %1$sPRO license%2$s.', 'seo-fury'),
                '<a href="' . esc_url($this->get_upgrade_url()) . '" target="_blank">',
                '</a>'
            );
        }

        if (is_numeric($limit)) {
            if ($this->should_hide_branding()) {
                return sprintf(
                    /* translators: %d: maximum number of items allowed */
                    __('FREE plan limited to %d items. Upgrade to PRO for unlimited.', 'seo-fury'),
                    $limit
                );
            }
            return sprintf(
                /* translators: %1$d: maximum number of items allowed, %2$s: opening link tag, %3$s: closing link tag */
                __('FREE plan limited to %1$d items. %2$sUpgrade to PRO%3$s for unlimited.', 'seo-fury'),
                $limit,
                '<a href="' . esc_url($this->get_upgrade_url()) . '" target="_blank">',
                '</a>'
            );
        }

        return '';
    }
}
