<?php

if (!defined('ABSPATH')) {
    exit;
}

class My_SEO_Plugin_Redirects {

    private $table_name;

    public function __construct($table_name) {
        $this->table_name = $table_name;
        $this->init_hooks();
    }

private function init_hooks() {
    add_action('template_redirect', array($this, 'check_redirects'), 1);

    add_action('wp_ajax_seo_add_redirect', array($this, 'ajax_add_redirect'));
    add_action('wp_ajax_seo_delete_redirect', array($this, 'ajax_delete_redirect'));
    add_action('wp_ajax_seo_toggle_redirect', array($this, 'ajax_toggle_redirect'));
    add_action('wp_ajax_seo_load_redirects', array($this, 'ajax_load_redirects'));
    add_action('wp_ajax_seo_update_redirect', array($this, 'ajax_update_redirect'));
    add_action('wp_ajax_seo_create_table', array($this, 'ajax_create_table'));
    add_action('wp_ajax_seo_export_redirects_csv', array($this, 'ajax_export_redirects_csv'));
    add_action('wp_ajax_seo_import_redirects_csv', array($this, 'ajax_import_redirects_csv'));
}

    public function check_redirects() {
    if (defined('REST_REQUEST') && REST_REQUEST) {
        return;
    }

    if (isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], '/wp-json/') !== false) {
        return;
    }

    if (is_admin()) {
        return;
    }

    global $wpdb;

    $request_uri = $_SERVER['REQUEST_URI'];

    $current_url = strtok($request_uri, '?');

    $current_url = urldecode($current_url);

    $url_with_slash = trailingslashit($current_url);
    $url_without_slash = untrailingslashit($current_url);

    $redirect = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$this->table_name}
        WHERE (source_url = %s OR source_url = %s)
        AND active = 1
        LIMIT 1",
        $url_with_slash,
        $url_without_slash
    ));

    if ($redirect) {
        $wpdb->update(
            $this->table_name,
            array('hits' => $redirect->hits + 1),
            array('id' => $redirect->id),
            array('%d'),
            array('%d')
        );

        wp_redirect($redirect->target_url, $redirect->redirect_type);
        exit;
    }
}

public function ajax_add_redirect() {
    check_ajax_referer('seo_redirects_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Insufficient permissions', 'seo-fury'));
    }

    global $wpdb;

    $core = SEO_Fury_Core::get_instance();
    $pro = $core->get_pro_features();

    $current_count = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name}");

    if ($pro->is_limit_reached('max_redirects', $current_count)) {
        $limit = $pro->get_limit('max_redirects');
        if (function_exists('seo_fury_hide_branding') && seo_fury_hide_branding()) {
            wp_send_json_error(
                sprintf(
                    /* translators: %d: maximum number of redirects allowed */
                    __('FREE plan limit reached (%d redirects). Upgrade to PRO for unlimited redirects.', 'seo-fury'),
                    $limit
                )
            );
        }
        wp_send_json_error(
            sprintf(
                /* translators: %1$d: maximum number of redirects allowed, %2$s: opening link tag, %3$s: closing link tag */
                __('FREE plan limit reached (%1$d redirects). %2$sUpgrade to PRO%3$s for unlimited redirects.', 'seo-fury'),
                $limit,
                '<a href="https://seo-fury.com/#pricing" target="_blank">',
                '</a>'
            )
        );
    }

    $source = isset($_POST['source']) ? sanitize_text_field($_POST['source']) : '';
    $target = isset($_POST['target']) ? sanitize_text_field($_POST['target']) : '';
    $type = isset($_POST['type']) ? intval($_POST['type']) : 301;

    if (empty($source) || empty($target)) {
        wp_send_json_error(__('Please fill in all fields.', 'seo-fury'));
    }

    if ($source !== '/') {
        $source = '/' . trim($source, '/');
    }

    if ($target !== '/') {
        $target = '/' . trim($target, '/');
    }

    $exists = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM {$this->table_name} WHERE source_url = %s",
        $source
    ));

    if ($exists > 0) {
        wp_send_json_error(__('A redirect with this source already exists.', 'seo-fury'));
    }

    $result = $wpdb->insert(
        $this->table_name,
        array(
            'source_url' => $source,
            'target_url' => $target,
            'redirect_type' => $type,
            'hits' => 0,
            'active' => 1,
            'created' => current_time('mysql'),
        ),
        array('%s', '%s', '%d', '%d', '%d', '%s')
    );

    if ($result) {
        wp_send_json_success(__('Redirect added successfully.', 'seo-fury'));
    } else {
        wp_send_json_error(__('Failed to add redirect.', 'seo-fury') . ' ' . $wpdb->last_error);
    }
}

    public function ajax_delete_redirect() {
        check_ajax_referer('seo_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'seo-fury'));
        }

        global $wpdb;

        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;

        if ($id <= 0) {
            wp_send_json_error(__('Invalid ID', 'seo-fury'));
        }

        $result = $wpdb->delete(
            $this->table_name,
            array('id' => $id),
            array('%d')
        );

        if ($result) {
            wp_send_json_success(__('Redirect deleted.', 'seo-fury'));
        } else {
            wp_send_json_error(__('Failed to delete redirect.', 'seo-fury'));
        }
    }

    public function ajax_toggle_redirect() {
        check_ajax_referer('seo_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'seo-fury'));
        }

        global $wpdb;

        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
        $active = isset($_POST['active']) ? intval($_POST['active']) : 0;

        if ($id <= 0) {
            wp_send_json_error(__('Invalid ID', 'seo-fury'));
        }

        $result = $wpdb->update(
            $this->table_name,
            array('active' => $active),
            array('id' => $id),
            array('%d'),
            array('%d')
        );

        if ($result !== false) {
            wp_send_json_success(__('Status updated.', 'seo-fury'));
        } else {
            wp_send_json_error(__('Failed to update status.', 'seo-fury'));
        }
    }

    public function ajax_load_redirects() {
        check_ajax_referer('seo_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'seo-fury'));
        }

        global $wpdb;

        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$this->table_name}'");

        if (!$table_exists) {
            wp_send_json_error(array(
                'message' => __('Redirects table is not created.', 'seo-fury'),
                'redirects' => array(),
                'total' => 0,
                'active' => 0
            ));
            return;
        }

        $redirects = $wpdb->get_results("SELECT * FROM {$this->table_name} ORDER BY id DESC");

        $active_count = 0;
        if (!empty($redirects)) {
            foreach ($redirects as $redirect) {
                if (isset($redirect->active) && $redirect->active == 1) {
                    $active_count++;
                }
            }
        }

        wp_send_json_success(array(
            'redirects' => $redirects ? $redirects : array(),
            'total' => count($redirects),
            'active' => $active_count
        ));
    }

    public function ajax_update_redirect() {
        check_ajax_referer('seo_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'seo-fury'));
        }

        global $wpdb;

        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
        $source = isset($_POST['source']) ? sanitize_text_field($_POST['source']) : '';
        $target = isset($_POST['target']) ? sanitize_text_field($_POST['target']) : '';
        $type = isset($_POST['type']) ? intval($_POST['type']) : 301;

        if ($id <= 0 || empty($source) || empty($target)) {
            wp_send_json_error(__('Please fill in all fields.', 'seo-fury'));
        }

        if ($source !== '/') {
            $source = '/' . trim($source, '/');
        }

        if ($target !== '/') {
            $target = '/' . trim($target, '/');
        }

        $result = $wpdb->update(
            $this->table_name,
            array(
                'source_url' => $source,
                'target_url' => $target,
                'redirect_type' => $type,
            ),
            array('id' => $id),
            array('%s', '%s', '%d'),
            array('%d')
        );

        if ($result !== false) {
            wp_send_json_success(__('Redirect updated.', 'seo-fury'));
        } else {
            wp_send_json_error(__('Failed to update redirect.', 'seo-fury') . ' ' . $wpdb->last_error);
        }
    }

    public function ajax_create_table() {
        check_ajax_referer('seo_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'seo-fury'));
        }

        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            source_url varchar(500) NOT NULL,
            target_url varchar(500) NOT NULL,
            redirect_type int(3) NOT NULL DEFAULT 301,
            hits bigint(20) UNSIGNED NOT NULL DEFAULT 0,
            active tinyint(1) NOT NULL DEFAULT 1,
            created datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY source_url (source_url(191)),
            KEY active (active)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);

        wp_send_json_success(__('Table created.', 'seo-fury'));
    }

    public function ajax_export_redirects_csv() {
        check_ajax_referer('seo_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'seo-fury'));
        }

        global $wpdb;
        $rows = $wpdb->get_results("SELECT source_url, target_url, redirect_type, active FROM {$this->table_name} ORDER BY id DESC", ARRAY_A);

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=seo-fury-redirects.csv');
        $output = fopen('php://output', 'w');
        fputcsv($output, array('source_url', 'target_url', 'redirect_type', 'active'));
        foreach ($rows as $row) {
            fputcsv($output, $row);
        }
        fclose($output);
        exit;
    }

    public function ajax_import_redirects_csv() {
        check_ajax_referer('seo_redirects_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'seo-fury'));
        }

        if (empty($_FILES['file']['tmp_name'])) {
            wp_send_json_error(__('CSV file is required.', 'seo-fury'));
        }

        $file = $_FILES['file']['tmp_name'];
        $handle = fopen($file, 'r');
        if (!$handle) {
            wp_send_json_error(__('Failed to read CSV file.', 'seo-fury'));
        }

        global $wpdb;
        $imported = 0;
        $updated = 0;
        $skipped = 0;

        $header = fgetcsv($handle);
        if (!$header || count($header) < 2) {
            fclose($handle);
            wp_send_json_error(__('Invalid CSV header.', 'seo-fury'));
        }

        while (($data = fgetcsv($handle)) !== false) {
            $source = isset($data[0]) ? trim($data[0]) : '';
            $target = isset($data[1]) ? trim($data[1]) : '';
            $type = isset($data[2]) ? intval($data[2]) : 301;
            $active = isset($data[3]) ? intval($data[3]) : 1;

            if (empty($source) || empty($target)) {
                $skipped++;
                continue;
            }

            if ($source !== '/') {
                $source = '/' . trim($source, '/');
            }
            if ($target !== '/') {
                $target = '/' . trim($target, '/');
            }

            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM {$this->table_name} WHERE source_url = %s",
                $source
            ));

            if ($exists) {
                $wpdb->update(
                    $this->table_name,
                    array(
                        'target_url' => $target,
                        'redirect_type' => $type,
                        'active' => $active,
                    ),
                    array('id' => $exists),
                    array('%s', '%d', '%d'),
                    array('%d')
                );
                $updated++;
            } else {
                $wpdb->insert(
                    $this->table_name,
                    array(
                        'source_url' => $source,
                        'target_url' => $target,
                        'redirect_type' => $type,
                        'active' => $active,
                        'hits' => 0,
                        'created' => current_time('mysql')
                    ),
                    array('%s', '%s', '%d', '%d', '%d', '%s')
                );
                $imported++;
            }
        }
        fclose($handle);

        wp_send_json_success(array(
            'imported' => $imported,
            'updated' => $updated,
            'skipped' => $skipped
        ));
    }
}
