<?php

if (!defined('ABSPATH')) {
    exit;
}

class My_SEO_Plugin_Robots {

    public function __construct() {
        $this->init_hooks();
    }

    private function init_hooks() {
        add_filter('robots_txt', array($this, 'filter_robots_txt'), 10, 2);
        add_action('wp_ajax_seo_reset_robots', array($this, 'ajax_reset_robots'));
    }

    public function register_settings() {
        register_setting('my_seo_robots_group', 'my_seo_robots_enabled');
        register_setting('my_seo_robots_group', 'my_seo_robots_content');
        register_setting('my_seo_robots_group', 'my_seo_robots_user_agents');
        register_setting('my_seo_robots_group', 'my_seo_robots_crawl_delay');
        register_setting('my_seo_robots_group', 'my_seo_robots_use_dynamic_vars');
        register_setting('my_seo_robots_group', 'my_seo_robots_block_ai_bots');
        register_setting('my_seo_robots_group', 'my_seo_robots_block_archive_bots');

        add_settings_section(
            'my_seo_robots_section',
            __('Robots.txt Settings', 'seo-fury'),
            array($this, 'robots_section_callback'),
            'my_seo_robots_group'
        );

        add_settings_field('my_seo_robots_enabled', __('Enable virtual robots.txt', 'seo-fury'), array($this, 'robots_enabled_callback'), 'my_seo_robots_group', 'my_seo_robots_section');
        add_settings_field('my_seo_robots_content', __('robots.txt Content', 'seo-fury'), array($this, 'robots_content_callback'), 'my_seo_robots_group', 'my_seo_robots_section');
        add_settings_field('my_seo_robots_user_agents', __('Additional User-agent Blocks', 'seo-fury'), array($this, 'robots_user_agents_callback'), 'my_seo_robots_group', 'my_seo_robots_section');
        add_settings_field('my_seo_robots_crawl_delay', __('Crawl-delay for bots', 'seo-fury'), array($this, 'robots_crawl_delay_callback'), 'my_seo_robots_group', 'my_seo_robots_section');
        add_settings_field('my_seo_robots_use_dynamic_vars', __('Use dynamic variables', 'seo-fury'), array($this, 'robots_use_dynamic_vars_callback'), 'my_seo_robots_group', 'my_seo_robots_section');
        add_settings_field('my_seo_robots_block_ai_bots', __('Block AI bots', 'seo-fury'), array($this, 'robots_block_ai_bots_callback'), 'my_seo_robots_group', 'my_seo_robots_section');
        add_settings_field('my_seo_robots_block_archive_bots', __('Block web archive bots', 'seo-fury'), array($this, 'robots_block_archive_bots_callback'), 'my_seo_robots_group', 'my_seo_robots_section');
        add_settings_field('my_seo_robots_reset', __('Reset settings', 'seo-fury'), array($this, 'robots_reset_callback'), 'my_seo_robots_group', 'my_seo_robots_section');
        add_settings_field('my_seo_robots_link', __('Current robots.txt', 'seo-fury'), array($this, 'robots_link_callback'), 'my_seo_robots_group', 'my_seo_robots_section');
    }

    public function robots_section_callback() {
        $robots_path = ABSPATH . 'robots.txt';
        if (file_exists($robots_path)) {
            $physical_content = file_get_contents($robots_path);
            ?>
            <div class="notice notice-warning inline">
                <p>
                    <strong><?php esc_html_e('Warning!', 'seo-fury'); ?></strong> <?php esc_html_e('A physical robots.txt file was found in the site root.', 'seo-fury'); ?>
                    <?php esc_html_e('It takes precedence over the virtual WordPress robots.txt.', 'seo-fury'); ?>
                </p>
                <details>
                    <summary><?php esc_html_e('Show physical file contents', 'seo-fury'); ?></summary>
                    <pre style="background: #f5f5f5; padding: 10px; overflow: auto; max-height: 300px;"><?php echo esc_html($physical_content); ?></pre>
                </details>
            </div>
            <?php
        } else {
            ?>
            <div class="notice notice-info inline">
                <p>
                    <?php esc_html_e('No physical robots.txt file found in the site root.', 'seo-fury'); ?>
                    <?php esc_html_e('The virtual WordPress robots.txt is used.', 'seo-fury'); ?>
                </p>
            </div>
            <?php
        }
    }

    public function robots_enabled_callback() {
        $enabled = get_option('my_seo_robots_enabled', 1);
        ?>
        <label>
            <input type="checkbox" name="my_seo_robots_enabled" value="1" <?php checked($enabled, 1); ?> />
            <?php esc_html_e('Enable virtual robots.txt', 'seo-fury'); ?>
        </label>
        <?php
    }

    public function robots_content_callback() {
        $content = get_option('my_seo_robots_content', '');
        if (empty($content)) {
            $content = $this->get_default_robots_template();
        }
        ?>
        <textarea name="my_seo_robots_content" rows="20" cols="80" class="large-text code" style="font-family: monospace; width: 100%;"><?php echo esc_textarea($content); ?></textarea>
        <p class="description">
            <?php esc_html_e('robots.txt editor. Available variables:', 'seo-fury'); ?>
            <code>{site_url}</code>,
            <code>{home_url}</code>,
            <code>{uploads_path}</code>.
        </p>
        <?php
    }

    public function robots_user_agents_callback() {
        $ua_rules = get_option('my_seo_robots_user_agents', '');
        if (is_array($ua_rules)) {
            $ua_rules = implode("\n", $ua_rules);
        }
        ?>
        <textarea name="my_seo_robots_user_agents" rows="10" cols="80" class="large-text code" style="font-family: monospace; width: 100%;"><?php echo esc_textarea($ua_rules); ?></textarea>
        <p class="description">
            <?php esc_html_e('Additional rule blocks for specific user agents. Example:', 'seo-fury'); ?><br>
            <code>User-agent: Googlebot</code><br>
            Disallow: /admin/
        </p>
        <?php
    }

    public function robots_crawl_delay_callback() {
        $crawl_delay = get_option('my_seo_robots_crawl_delay', '');
        ?>
        <textarea name="my_seo_robots_crawl_delay" rows="5" cols="80" class="large-text code" style="font-family: monospace; width: 100%;"><?php echo esc_textarea($crawl_delay); ?></textarea>
        <p class="description">
            <?php
            printf(
                /* translators: %s: code example of format */
                esc_html__('Format: one line = %s, for example:', 'seo-fury'),
                '<code>User-agent:Name = seconds</code>'
            );
            ?><br>
            <code>Googlebot=2</code><br>
            <code>Yandex=5</code><br>
            <?php
            printf(
                /* translators: %s: Crawl-delay directive name */
                esc_html__('Converted to %s directives for the listed bots.', 'seo-fury'),
                '<code>Crawl-delay</code>'
            );
            ?>
        </p>
        <?php
    }

    public function robots_use_dynamic_vars_callback() {
        $use_vars = get_option('my_seo_robots_use_dynamic_vars', 1);
        ?>
        <label>
            <input type="checkbox" name="my_seo_robots_use_dynamic_vars" value="1" <?php checked($use_vars, 1); ?> />
            <?php esc_html_e('Automatically replace', 'seo-fury'); ?> <code>{site_url}</code>, <code>{home_url}</code>, <code>{uploads_path}</code>
        </label>
        <?php
    }

    public function robots_block_ai_bots_callback() {
        $block_ai = get_option('my_seo_robots_block_ai_bots', 0);
        $core = SEO_Fury_Core::get_instance();
        $pro = $core->get_pro_features();
        $is_pro = $pro->is_pro();
        $should_hide = function_exists('seo_fury_hide_branding') ? seo_fury_hide_branding() : false;
        ?>
        <label style="<?php echo !$is_pro ? 'opacity: 0.5;' : ''; ?>">
            <input type="checkbox" name="my_seo_robots_block_ai_bots" value="1" <?php checked($block_ai, 1); ?> <?php echo !$is_pro ? 'disabled' : ''; ?> />
            <?php esc_html_e('Block AI crawlers (GPTBot, ChatGPT-User, Google-Extended, Omgilibot, ClaudeBot, Claude-Web)', 'seo-fury'); ?>
            <?php if (!$is_pro): ?>
                <span style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 2px 8px; border-radius: 3px; font-size: 11px; font-weight: 600; margin-left: 8px;"><?php esc_html_e('PRO', 'seo-fury'); ?></span>
            <?php endif; ?>
        </label>
        <p class="description">
            <?php esc_html_e('When enabled, robots.txt will include Disallow rules for popular AI crawlers.', 'seo-fury'); ?>
            <?php if (!$is_pro && !$should_hide): ?>
                <br><strong style="color: #667eea;"><a href="https://seo-fury.com/#pricing" target="_blank"><?php esc_html_e('Upgrade to PRO', 'seo-fury'); ?></a> <?php esc_html_e('to unlock', 'seo-fury'); ?></strong>
            <?php endif; ?>
        </p>
        <?php
    }

    public function robots_block_archive_bots_callback() {
        $block_archive = get_option('my_seo_robots_block_archive_bots', 0);
        $core = SEO_Fury_Core::get_instance();
        $pro = $core->get_pro_features();
        $is_pro = $pro->is_pro();
        $should_hide = function_exists('seo_fury_hide_branding') ? seo_fury_hide_branding() : false;
        ?>
        <label style="<?php echo !$is_pro ? 'opacity: 0.5;' : ''; ?>">
            <input type="checkbox" name="my_seo_robots_block_archive_bots" value="1" <?php checked($block_archive, 1); ?> <?php echo !$is_pro ? 'disabled' : ''; ?> />
            <?php esc_html_e('Block web archive bots (Wayback Machine, Archive.org, archive.is, and others)', 'seo-fury'); ?>
            <?php if (!$is_pro): ?>
                <span style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 2px 8px; border-radius: 3px; font-size: 11px; font-weight: 600; margin-left: 8px;"><?php esc_html_e('PRO', 'seo-fury'); ?></span>
            <?php endif; ?>
        </label>
        <p class="description">
            <?php esc_html_e('When enabled, web archive services cannot store copies of your site.', 'seo-fury'); ?>
            <?php esc_html_e('This helps protect unique content and prevents unauthorized archiving.', 'seo-fury'); ?>
            <?php if (!$is_pro && !$should_hide): ?>
                <br><strong style="color: #667eea;"><a href="https://seo-fury.com/#pricing" target="_blank"><?php esc_html_e('Upgrade to PRO', 'seo-fury'); ?></a> <?php esc_html_e('to unlock', 'seo-fury'); ?></strong>
            <?php endif; ?>
        </p>
        <?php
    }

    public function robots_reset_callback() {
        ?>
        <button type="button" id="seo-reset-robots" class="button button-secondary">
            <?php esc_html_e('Reset to defaults', 'seo-fury'); ?>
        </button>
        <span id="seo-reset-robots-status" style="margin-left: 10px;"></span>
        <p class="description">
            <?php esc_html_e('Restores robots.txt to default values.', 'seo-fury'); ?>
            <?php esc_html_e('All custom changes will be lost.', 'seo-fury'); ?>
        </p>
        <script>
        jQuery(document).ready(function($) {
            $('#seo-reset-robots').on('click', function() {
                if (!confirm('<?php echo esc_js(__('Are you sure? All robots.txt settings will be reset.', 'seo-fury')); ?>')) {
                    return;
                }
                var $btn = $(this);
                var $status = $('#seo-reset-robots-status');
                $btn.prop('disabled', true);
                $status.html('<span style="color: orange;"><?php echo esc_js(__('Resetting...', 'seo-fury')); ?></span>');
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'seo_reset_robots',
                        _ajax_nonce: '<?php echo wp_create_nonce('seo_reset_robots_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $status.html('<span style="color: green;"><?php echo esc_js(__('Reset complete!', 'seo-fury')); ?></span>');
                            setTimeout(function() { location.reload(); }, 800);
                        } else {
                            $status.html('<span style="color: red;"><?php echo esc_js(__('Error:', 'seo-fury')); ?> ' + (response.data || '<?php echo esc_js(__('unknown', 'seo-fury')); ?>') + '</span>');
                            $btn.prop('disabled', false);
                        }
                    },
                    error: function() {
                        $status.html('<span style="color: red;"><?php echo esc_js(__('AJAX error', 'seo-fury')); ?></span>');
                        $btn.prop('disabled', false);
                    }
                });
            });
        });
        </script>
        <?php
    }

    public function robots_link_callback() {
        $robots_url = home_url('/robots.txt');
        ?>
        <a href="<?php echo esc_url($robots_url); ?>" target="_blank" style="font-size: 14px; font-weight: 600;">
            <?php echo esc_html($robots_url); ?>
        </a>
        <?php
    }

    public function ajax_reset_robots() {
        check_ajax_referer('seo_reset_robots_nonce', '_ajax_nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions.', 'seo-fury'));
        }
        $default = $this->get_default_robots_template();
        update_option('my_seo_robots_content', $default);
        update_option('my_seo_robots_enabled', 1);
        update_option('my_seo_robots_user_agents', '');
        update_option('my_seo_robots_crawl_delay', '');
        update_option('my_seo_robots_use_dynamic_vars', 1);
        update_option('my_seo_robots_block_ai_bots', 0);
        update_option('my_seo_robots_block_archive_bots', 0);
        wp_send_json_success(__('Settings reset.', 'seo-fury'));
    }

    private function get_default_robots_template() {
        $template = <<<EOT
# Robots.txt for {site_url}

# General rules for all bots
User-agent: *
Disallow: /wp-admin/
Allow: /wp-admin/admin-ajax.php
Disallow: /wp-includes/
Disallow: /wp-content/plugins/
Disallow: /wp-content/themes/
Disallow: /trackback/
Disallow: /feed/
Disallow: /comments/
Disallow: /category/*/*
Disallow: */trackback
Disallow: */feed
Disallow: */comments
Disallow: /*?*
Disallow: /*?

# Rules for Yandex
User-agent: Yandex
Disallow: /wp-admin/
Allow: /wp-admin/admin-ajax.php
Disallow: /wp-includes/
Disallow: /trackback/
Disallow: /feed/
Disallow: /comments/
Disallow: */trackback
Disallow: */feed
Disallow: */comments
Clean-param: utm_source&utm_medium&utm_campaign

# Rules for Google
User-agent: Googlebot
Disallow: /wp-admin/
Allow: /wp-admin/admin-ajax.php
Disallow: /wp-includes/
Disallow: /trackback/
Disallow: /feed/
Disallow: /comments/
Disallow: */trackback
Disallow: */feed
Disallow: */comments
EOT;
        return $template;
    }

    private function apply_dynamic_vars($content) {
        $uploads = wp_upload_dir();
        $repl = array(
            '{site_url}'     => site_url('/'),
            '{home_url}'     => home_url('/'),
            '{uploads_path}' => isset($uploads['baseurl']) ? trailingslashit($uploads['baseurl']) : '',
        );
        return strtr($content, $repl);
    }

    private function build_crawl_delay_block() {
        $raw = get_option('my_seo_robots_crawl_delay', '');
        if (empty($raw)) {
            return '';
        }
        $lines = explode("\n", $raw);
        $out = '';
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line) || strpos($line, '=') === false) {
                continue;
            }
            list($agent, $delay) = array_map('trim', explode('=', $line, 2));
            if ($agent === '' || $delay === '') {
                continue;
            }
            $out .= "User-agent: {$agent}\nCrawl-delay: {$delay}\n\n";
        }
        return $out;
    }

    private function build_ai_bots_block() {
        $block_ai = get_option('my_seo_robots_block_ai_bots', 0);
        if (!$block_ai) {
            return '';
        }
        $ai_block = <<<EOT
# Block AI bots
User-agent: GPTBot
Disallow: /

User-agent: ChatGPT-User
Disallow: /

User-agent: Google-Extended
Disallow: /

User-agent: Omgilibot
Disallow: /

User-agent: ClaudeBot
Disallow: /

User-agent: Claude-Web
Disallow: /

EOT;
        return $ai_block;
    }

    private function build_archive_bots_block() {
        $block_archive = get_option('my_seo_robots_block_archive_bots', 0);
        if (!$block_archive) {
            return '';
        }
        $core = SEO_Fury_Core::get_instance();
        $pro = $core->get_pro_features();
        if (!$pro->is_pro()) {
            return '';
        }
        $archive_block = <<<EOT
# Block web archive bots
User-agent: ia_archiver
Disallow: /

User-agent: archive.org_bot
Disallow: /

User-agent: Wayback
Disallow: /

User-agent: archive-it
Disallow: /

User-agent: ArchiveBot
Disallow: /

User-agent: webcitation
Disallow: /

User-agent: Archivarix
Disallow: /

User-agent: Citoid
Disallow: /

EOT;
        return $archive_block;
    }

    private function build_sitemaps_block() {
        $sitemap_enabled = get_option('my_seo_sitemap_enabled', 1);
        if (!$sitemap_enabled) {
            return '';
        }
        $out = "\n# Sitemap\n";
        $xml_url = home_url('sitemap.xml');
        $out .= "Sitemap: {$xml_url}\n";
        return $out;
    }

    public function filter_robots_txt($output, $public) {
        if (is_admin()) {
            return $output;
        }
        $enabled = get_option('my_seo_robots_enabled', 1);
        if (!$enabled) {
            return $output;
        }
        $content = get_option('my_seo_robots_content', '');
        if (empty($content)) {
            $content = $this->get_default_robots_template();
        }
        if (get_option('my_seo_robots_use_dynamic_vars', 1)) {
            $content = $this->apply_dynamic_vars($content);
        }
        $content .= "\n" . $this->build_sitemaps_block();
        $ai_block = $this->build_ai_bots_block();
        if (!empty($ai_block)) {
            $content .= "\n" . $ai_block;
        }
        $archive_block = $this->build_archive_bots_block();
        if (!empty($archive_block)) {
            $content .= "\n" . $archive_block;
        }
        $content .= $this->build_crawl_delay_block();
        $ua_rules = get_option('my_seo_robots_user_agents', array());
        if (is_array($ua_rules) && !empty($ua_rules)) {
            $content .= "\n" . implode("\n", $ua_rules) . "\n";
        } elseif (is_string($ua_rules) && !empty($ua_rules)) {
            $content .= "\n" . $ua_rules . "\n";
        }
        return trim($content) . "\n";
    }
}
