<?php

if (!defined('ABSPATH')) {
    exit;
}

class My_SEO_Plugin_Sitemap {

    private $plugin_version = '1.0.0';

    public function __construct() {
        $this->init_hooks();
    }

private function init_hooks()
{
    add_action('init', array($this, 'register_sitemap_rewrite'));
    add_filter('query_vars', array($this, 'add_sitemap_query_var'));
    add_action('template_redirect', array($this, 'maybe_generate_sitemaps'));

    add_action('init', array($this, 'register_xsl_rewrite'));

    add_shortcode('my_seo_html_sitemap', array($this, 'shortcode_html_sitemap'));
}

    public function register_settings() {
        register_setting('my_seo_sitemap_group', 'my_seo_sitemap_enabled');
        register_setting('my_seo_sitemap_group', 'my_seo_sitemap_post_types');
        register_setting('my_seo_sitemap_group', 'my_seo_sitemap_post_types_settings');
        register_setting('my_seo_sitemap_group', 'my_seo_sitemap_include_images');
        register_setting('my_seo_sitemap_group', 'my_seo_sitemap_exclude_urls');
        register_setting('my_seo_sitemap_group', 'my_seo_sitemap_additional_urls');
        register_setting('my_seo_sitemap_group', 'my_seo_sitemap_additional_urls_data', array(
            'type' => 'array',
            'sanitize_callback' => array($this, 'sanitize_additional_urls_data'),
        ));
        register_setting('my_seo_sitemap_group', 'my_seo_sitemap_additional_default_priority');
        register_setting('my_seo_sitemap_group', 'my_seo_sitemap_additional_default_changefreq');
        register_setting('my_seo_sitemap_group', 'my_seo_html_sitemap_slug');
        register_setting('my_seo_sitemap_group', 'my_seo_html_sitemap_post_types');

        add_settings_section(
            'my_seo_sitemap_xml_section',
            __('XML Sitemap Settings', 'seo-fury'),
            array($this, 'sitemap_xml_section_callback'),
            'my_seo_sitemap_group'
        );

        add_settings_field(
            'my_seo_sitemap_enabled',
            __('Enable XML Sitemap', 'seo-fury'),
            array($this, 'sitemap_enabled_callback'),
            'my_seo_sitemap_group',
            'my_seo_sitemap_xml_section'
        );

        add_settings_field(
            'my_seo_sitemap_post_types',
            __('Post Types for XML Sitemap', 'seo-fury'),
            array($this, 'sitemap_post_types_callback'),
            'my_seo_sitemap_group',
            'my_seo_sitemap_xml_section'
        );

        add_settings_field(
            'my_seo_sitemap_include_images',
            __('Include Images', 'seo-fury'),
            array($this, 'sitemap_include_images_callback'),
            'my_seo_sitemap_group',
            'my_seo_sitemap_xml_section'
        );

        add_settings_field(
            'my_seo_sitemap_exclude_urls',
            __('Exclude URLs', 'seo-fury'),
            array($this, 'sitemap_exclude_urls_callback'),
            'my_seo_sitemap_group',
            'my_seo_sitemap_xml_section'
        );

        add_settings_field(
            'my_seo_sitemap_additional_urls',
            __('Additional URLs', 'seo-fury'),
            array($this, 'sitemap_additional_urls_callback'),
            'my_seo_sitemap_group',
            'my_seo_sitemap_xml_section'
        );

        add_settings_section(
            'my_seo_sitemap_html_section',
            __('HTML Sitemap Settings', 'seo-fury'),
            array($this, 'sitemap_html_section_callback'),
            'my_seo_sitemap_group'
        );

        add_settings_field(
            'my_seo_html_sitemap_slug',
            __('HTML Sitemap Slug', 'seo-fury'),
            array($this, 'html_sitemap_slug_callback'),
            'my_seo_sitemap_group',
            'my_seo_sitemap_html_section'
        );

        add_settings_field(
            'my_seo_html_sitemap_post_types',
            __('Post Types for HTML Sitemap', 'seo-fury'),
            array($this, 'html_sitemap_post_types_callback'),
            'my_seo_sitemap_group',
            'my_seo_sitemap_html_section'
        );
    }

    public function sitemap_xml_section_callback() {
        echo '<p>' . esc_html__('XML sitemap generation settings for search engines.', 'seo-fury') . '</p>';
    }

    public function sitemap_html_section_callback() {
        echo '<p>' . esc_html__('HTML sitemap settings for site visitors.', 'seo-fury') . '</p>';
    }

    public function sitemap_enabled_callback() {
        $enabled = get_option('my_seo_sitemap_enabled', 0);
        ?>
        <label>
            <input type="checkbox" name="my_seo_sitemap_enabled" value="1" <?php checked($enabled, 1); ?>>
            <?php esc_html_e('Enable XML sitemap generation', 'seo-fury'); ?>
        </label>
        <p class="description"><?php esc_html_e('Generates dynamic XML sitemap files for search engines.', 'seo-fury'); ?></p>
        <?php
    }

    public function sitemap_post_types_callback() {
        $selected = get_option('my_seo_sitemap_post_types', array('post', 'page'));
        if (!is_array($selected)) {
            $selected = array();
        }

        $post_types_settings = get_option('my_seo_sitemap_post_types_settings', array());
        if (!is_array($post_types_settings)) {
            $post_types_settings = array();
        }

        $post_types = get_post_types(array('public' => true), 'objects');

        echo '<p>' . esc_html__('Select post types to include in the XML sitemap and configure parameters for each type.', 'seo-fury') . '</p>';
        echo '<style>
            .sitemap-post-type-row {
                display: flex;
                align-items: center;
                margin-bottom: 10px;
                padding: 8px 0;
            }
            .sitemap-post-type-checkbox {
                flex: 0 0 200px;
            }
            .sitemap-post-type-settings {
                display: flex;
                gap: 20px;
                align-items: center;
            }
            .sitemap-setting-group {
                display: flex;
                align-items: center;
                gap: 5px;
            }
            .sitemap-setting-group label {
                font-weight: 500;
                font-size: 12px;
            }
        </style>';

        foreach ($post_types as $post_type) {
            if ($post_type->name === 'attachment') {
                continue;
            }

            $checked = in_array($post_type->name, $selected, true) ? 'checked' : '';

            $default_priority = '0.7';
            $default_changefreq = 'weekly';

            if ($post_type->name === 'page') {
                $default_priority = '0.8';
            }

            $priority = isset($post_types_settings[$post_type->name]['priority'])
                ? $post_types_settings[$post_type->name]['priority']
                : $default_priority;

            $changefreq = isset($post_types_settings[$post_type->name]['changefreq'])
                ? $post_types_settings[$post_type->name]['changefreq']
                : $default_changefreq;
            ?>
            <div class="sitemap-post-type-row">
                <div class="sitemap-post-type-checkbox">
                    <label>
                        <input type="checkbox"
                               name="my_seo_sitemap_post_types[]"
                               value="<?php echo esc_attr($post_type->name); ?>"
                               <?php echo $checked; ?>>
                        <strong><?php echo esc_html($post_type->label); ?></strong> (<?php echo esc_html($post_type->name); ?>)
                    </label>
                </div>

                <div class="sitemap-post-type-settings">
                    <div class="sitemap-setting-group">
                        <label><?php esc_html_e('Priority:', 'seo-fury'); ?></label>
                        <input type="number"
                               name="my_seo_sitemap_post_types_settings[<?php echo esc_attr($post_type->name); ?>][priority]"
                               value="<?php echo esc_attr($priority); ?>"
                               min="0"
                               max="1"
                               step="0.1"
                               style="width: 80px;">
                    </div>

                    <div class="sitemap-setting-group">
                        <label><?php esc_html_e('Changefreq:', 'seo-fury'); ?></label>
                        <select name="my_seo_sitemap_post_types_settings[<?php echo esc_attr($post_type->name); ?>][changefreq]"
                                style="width: 120px;">
                            <option value="always" <?php selected($changefreq, 'always'); ?>><?php esc_html_e('Always', 'seo-fury'); ?></option>
                            <option value="hourly" <?php selected($changefreq, 'hourly'); ?>><?php esc_html_e('Hourly', 'seo-fury'); ?></option>
                            <option value="daily" <?php selected($changefreq, 'daily'); ?>><?php esc_html_e('Daily', 'seo-fury'); ?></option>
                            <option value="weekly" <?php selected($changefreq, 'weekly'); ?>><?php esc_html_e('Weekly', 'seo-fury'); ?></option>
                            <option value="monthly" <?php selected($changefreq, 'monthly'); ?>><?php esc_html_e('Monthly', 'seo-fury'); ?></option>
                            <option value="yearly" <?php selected($changefreq, 'yearly'); ?>><?php esc_html_e('Yearly', 'seo-fury'); ?></option>
                            <option value="never" <?php selected($changefreq, 'never'); ?>><?php esc_html_e('Never', 'seo-fury'); ?></option>
                        </select>
                    </div>
                </div>
            </div>
            <?php
        }
    }

    public function sitemap_include_images_callback() {
        $include = get_option('my_seo_sitemap_include_images', 0);
        ?>
        <label>
            <input type="checkbox" name="my_seo_sitemap_include_images" value="1" <?php checked($include, 1); ?>>
            <?php esc_html_e('Include images in sitemap', 'seo-fury'); ?>
        </label>
        <?php
    }

    public function sitemap_exclude_urls_callback() {
        $exclude = get_option('my_seo_sitemap_exclude_urls', '');
        ?>
        <textarea name="my_seo_sitemap_exclude_urls" rows="5" class="large-text"><?php echo esc_textarea($exclude); ?></textarea>
        <p class="description"><?php esc_html_e('Full or relative URLs to exclude from the XML sitemap (one per line).', 'seo-fury'); ?></p>
        <?php
    }

    public function sitemap_additional_urls_callback() {
        $additional = get_option('my_seo_sitemap_additional_urls', '');
        $default_priority = get_option('my_seo_sitemap_additional_default_priority', '0.5');
        $default_changefreq = get_option('my_seo_sitemap_additional_default_changefreq', 'weekly');
        $urls_data = get_option('my_seo_sitemap_additional_urls_data', array());
        if (!is_array($urls_data)) $urls_data = array();

        $core = SEO_Fury_Core::get_instance();
        $pro = $core->get_pro_features();
        $is_pro = $pro && $pro->is_pro();
        $hide_branding = function_exists('seo_fury_hide_branding') ? seo_fury_hide_branding() : false;
        ?>

        <!-- Default settings for all additional URLs -->
        <div style="margin-bottom: 20px;">
            <p class="description" style="margin-bottom: 10px;"><?php esc_html_e('Default priority and change frequency for all additional URLs (unless overridden individually below).', 'seo-fury'); ?></p>
            <div style="display: flex; align-items: center; gap: 20px; flex-wrap: wrap;">
                <div class="sitemap-setting-group">
                    <label><?php esc_html_e('Default Priority:', 'seo-fury'); ?></label>
                    <input type="number"
                           name="my_seo_sitemap_additional_default_priority"
                           value="<?php echo esc_attr($default_priority); ?>"
                           min="0" max="1" step="0.1"
                           style="width: 80px;">
                </div>
                <div class="sitemap-setting-group">
                    <label><?php esc_html_e('Default Changefreq:', 'seo-fury'); ?></label>
                    <select name="my_seo_sitemap_additional_default_changefreq" style="width: 120px;">
                        <option value="always" <?php selected($default_changefreq, 'always'); ?>><?php esc_html_e('Always', 'seo-fury'); ?></option>
                        <option value="hourly" <?php selected($default_changefreq, 'hourly'); ?>><?php esc_html_e('Hourly', 'seo-fury'); ?></option>
                        <option value="daily" <?php selected($default_changefreq, 'daily'); ?>><?php esc_html_e('Daily', 'seo-fury'); ?></option>
                        <option value="weekly" <?php selected($default_changefreq, 'weekly'); ?>><?php esc_html_e('Weekly', 'seo-fury'); ?></option>
                        <option value="monthly" <?php selected($default_changefreq, 'monthly'); ?>><?php esc_html_e('Monthly', 'seo-fury'); ?></option>
                        <option value="yearly" <?php selected($default_changefreq, 'yearly'); ?>><?php esc_html_e('Yearly', 'seo-fury'); ?></option>
                        <option value="never" <?php selected($default_changefreq, 'never'); ?>><?php esc_html_e('Never', 'seo-fury'); ?></option>
                    </select>
                </div>
            </div>
        </div>

        <!-- Simple textarea for basic URLs (FREE) -->
        <textarea name="my_seo_sitemap_additional_urls" rows="5" class="large-text"><?php echo esc_textarea($additional); ?></textarea>
        <p class="description"><?php esc_html_e('Additional URLs to add to the XML sitemap (one per line). These will use the default priority and changefreq above.', 'seo-fury'); ?></p>

        <!-- Per-URL settings table (PRO) -->
        <div style="margin-top: 25px; padding-top: 20px; border-top: 1px solid #e2e8f0;">
            <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 12px;">
                <h4 style="margin: 0; font-size: 14px;">
                    <?php esc_html_e('Per-URL Priority & Changefreq', 'seo-fury'); ?>
                    <?php if (!$is_pro): ?>
                        <span style="background: linear-gradient(135deg, #4F46E5, #7C3AED); color: #fff; font-size: 10px; padding: 2px 8px; border-radius: 10px; margin-left: 8px; font-weight: 600;">PRO</span>
                    <?php endif; ?>
                </h4>
                <?php if ($is_pro): ?>
                    <button type="button" id="sf-add-additional-url" class="button button-small" style="margin-left: auto;"><?php esc_html_e('+ Add URL', 'seo-fury'); ?></button>
                <?php endif; ?>
            </div>

            <?php if (!$is_pro): ?>
                <div style="background: #f8fafc; border: 1px dashed #cbd5e1; border-radius: 8px; padding: 20px; text-align: center; color: #64748b;">
                    <p style="margin: 0 0 8px; font-size: 14px;"><strong><?php esc_html_e('Override priority & changefreq per URL', 'seo-fury'); ?></strong></p>
                    <p style="margin: 0 0 12px; font-size: 13px;"><?php esc_html_e('Upgrade to PRO to set individual priority and change frequency for each additional URL in your sitemap.', 'seo-fury'); ?></p>
                    <?php if (!$hide_branding): ?>
                        <a href="https://seo-fury.com/#pricing" target="_blank" class="button button-primary button-small"><?php esc_html_e('Upgrade to PRO', 'seo-fury'); ?></a>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped" id="sf-additional-urls-table" style="margin-top: 5px;">
                    <thead>
                        <tr>
                            <th style="width: 45%;"><?php esc_html_e('URL', 'seo-fury'); ?></th>
                            <th style="width: 15%;"><?php esc_html_e('Priority', 'seo-fury'); ?></th>
                            <th style="width: 20%;"><?php esc_html_e('Changefreq', 'seo-fury'); ?></th>
                            <th style="width: 20%;"><?php esc_html_e('Actions', 'seo-fury'); ?></th>
                        </tr>
                    </thead>
                    <tbody id="sf-additional-urls-body">
                        <?php if (!empty($urls_data)): ?>
                            <?php foreach ($urls_data as $i => $row): ?>
                                <tr data-index="<?php echo $i; ?>">
                                    <td>
                                        <input type="url" name="my_seo_sitemap_additional_urls_data[<?php echo $i; ?>][url]"
                                               value="<?php echo esc_attr($row['url'] ?? ''); ?>"
                                               class="regular-text" style="width:100%;" placeholder="https://" required>
                                    </td>
                                    <td>
                                        <input type="number" name="my_seo_sitemap_additional_urls_data[<?php echo $i; ?>][priority]"
                                               value="<?php echo esc_attr($row['priority'] ?? '0.5'); ?>"
                                               min="0" max="1" step="0.1" style="width: 70px;">
                                    </td>
                                    <td>
                                        <select name="my_seo_sitemap_additional_urls_data[<?php echo $i; ?>][changefreq]" style="width: 110px;">
                                            <?php
                                            $freqs = array(
                                                'always'  => __('Always', 'seo-fury'),
                                                'hourly'  => __('Hourly', 'seo-fury'),
                                                'daily'   => __('Daily', 'seo-fury'),
                                                'weekly'  => __('Weekly', 'seo-fury'),
                                                'monthly' => __('Monthly', 'seo-fury'),
                                                'yearly'  => __('Yearly', 'seo-fury'),
                                                'never'   => __('Never', 'seo-fury'),
                                            );
                                            $cur_freq = $row['changefreq'] ?? 'weekly';
                                            foreach ($freqs as $f => $f_label):
                                            ?>
                                                <option value="<?php echo esc_attr($f); ?>" <?php selected($cur_freq, $f); ?>><?php echo esc_html($f_label); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    </td>
                                    <td>
                                        <button type="button" class="button button-small sf-remove-url-row" style="color:#d63638;"><?php esc_html_e('Remove', 'seo-fury'); ?></button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr id="sf-no-urls-row">
                                <td colspan="4" style="text-align:center; color:#94a3b8; padding:15px;"><?php esc_html_e('No per-URL overrides yet. Click "+ Add URL" to add one.', 'seo-fury'); ?></td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>

                <script>
                (function(){
                    var idx = <?php echo !empty($urls_data) ? max(array_keys($urls_data)) + 1 : 0; ?>;
                    var freqs = ['always','hourly','daily','weekly','monthly','yearly','never'];
                    var removeLabel = <?php echo wp_json_encode(__('Remove', 'seo-fury')); ?>;
                    var emptyMessage = <?php echo wp_json_encode(__('No per-URL overrides yet. Click "+ Add URL" to add one.', 'seo-fury')); ?>;

                    document.getElementById('sf-add-additional-url').addEventListener('click', function(){
                        var tbody = document.getElementById('sf-additional-urls-body');
                        var noRow = document.getElementById('sf-no-urls-row');
                        if (noRow) noRow.remove();

                        var tr = document.createElement('tr');
                        tr.setAttribute('data-index', idx);

                        var optsHtml = '';
                        for (var fi = 0; fi < freqs.length; fi++) {
                            var sel = (freqs[fi] === 'weekly') ? ' selected' : '';
                            optsHtml += '<option value="' + freqs[fi] + '"' + sel + '>' + freqs[fi].charAt(0).toUpperCase() + freqs[fi].slice(1) + '</option>';
                        }

                        tr.innerHTML =
                            '<td><input type="url" name="my_seo_sitemap_additional_urls_data[' + idx + '][url]" class="regular-text" style="width:100%;" placeholder="https://" required></td>' +
                            '<td><input type="number" name="my_seo_sitemap_additional_urls_data[' + idx + '][priority]" value="0.5" min="0" max="1" step="0.1" style="width:70px;"></td>' +
                            '<td><select name="my_seo_sitemap_additional_urls_data[' + idx + '][changefreq]" style="width:110px;">' + optsHtml + '</select></td>' +
                            '<td><button type="button" class="button button-small sf-remove-url-row" style="color:#d63638;">' + removeLabel + '</button></td>';

                        tbody.appendChild(tr);
                        idx++;
                    });

                    document.addEventListener('click', function(e){
                        if (e.target.classList.contains('sf-remove-url-row')) {
                            var row = e.target.closest('tr');
                            if (row) row.remove();
                            var tbody = document.getElementById('sf-additional-urls-body');
                            if (tbody && tbody.querySelectorAll('tr').length === 0) {
                                var emptyTr = document.createElement('tr');
                                emptyTr.id = 'sf-no-urls-row';
                                emptyTr.innerHTML = '<td colspan="4" style="text-align:center;color:#94a3b8;padding:15px;">' + emptyMessage + '</td>';
                                tbody.appendChild(emptyTr);
                            }
                        }
                    });
                })();
                </script>
            <?php endif; ?>
        </div>
        <?php
    }

    public function sanitize_additional_urls_data($input) {
        if (!is_array($input)) return array();
        $clean = array();
        foreach ($input as $row) {
            $url = isset($row['url']) ? esc_url_raw(trim($row['url'])) : '';
            if (empty($url)) continue;
            $priority = isset($row['priority']) ? floatval($row['priority']) : 0.5;
            $priority = max(0, min(1, $priority));
            $valid_freqs = array('always','hourly','daily','weekly','monthly','yearly','never');
            $changefreq = (isset($row['changefreq']) && in_array($row['changefreq'], $valid_freqs)) ? $row['changefreq'] : 'weekly';
            $clean[] = array(
                'url' => $url,
                'priority' => round($priority, 1),
                'changefreq' => $changefreq,
            );
        }
        return $clean;
    }

    public function html_sitemap_slug_callback() {
        $slug = get_option('my_seo_html_sitemap_slug', 'html-sitemap');
        ?>
        <input type="text" name="my_seo_html_sitemap_slug" value="<?php echo esc_attr($slug); ?>" class="regular-text">
        <span class="description" style="margin-left: 10px;"><?php esc_html_e('HTML sitemap slug (e.g., html-sitemap)', 'seo-fury'); ?></span>
        <?php
    }

    public function html_sitemap_post_types_callback() {
        $selected = get_option('my_seo_html_sitemap_post_types', array('post', 'page'));
        if (!is_array($selected)) {
            $selected = array();
        }

        $post_types = get_post_types(array('public' => true), 'objects');

        foreach ($post_types as $post_type) {
            if ($post_type->name === 'attachment') {
                continue;
            }

            $checked = in_array($post_type->name, $selected, true) ? 'checked' : '';
            ?>
            <label style="display: block; margin-bottom: 5px;">
                <input type="checkbox" name="my_seo_html_sitemap_post_types[]" value="<?php echo esc_attr($post_type->name); ?>" <?php echo $checked; ?>>
                <?php echo esc_html($post_type->label); ?>
            </label>
            <?php
        }

        echo '<p class="description" style="margin-top: 10px;">' . esc_html__('Select which post types to show in the HTML sitemap.', 'seo-fury') . '</p>';
    }

    public function register_sitemap_rewrite() {
        add_rewrite_rule(
            '^sitemap\.xml$',
            'index.php?my_seo_sitemap=index',
            'top'
        );

        add_rewrite_rule(
            '^sitemap-([a-z0-9_-]+)\.xml$',
            'index.php?my_seo_sitemap=$matches[1]',
            'top'
        );

        $html_slug = get_option('my_seo_html_sitemap_slug', 'html-sitemap');
        add_rewrite_rule(
            '^' . $html_slug . '/?$',
            'index.php?my_seo_sitemap=html',
            'top'
        );
    }

public function register_xsl_rewrite() {
    add_rewrite_rule(
        '^sitemap\.xsl$',
        'index.php?my_seo_sitemap_xsl=1',
        'top'
    );

    add_filter('query_vars', array($this, 'add_xsl_query_var'));

    add_action('template_redirect', array($this, 'maybe_serve_xsl'));
}

    public function maybe_serve_xsl() {
        if (get_query_var('my_seo_sitemap_xsl')) {
            header('Content-Type: application/xml; charset=utf-8');
            echo $this->get_xsl_stylesheet();
            exit;
        }
    }

public function add_xsl_query_var($vars) {
    if (wp_doing_ajax() || (defined('REST_REQUEST') && REST_REQUEST)) {
        return $vars;
    }

    $vars[] = 'my_seo_sitemap_xsl';
    return $vars;
}

public function add_sitemap_query_var($vars)
{
    if ($this->is_rest_request()) {
        return $vars;
    }

    $vars[] = 'my_seo_sitemap';
    return $vars;
}

private function is_rest_request()
{
    if (defined('REST_REQUEST') && REST_REQUEST) {
        return true;
    }

    if (isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], '/wp-json/') !== false) {
        return true;
    }

    return false;
}

    public function maybe_generate_sitemaps() {
        $sitemap_type = get_query_var('my_seo_sitemap');

        if ($sitemap_type) {
            if ($sitemap_type === 'index') {
                $this->generate_sitemap_index();
                exit;
            } elseif ($sitemap_type === 'html') {
                $this->generate_html_sitemap();
                exit;
            } else {
                $this->generate_xml_sitemap_by_type($sitemap_type);
                exit;
            }
        }
    }

    public function generate_sitemap_index() {
        $enabled = get_option('my_seo_sitemap_enabled', 0);
        if (!$enabled) {
            status_header(404);
            return;
        }

        $post_types = get_option('my_seo_sitemap_post_types', array('post', 'page'));
        if (!is_array($post_types) || empty($post_types)) {
            status_header(404);
            return;
        }

        header('Content-Type: application/xml; charset=utf-8');

        echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        echo '<?xml-stylesheet type="text/xsl" href="' . home_url('/sitemap.xsl') . '"?>' . "\n";
        echo '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        foreach ($post_types as $post_type) {
            $post_type_obj = get_post_type_object($post_type);
            if (!$post_type_obj) {
                continue;
            }

            $latest_post = get_posts(array(
                'post_type' => $post_type,
                'post_status' => 'publish',
                'posts_per_page' => 1,
                'orderby' => 'modified',
                'order' => 'DESC'
            ));

            $lastmod = !empty($latest_post) ? get_post_modified_time('c', false, $latest_post[0]) : current_time('c');

            echo '  <sitemap>' . "\n";
            echo '    <loc>' . esc_url(home_url("/sitemap-{$post_type}.xml")) . '</loc>' . "\n";
            echo '    <lastmod>' . esc_html($lastmod) . '</lastmod>' . "\n";
            echo '  </sitemap>' . "\n";
        }

        echo '</sitemapindex>';
        exit;
    }

    public function generate_xml_sitemap_by_type($post_type) {
        $enabled = get_option('my_seo_sitemap_enabled', 0);
        if (!$enabled) {
            status_header(404);
            return;
        }

        $allowed_types = get_option('my_seo_sitemap_post_types', array('post', 'page'));
        if (!in_array($post_type, $allowed_types, true)) {
            status_header(404);
            return;
        }

        $exclude_urls = $this->get_exclude_urls();
        $include_images = get_option('my_seo_sitemap_include_images', 0);

        $args = array(
            'post_type' => $post_type,
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'orderby' => 'modified',
            'order' => 'DESC',
            'no_found_rows' => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
        );

        $posts = get_posts($args);

        header('Content-Type: application/xml; charset=utf-8');

        echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        echo '<?xml-stylesheet type="text/xsl" href="' . home_url('/sitemap.xsl') . '?type=' . urlencode($post_type) . '"?>' . "\n";
        echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"';
        if ($include_images) {
            echo ' xmlns:image="http://www.google.com/schemas/sitemap-image/1.1"';
        }
        echo '>' . "\n";

        $priority = $this->get_priority($post_type);
        $changefreq = $this->get_changefreq($post_type);

        foreach ($posts as $post) {
            $permalink = get_permalink($post);

            if (in_array($permalink, $exclude_urls, true)) {
                continue;
            }

            $lastmod = get_post_modified_time('c', false, $post);

            echo '  <url>' . "\n";
            echo '    <loc>' . esc_url($permalink) . '</loc>' . "\n";
            echo '    <lastmod>' . esc_html($lastmod) . '</lastmod>' . "\n";
            echo '    <changefreq>' . esc_html($changefreq) . '</changefreq>' . "\n";
            echo '    <priority>' . esc_html($priority) . '</priority>' . "\n";

            if ($include_images) {
                $images = $this->get_post_images($post);
                foreach ($images as $image_url) {
                    echo '    <image:image>' . "\n";
                    echo '      <image:loc>' . esc_url($image_url) . '</image:loc>' . "\n";
                    echo '    </image:image>' . "\n";
                }
            }

            echo '  </url>' . "\n";
        }

        if ($post_type === 'page') {
            $additional_urls = $this->get_additional_urls();
            foreach ($additional_urls as $url_entry) {
                echo '  <url>' . "\n";
                echo '    <loc>' . esc_url($url_entry['url']) . '</loc>' . "\n";
                echo '    <changefreq>' . esc_html($url_entry['changefreq']) . '</changefreq>' . "\n";
                echo '    <priority>' . esc_html($url_entry['priority']) . '</priority>' . "\n";
                echo '  </url>' . "\n";
            }
        }

        echo '</urlset>';
        exit;
    }

    private function get_xsl_stylesheet() {
        $home_url = esc_url(home_url('/sitemap.xml'));

        $post_type = isset($_GET['type']) ? sanitize_text_field($_GET['type']) : '';
        $post_type_label = '';
        if ($post_type) {
            $post_type_obj = get_post_type_object($post_type);
            if ($post_type_obj) {
                $post_type_label = ' - ' . $post_type_obj->label . ' (' . $post_type . ')';
            }
        }

        return '<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="2.0"
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:sitemap="http://www.sitemaps.org/schemas/sitemap/0.9"
    xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">

    <xsl:output method="html" version="1.0" encoding="UTF-8" indent="yes"/>

    <xsl:template match="/">
        <html xmlns="http://www.w3.org/1999/xhtml">
            <head>
                <title>' . esc_html__('XML Sitemap', 'seo-fury') . '</title>
                <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
                <style type="text/css">
                    body {
                        font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
                        font-size: 14px;
                        color: #333;
                        margin: 0;
                        padding: 20px;
                        background: #f5f5f5;
                    }
                    .container {
                        max-width: 1200px;
                        margin: 0 auto;
                        background: white;
                        padding: 30px;
                        border-radius: 8px;
                        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
                        position: relative;
                    }
                    .plugin-badge {
                        position: absolute;
                        top: 20px;
                        right: 20px;
                        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                        color: white;
                        padding: 10px 20px;
                        border-radius: 20px;
                        font-size: 12px;
                        font-weight: 600;
                        box-shadow: 0 2px 8px rgba(102, 126, 234, 0.4);
                        white-space: nowrap;
                    }
                    h1 {
                        font-size: 24px;
                        margin: 0 0 10px 0;
                        color: #1a1a1a;
                        padding-right: 280px;
                    }
                    .description {
                        color: #666;
                        margin-bottom: 20px;
                        line-height: 1.6;
                    }
                    .back-link {
                        display: inline-block;
                        margin-bottom: 20px;
                        color: #0073aa;
                        text-decoration: none;
                        font-weight: 500;
                    }
                    .back-link:hover {
                        text-decoration: underline;
                    }
                    .tabs {
                        display: flex;
                        gap: 10px;
                        margin-bottom: 20px;
                        border-bottom: 2px solid #e9ecef;
                    }
                    .tab {
                        padding: 12px 20px;
                        cursor: pointer;
                        border: none;
                        background: none;
                        font-size: 14px;
                        font-weight: 500;
                        color: #666;
                        border-bottom: 3px solid transparent;
                        margin-bottom: -2px;
                        transition: all 0.2s;
                    }
                    .tab:hover {
                        color: #0073aa;
                    }
                    .tab.active {
                        color: #0073aa;
                        border-bottom-color: #0073aa;
                    }
                    .tab-content {
                        display: none;
                    }
                    .tab-content.active {
                        display: block;
                    }
                    table {
                        width: 100%;
                        border-collapse: collapse;
                        margin-top: 20px;
                    }
                    th {
                        background: #f8f9fa;
                        padding: 12px 15px;
                        text-align: left;
                        font-weight: 600;
                        border-bottom: 2px solid #dee2e6;
                        color: #495057;
                    }
                    td {
                        padding: 12px 15px;
                        border-bottom: 1px solid #e9ecef;
                    }
                    tr:hover {
                        background: #f8f9fa;
                    }
                    a {
                        color: #0073aa;
                        text-decoration: none;
                    }
                    a:hover {
                        text-decoration: underline;
                    }
                    .url-cell {
                        max-width: 500px;
                        word-break: break-all;
                    }
                    .stats {
                        background: #f8f9fa;
                        padding: 15px;
                        border-radius: 4px;
                        margin-bottom: 20px;
                    }
                    .stats strong {
                        color: #1a1a1a;
                    }
                    .image-preview {
                        max-width: 100px;
                        max-height: 60px;
                        border-radius: 4px;
                        border: 1px solid #ddd;
                    }
                </style>
            </head>
            <body>
                <div class="container">
                    <div class="plugin-badge">' . esc_html__('Sitemap', 'seo-fury') . ' · My SEO Plugin</div>

                    <xsl:choose>
                        <!-- Sitemap index -->
                        <xsl:when test="sitemap:sitemapindex">
                            <h1>' . esc_html__('XML Sitemap Index', 'seo-fury') . '</h1>
                            <div class="description">
                                ' . esc_html__('This is the XML sitemap index. Each link below points to a sitemap for a specific content type.', 'seo-fury') . '
                            </div>
                            <div class="stats">
                                <strong>' . esc_html__('Total sitemap files:', 'seo-fury') . ' </strong> <xsl:value-of select="count(sitemap:sitemapindex/sitemap:sitemap)"/>
                            </div>

                            <table>
                                <tr>
                                    <th>#</th>
                                    <th>' . esc_html__('URL', 'seo-fury') . '</th>
                                    <th>' . esc_html__('Last Modified', 'seo-fury') . '</th>
                                </tr>
                                <xsl:for-each select="sitemap:sitemapindex/sitemap:sitemap">
                                    <tr>
                                        <td><xsl:value-of select="position()"/></td>
                                        <td class="url-cell">
                                            <a href="{sitemap:loc}">
                                                <xsl:value-of select="sitemap:loc"/>
                                            </a>
                                        </td>
                                        <td><xsl:value-of select="sitemap:lastmod"/></td>
                                    </tr>
                                </xsl:for-each>
                            </table>
                        </xsl:when>

                        <!-- URL Sitemap -->
                        <xsl:otherwise>
                            <a href="' . $home_url . '" class="back-link">&larr; ' . esc_html__('Back to sitemap index', 'seo-fury') . '</a>

                            <h1>' . esc_html__('XML Sitemap', 'seo-fury') . esc_html($post_type_label) . '</h1>
                            <div class="description">
                                ' . esc_html__('This XML sitemap is intended for search engines such as Google, Bing, and Yandex.', 'seo-fury') . '
                            </div>

                            <div class="stats">
                                <strong>' . esc_html__('Total URLs:', 'seo-fury') . ' </strong> <xsl:value-of select="count(sitemap:urlset/sitemap:url)"/>
                            </div>

                            <!-- Tabs -->
                            <div class="tabs">
                                <button class="tab active" onclick="showTab(\'urls\')">' . esc_html__('URLs', 'seo-fury') . '</button>
                                <xsl:if test="count(sitemap:urlset/sitemap:url/image:image) &gt; 0">
                                    <button class="tab" onclick="showTab(\'images\')">
                                        ' . esc_html__('Images', 'seo-fury') . ' (<xsl:value-of select="count(sitemap:urlset/sitemap:url/image:image)"/>)
                                    </button>
                                </xsl:if>
                            </div>

                            <!-- URL tab -->
                            <div id="urls-tab" class="tab-content active">
                                <table>
                                    <tr>
                                        <th>#</th>
                                        <th>' . esc_html__('URL', 'seo-fury') . '</th>
                                        <th>' . esc_html__('Changefreq', 'seo-fury') . '</th>
                                        <th>' . esc_html__('Priority', 'seo-fury') . '</th>
                                        <th>' . esc_html__('Last Modified', 'seo-fury') . '</th>
                                    </tr>
                                    <xsl:for-each select="sitemap:urlset/sitemap:url">
                                        <tr>
                                            <td><xsl:value-of select="position()"/></td>
                                            <td class="url-cell">
                                                <a href="{sitemap:loc}">
                                                    <xsl:value-of select="sitemap:loc"/>
                                                </a>
                                            </td>
                                            <td><xsl:value-of select="sitemap:changefreq"/></td>
                                            <td><xsl:value-of select="sitemap:priority"/></td>
                                            <td><xsl:value-of select="sitemap:lastmod"/></td>
                                        </tr>
                                    </xsl:for-each>
                                </table>
                            </div>

                            <!-- Images tab -->
                            <xsl:if test="count(sitemap:urlset/sitemap:url/image:image) &gt; 0">
                                <div id="images-tab" class="tab-content">
                                    <table>
                                        <tr>
                                            <th>#</th>
                                            <th>' . esc_html__('Page', 'seo-fury') . '</th>
                                            <th>' . esc_html__('Image', 'seo-fury') . '</th>
                                            <th>' . esc_html__('Preview', 'seo-fury') . '</th>
                                        </tr>
                                        <xsl:for-each select="sitemap:urlset/sitemap:url">
                                            <xsl:variable name="pageUrl" select="sitemap:loc"/>
                                            <xsl:for-each select="image:image">
                                                <tr>
                                                    <td><xsl:value-of select="position()"/></td>
                                                    <td class="url-cell">
                                                        <a href="{$pageUrl}">
                                                            <xsl:value-of select="$pageUrl"/>
                                                        </a>
                                                    </td>
                                                    <td class="url-cell">
                                                        <a href="{image:loc}" target="_blank">
                                                            <xsl:value-of select="image:loc"/>
                                                        </a>
                                                    </td>
                                                    <td>
                                                        <img src="{image:loc}" alt="' . esc_attr__('Image', 'seo-fury') . '" class="image-preview"/>
                                                    </td>
                                                </tr>
                                            </xsl:for-each>
                                        </xsl:for-each>
                                    </table>
                                </div>
                            </xsl:if>
                        </xsl:otherwise>
                    </xsl:choose>
                </div>

                <script type="text/javascript">
                    function showTab(tabName) {
                        var tabs = document.querySelectorAll(\'.tab-content\');
                        tabs.forEach(function(tab) {
                            tab.classList.remove(\'active\');
                        });

                        var buttons = document.querySelectorAll(\'.tab\');
                        buttons.forEach(function(btn) {
                            btn.classList.remove(\'active\');
                        });

                        document.getElementById(tabName + \'-tab\').classList.add(\'active\');
                        event.target.classList.add(\'active\');
                    }
                </script>
            </body>
        </html>
    </xsl:template>
</xsl:stylesheet>';
    }

    private function get_exclude_urls() {
        $exclude = get_option('my_seo_sitemap_exclude_urls', '');
        $urls = array_filter(array_map('trim', explode("\n", $exclude)));

        $normalized = array();
        foreach ($urls as $url) {
            if (strpos($url, 'http') === 0) {
                $normalized[] = $url;
            } else {
                $normalized[] = home_url($url);
            }
        }

        return $normalized;
    }

    private function get_additional_urls() {
        $default_priority = get_option('my_seo_sitemap_additional_default_priority', '0.5');
        $default_changefreq = get_option('my_seo_sitemap_additional_default_changefreq', 'weekly');

        $urls_data = get_option('my_seo_sitemap_additional_urls_data', array());
        $overrides = array();
        if (is_array($urls_data)) {
            foreach ($urls_data as $row) {
                if (!empty($row['url'])) {
                    $overrides[$row['url']] = array(
                        'priority'   => isset($row['priority']) ? round(floatval($row['priority']), 1) : floatval($default_priority),
                        'changefreq' => isset($row['changefreq']) ? $row['changefreq'] : $default_changefreq,
                    );
                }
            }
        }

        $additional = get_option('my_seo_sitemap_additional_urls', '');
        $raw_urls = array_filter(array_map('trim', explode("\n", $additional)));

        $result = array();
        $seen = array();

        foreach ($raw_urls as $url) {
            $normalized = (strpos($url, 'http') === 0) ? $url : home_url($url);
            if (isset($seen[$normalized])) continue;
            $seen[$normalized] = true;

            if (isset($overrides[$normalized])) {
                $result[] = array(
                    'url'        => $normalized,
                    'priority'   => $overrides[$normalized]['priority'],
                    'changefreq' => $overrides[$normalized]['changefreq'],
                );
                unset($overrides[$normalized]);
            } else {
                $result[] = array(
                    'url'        => $normalized,
                    'priority'   => floatval($default_priority),
                    'changefreq' => $default_changefreq,
                );
            }
        }

        foreach ($overrides as $url => $settings) {
            if (isset($seen[$url])) continue;
            $result[] = array(
                'url'        => $url,
                'priority'   => $settings['priority'],
                'changefreq' => $settings['changefreq'],
            );
        }

        return $result;
    }

    private function get_priority($post_type) {
        $post_types_settings = get_option('my_seo_sitemap_post_types_settings', array());

        if (isset($post_types_settings[$post_type]['priority'])) {
            return $post_types_settings[$post_type]['priority'];
        }

        $priorities = array(
            'page' => '0.8',
            'post' => '0.7',
        );

        return isset($priorities[$post_type]) ? $priorities[$post_type] : '0.5';
    }

    private function get_changefreq($post_type) {
        $post_types_settings = get_option('my_seo_sitemap_post_types_settings', array());

        if (isset($post_types_settings[$post_type]['changefreq'])) {
            return $post_types_settings[$post_type]['changefreq'];
        }

        return 'weekly';
    }

    private function get_post_images($post) {
        $images = array();

        if (has_post_thumbnail($post->ID)) {
            $thumbnail_id = get_post_thumbnail_id($post->ID);
            $image_url = wp_get_attachment_image_url($thumbnail_id, 'full');
            if ($image_url) {
                $images[] = $image_url;
            }
        }

        preg_match_all('/<img[^>]+src=["\']([^"\']+)["\'][^>]*>/i', $post->post_content, $matches);
        if (!empty($matches[1])) {
            foreach ($matches[1] as $img_url) {
                if (!in_array($img_url, $images, true)) {
                    $images[] = $img_url;
                }
            }
        }

        return array_slice($images, 0, 10);
    }

    public function generate_html_sitemap() {
        $post_types = get_option('my_seo_html_sitemap_post_types', array('post', 'page'));

        if (!is_array($post_types) || empty($post_types)) {
            status_header(404);
            return;
        }

        get_header();
        ?>
        <div class="html-sitemap-container" style="padding: 30px 15px; max-width: 1200px; margin: 0 auto; font-size: 0.75em;">
            <h1 style="margin-bottom: 22px; font-size: 2em;"><?php esc_html_e('Sitemap', 'seo-fury'); ?></h1>

            <?php foreach ($post_types as $post_type): ?>
                <?php
                $post_type_obj = get_post_type_object($post_type);
                if (!$post_type_obj) {
                    continue;
                }

                $posts = get_posts(array(
                    'post_type' => $post_type,
                    'post_status' => 'publish',
                    'posts_per_page' => -1,
                    'orderby' => 'title',
                    'order' => 'ASC',
                ));

                if (empty($posts)) {
                    continue;
                }
                ?>

                <div class="sitemap-section" style="margin-bottom: 30px;">
                    <h2 style="border-bottom: 2px solid #0073aa; padding-bottom: 8px; margin-bottom: 15px; font-size: 1.6em;">
                        <?php echo esc_html($post_type_obj->label); ?>
                    </h2>

                    <ul style="list-style: none; padding: 0; columns: 2; column-gap: 30px;">
                        <?php foreach ($posts as $post): ?>
                            <li style="margin-bottom: 8px; break-inside: avoid;">
                                <a href="<?php echo esc_url(get_permalink($post)); ?>" style="color: #0073aa; text-decoration: none;">
                                    <?php echo esc_html($post->post_title); ?>
                                </a>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endforeach; ?>

            <?php if (empty($post_types)): ?>
                <p><?php esc_html_e('No entries available to display in the sitemap.', 'seo-fury'); ?></p>
            <?php endif; ?>
        </div>
        <?php
        get_footer();
        exit;
    }

    public function shortcode_html_sitemap($atts) {
        $atts = shortcode_atts(array(
            'post_types' => '',
        ), $atts);

        if (!empty($atts['post_types'])) {
            $post_types = array_map('trim', explode(',', $atts['post_types']));
        } else {
            $post_types = get_option('my_seo_html_sitemap_post_types', array('post', 'page'));
        }

        if (!is_array($post_types) || empty($post_types)) {
            return '<p>' . esc_html__('No entries available to display in the sitemap.', 'seo-fury') . '</p>';
        }

        ob_start();
        ?>
        <div class="html-sitemap-shortcode" style="font-size: 0.75em;">
            <?php foreach ($post_types as $post_type): ?>
                <?php
                $post_type_obj = get_post_type_object($post_type);
                if (!$post_type_obj) {
                    continue;
                }

                $posts = get_posts(array(
                    'post_type' => $post_type,
                    'post_status' => 'publish',
                    'posts_per_page' => -1,
                    'orderby' => 'title',
                    'order' => 'ASC',
                ));

                if (empty($posts)) {
                    continue;
                }
                ?>

                <div class="sitemap-section" style="margin-bottom: 22px;">
                    <h3 style="font-size: 1.4em;"><?php echo esc_html($post_type_obj->label); ?></h3>
                    <ul>
                        <?php foreach ($posts as $post): ?>
                            <li style="margin-bottom: 6px;">
                                <a href="<?php echo esc_url(get_permalink($post)); ?>">
                                    <?php echo esc_html($post->post_title); ?>
                                </a>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            <?php endforeach; ?>
        </div>
        <?php
        return ob_get_clean();
    }
}
